/**************************************************************************//**
 * @file     spi_data_szie_select_test.c
 * @version  V1.0
 * $Date:    19/10/23 15:30 $
 * @brief    SPI test case 1, test different data size config (SSPCR0[3:0]) from
 *           4-bit to 16-bit.
 *
 * @note
 * Copyright (C) 2019 Panchip Technology Corp. All rights reserved.
 *
 ******************************************************************************/
#include "pan271x.h"
#include "spi_common.h"


static const uint8_t TestValid4BitData[] =    {0x00, 0x01, 0x02, 0x03, 0x0C, 0x0D, 0x0E, 0x0F};
static const uint8_t TestInvalid4BitData[] =  {0x10, 0x11, 0x12, 0x13, 0xCC, 0xDD, 0xEE, 0xFF};

static const uint8_t TestValid5BitData[] =    {0x00, 0x01, 0x02, 0x03, 0x1C, 0x1D, 0x1E, 0x1F};
static const uint8_t TestInvalid5BitData[] =  {0x20, 0x21, 0x22, 0x23, 0xCC, 0xDD, 0xEE, 0xFF};

static const uint8_t TestValid6BitData[] =    {0x00, 0x01, 0x02, 0x03, 0x3C, 0x3D, 0x3E, 0x3F};
static const uint8_t TestInvalid6BitData[] =  {0x40, 0x41, 0x42, 0x43, 0xCC, 0xDD, 0xEE, 0xFF};

static const uint8_t TestValid7BitData[] =    {0x00, 0x01, 0x02, 0x03, 0x7c, 0x7d, 0x7E, 0x7F};
static const uint8_t TestInvalid7BitData[] =  {0x80, 0x81, 0x82, 0x83, 0xCC, 0xDD, 0xEE, 0xFF};
											  
static const uint8_t TestValid8BitData[] =    {0x00, 0x01, 0x02, 0x03, 0xCC, 0xDD, 0xEE, 0xFF};


static void SPI_PrintInfoCase1(void)
{
    SYS_TEST("\n");
    SYS_TEST("+-----------------------------------------------------------------+ \n");
    SYS_TEST("|    Press key to test specific function:                         | \n");
    SYS_TEST("|                                                                 | \n");
    SYS_TEST("|    Input 'A'    Transfer  4-bit data.                           | \n");
    SYS_TEST("|    Input 'B'    Transfer  5-bit data.                           | \n");
    SYS_TEST("|    Input 'C'    Transfer  6-bit data.                           | \n");
    SYS_TEST("|    Input 'D'    Transfer  7-bit data.                           | \n");
    SYS_TEST("|    Input 'E'    Transfer  8-bit data.                           | \n");
    SYS_TEST("|    Press ESC key to back to the top level case list.            | \n");
    SYS_TEST("+-----------------------------------------------------------------+ \n");
}

static void SPI_SendMultiByteSync(SPI_T* SPIx, const uint8_t* data, size_t expect_send_size)
{
    size_t i = 0;

    while (i < expect_send_size)
    {
        if (!SPI_IsTxFifoFull(SPIx))
        {
            SPI_SendData(SPIx, data[i++]);
        }
    }
    while(SPI_IsBusy(SPIx));    // Make sure all data in FIFO are sent out successfully
	SYS_TEST("Send done!! \n");
}

T_SPI_TEST_RESULT SPI_DataSizeSelectTestCase1(void)
{
    char c;
    SPI_InitTypeDef spiInitStruct;

    spiInitStruct.SPI_baudRateDiv = SPI_BAUD_RATE_DIV(6);//SPI_BaudRateDiv_96;
    spiInitStruct.SPI_CPHA = SPI_ClockPhaseSecondEdge;
    spiInitStruct.SPI_CPOL = SPI_ClockPolarityLow;

    while(1)
    {
        SPI_PrintInfoCase1();
		c = getchar();//'A';
        switch(c)
        {
        /* A. Transfer 4-bit data. */
        case 'A':
        case 'a':
            /* Init Target SPI */
            spiInitStruct.SPI_role = SPI_RoleMaster;
            spiInitStruct.SPI_dataFrameSize = SPI_DataFrame_4b;
            SPI_Init(TGT_SPI, &spiInitStruct);
            SPI_EnableSpi(TGT_SPI);
			/* Transmit data */
			SPI_SendMultiByteSync(TGT_SPI, TestValid4BitData, sizeof(TestValid4BitData));
			SPI_SendMultiByteSync(TGT_SPI, TestInvalid4BitData, sizeof(TestInvalid4BitData));
			
            /* Disable SPI after use */
            SPI_DisableSpi(TGT_SPI);
            break;
        /* B. Transfer 5-bit data. */
        case 'B':
        case 'b':
            /* Init Target SPI */
            spiInitStruct.SPI_role = SPI_RoleMaster;
            spiInitStruct.SPI_dataFrameSize = SPI_DataFrame_5b;
            SPI_Init(TGT_SPI, &spiInitStruct);
            SPI_EnableSpi(TGT_SPI);

            /* Transmit and reseive data */
			SPI_SendMultiByteSync(TGT_SPI, TestValid5BitData, sizeof(TestValid5BitData));
			SPI_SendMultiByteSync(TGT_SPI, TestInvalid5BitData, sizeof(TestInvalid5BitData));
			
			/* Disable SPI after use */
            SPI_DisableSpi(TGT_SPI);
            break;
        /* C. Transfer 6-bit data. */
        case 'C':
        case 'c':
            /* Init Target SPI */
            spiInitStruct.SPI_role = SPI_RoleMaster;
            spiInitStruct.SPI_dataFrameSize = SPI_DataFrame_6b;
            SPI_Init(TGT_SPI, &spiInitStruct);
            SPI_EnableSpi(TGT_SPI);

            /* Transmit and reseive data */
            SPI_SendMultiByteSync(TGT_SPI, TestValid6BitData, sizeof(TestValid6BitData));
			SPI_SendMultiByteSync(TGT_SPI, TestInvalid6BitData, sizeof(TestInvalid6BitData));
			/* Disable SPI after use */
            SPI_DisableSpi(TGT_SPI);
            break;
        /* D. Transfer 7-bit data. */
        case 'D':
        case 'd':
            /* Init Target SPI */
            spiInitStruct.SPI_role = SPI_RoleMaster;
            spiInitStruct.SPI_dataFrameSize = SPI_DataFrame_7b;
            SPI_Init(TGT_SPI, &spiInitStruct);
            SPI_EnableSpi(TGT_SPI);

            /* Transmit and reseive data */
			SPI_SendMultiByteSync(TGT_SPI, TestValid7BitData, sizeof(TestValid7BitData));
			SPI_SendMultiByteSync(TGT_SPI, TestInvalid7BitData, sizeof(TestInvalid7BitData));
			/* Disable SPI after use */
            SPI_DisableSpi(TGT_SPI);
            break;
        /* E. Transfer 8-bit data. */
        case 'E':
        case 'e':
            /* Init Target SPI */
            spiInitStruct.SPI_role = SPI_RoleMaster;
            spiInitStruct.SPI_dataFrameSize = SPI_DataFrame_8b;
            SPI_Init(TGT_SPI, &spiInitStruct);
            SPI_EnableSpi(TGT_SPI);

            /* Transmit and reseive data */
			SPI_SendMultiByteSync(TGT_SPI, TestValid8BitData, sizeof(TestValid8BitData));
			/* Disable SPI after use */
            SPI_DisableSpi(TGT_SPI);
            break;

        case 0x1B:  // Keyboard code <ESC>
            goto OUT;
        default:
            SYS_WARN("Cannot find subtest case %c!\n", c);
            break;
        }
    }
OUT:
    return SPI_TST_OK;
}
