/**************************************************************************//**
 * @file     pan_i2c.h
 * @version  V1.00
 * $Revision: 2 $
 * $Date: 2023/11/08  $
 * @brief    Panchip series I2C driver header file
 *
 * @note
 * Copyright (C) 2023 Panchip Technology Corp. All rights reserved.
 *****************************************************************************/
#ifndef __PAN_I2C_H__
#define __PAN_I2C_H__

/**
 * @brief I2c Interface
 * @defgroup i2c_interface I2c Interface
 * @{
 */

#ifdef __cplusplus
extern "C"
{
#endif

#include "pan271x.h"


/** @addtogroup PAN262x_Std_Driver
 * @{
 */

/** @defgroup I2C
 * @brief I2C driver modules
 * @{
 */

/** @defgroup I2C_Private_Constants
 * @{
 */

#define I2C_TIMEOUT		      1000

typedef enum{
	I2C_DIV_256 = 0x00,
	I2C_DIV_224 = 0x01,
	I2C_DIV_192 = 0x02,
	I2C_DIV_160 = 0x03,
	I2C_DIV_960 = 0x80,
	I2C_DIV_120 = 0x81,
	I2C_DIV_60  = 0x82,
	I2C_DIV_MSK = 0x83
}I2C_Division_t;

typedef enum{
	STARTED			=0x08,
	REPEAT_STARTED	=0x10,
	WR_ADDR_ACK		=0x18,
	WR_ADDR_NACK	=0x20,
	WR_DATA_ACK		=0x28,
	WR_DATA_NACK	=0x30,
	RD_ADDR_ACK		=0x40,
	RD_ADDR_NACK	=0x48,
	RD_DATA_ACK		=0x50,
	RD_DATA_NACK	=0x58,
	ARB_LOST		=0x38,
	IDLE			=0xf8,
	BUS_ERR			=0x00
}I2C_Status_t;

typedef enum
{
    I2C_WRITE,
    I2C_READ,
}I2C_Op_t;

typedef enum
{
    I2C_TYPE_ACK,
    I2C_TYPE_NACK,
}I2C_TypeAck_t;


/**
 * @brief       Set iic division, iic_clk = apbclk / div
 * @param  Division: I2C clock division
 *         @arg: I2C_DIV_256
 *         @arg: I2C_DIV_224
 *         @arg: I2C_DIV_192
 *         @arg: I2C_DIV_160
 *         @arg: I2C_DIV_960
 *         @arg: I2C_DIV_120
 *         @arg: I2C_DIV_60
 * @return      None
 */
__STATIC_INLINE void I2C_SetDivision(I2C_Division_t div)
{
    I2C->CON = (I2C->CON & ~I2C_DIV_MSK) | div;
}



/**
 * @}
 */

/**
 * @}
 */

/**
 * @}
 */

void I2C_Init(I2C_Division_t Division);
void I2C_DeInit(void);
/**
 * @brief       Get current iic division
 * @return      division value
 */
uint16_t I2C_GetDivision(void);


uint8_t I2C_Start(void);
uint8_t I2C_Restart(void);
void I2C_Stop(void);
uint8_t I2C_SendDevAddr(uint8_t Addr, I2C_Op_t Op);
uint8_t I2C_WriteByte(uint8_t Data);
uint8_t I2C_ReadByte(uint8_t *pData, I2C_TypeAck_t Ack);
void I2C_EnableAck(void);
void I2C_DisableAck(void);
bool I2C_GetStaus(I2C_Status_t status, uint16_t timeout);
void I2C_EnableIRQ(void);
void I2C_DisableIRQ(void);


/**@} */

#ifdef __cplusplus
}
#endif

#endif //__PAN_I2C_H__

/*** (C) COPYRIGHT 2016 Panchip Technology Corp. ***/
