/**************************************************************************
 * @file     pan_clk.h
 * @version  V1.00
 * $Revision: 3$
 * $Date: 2023/11/08 $ 
 * @brief    Panchip series CLK driver header file
 *
 * @note
 * Copyright (C) 2023 Panchip Technology Corp. All rights reserved.
 *****************************************************************************/ 
#ifndef __PAN_CLK_H__
#define __PAN_CLK_H__

/**
 * @brief Clk Interface
 * @defgroup clk_interface Clk Interface
 * @{
 */

#ifdef __cplusplus
extern "C"
{
#endif

#include "pan271x.h"



#define CLK_APB1_WDTSEL_MILLI_PULSE             (0x00010000UL)   
#define CLK_APB1_WDTSEL_RCL32K                  (0x00000000UL)

#define CLK_APB1_WWDTSEL_MILLI_PULSE            (0x00000000UL)  /*!< Wwdt clk source select apb_clk / 2048 */
#define CLK_APB1_WWDTSEL_RCL32K                 (0x00020000UL)  /*!< Wwdt clk source select rcl 32k */

#define CLK_APB1_TMR0SEL_APB1CLK                (0x00000000UL)  /*!< Timer0 clk source select apb_clk */
#define CLK_APB1_TMR0SEL_RCL32K                 (0x00040000UL)  /*!< Timer0 clk source select rcl 32k */
#define CLK_APB1_TMR0SEL_TM0                    (0x00080000UL)  /*!< Timer0 clk source select externel input */

#define CLK_APB1_PWM_CH01_SEL_APB               (0x00000000UL)  /*!< PWM CH01 clk source select apb clk */
#define CLK_APB1_PWM_CH01_SEL_CLK32K            (0x00100000UL)  /*!< PWM CH01 clk source select 32k clk */
#define CLK_APB1_PWM_CH23_SEL_APB               (0x00000000UL)  /*!< PWM CH23 clk source select apb clk */
#define CLK_APB1_PWM_CH23_SEL_CLK32K            (0x00200000UL)  /*!< PWM CH23 clk source select 32k clk */
#define CLK_APB1_PWM_CH45_SEL_APB               (0x00000000UL)  /*!< PWM CH45 clk source select apb clk */
#define CLK_APB1_PWM_CH45_SEL_CLK32K            (0x00400000UL)  /*!< PWM CH45 clk source select 32k clk */

#define CLK_APB1_ADC_SEL_APB                    (0x00000000UL)  /*!< ADC clk source select apb clk */
#define CLK_APB1_ADC_SEL_XTH                    (0x01000000UL)  /*!< ADC clk source select xth clk */

#define SYS_CTRL_CAP_SRC_SEL_MAC_ACC            (0x00000000UL)  /*!< ADC clk source select apb clk */
#define SYS_CTRL_CAP_SRC_SEL_USB_SOF            (0x04000000UL)  /*!< ADC clk source select xth clk */


/** @defgroup CLK_FREQUENCT_FLAG Clk frequence
 * @brief       Clk frequence definitions
 * @{ */

 #define FREQ_1MHZ                            (1000000)
 #define FREQ_8MHZ                            (8000000)
 #define FREQ_16MHZ                           (16000000)
 #define FREQ_25MHZ                           (25000000)
 #define FREQ_32MHZ                           (32000000)
 #define FREQ_48MHZ                           (48000000)
 #define FREQ_64MHZ                           (64000000)
 #define FREQ_96MHZ                           (96000000)
 #define FREQ_50MHZ                           (50000000)
 #define FREQ_72MHZ                           (72000000)
 #define FREQ_100MHZ                          (100000000)
/**@} */


/** @defgroup SYS_CLK_SOURCE_FLAG System clk source select
 * @brief       System clk source definitions
 * @{ */
#define CLK_SYS_SRCSEL_RCH         	          ((uint32_t)0x00000000)  /*!< System clk source frequence select as rch */
#define CLK_SYS_SRCSEL_XTH        	          ((uint32_t)0x00000001)  /*!< System clk source frequence select as xth */
#define CLK_SYS_SRCSEL_DPLL         	      ((uint32_t)0x00000002)  /*!< System clk source frequence select as dpll */
#define CLK_SYS_SRCSEL_DPLL_1         	      ((uint32_t)0x00000003)  /*!< System clk source frequence select as dpll */
/**@} */

/*SYS CLK DPLL SELECTED*/
/** @defgroup DPLL_SOURCE_FLAG Dpll clk reference source select
 * @brief       Dpll clk reference source select definitions
 * @{ */
#define CLK_DPLL_REF_CLKSEL_RCH         	  ((uint32_t)0x00000000)
#define CLK_DPLL_REF_CLKSEL_XTH         	  ((uint32_t)0x00000008)
#define CLK_DPLL_OUT_48M       	              ((uint32_t)0x00000004)
#define CLK_DPLL_OUT_32M        	          ((uint32_t)0x00000000)
/**@} */


/** @defgroup APB1_PERIPHERAL_CLK_FLAG Apb1 clk enable
 * @brief       Apb1 clk enable definitions
 * @{ */

#define CLK_APB1Periph_I2C0               ((uint32_t)0x00000001)
#define CLK_APB1Periph_SPI0               ((uint32_t)0x00000002)
#define CLK_APB1Periph_UART0              ((uint32_t)0x00000008)
#define CLK_APB1Periph_PWM0_CH01          ((uint32_t)0x00000010)
#define CLK_APB1Periph_PWM0_CH23          ((uint32_t)0x00000020)
#define CLK_APB1Periph_PWM0_CH45          ((uint32_t)0x00000040)
#define CLK_APB1Periph_PWM0_EN            ((uint32_t)0x00000100)
#define CLK_APB1Periph_ADC                ((uint32_t)0x00000200)
#define CLK_APB1Periph_WDT                ((uint32_t)0x00000400)
#define CLK_APB1Periph_TMR0               ((uint32_t)0x00001000)
#define CLK_APB1Periph_TRIM               ((uint32_t)0x00002000)
#define CLK_APB1Periph_KSCAN              ((uint32_t)0x00004000)
#define CLK_APB1Periph_UART1              ((uint32_t)0x00008000)
#define CLK_APB1Periph_All                ((uint32_t)0x0000F77b)
/**@} */

/** @defgroup AHB_PERIPHERAL_CLK_FLAG Ahb peripheral clk enable
 * @brief       Ahb peripheral clk enable definitions
 * @{ */
 
#define CLK_AHBPeriph_GPIO                ((uint32_t)0x00002000)
#define CLK_AHBPeriph_SYSTICK             ((uint32_t)0x00004000)
#define CLK_AHBPeriph_APB1                ((uint32_t)0x00008000)
#define CLK_AHBPeriph_LL                  ((uint32_t)0x00020000)
#define CLK_AHBPeriph_PHY                 ((uint32_t)0x00040000)
#define CLK_AHBPeriph_USB_AHB             ((uint32_t)0x00080000)
#define CLK_AHBPeriph_USB_48M             ((uint32_t)0x00100000)
#define CLK_AHBPeriph_All                 ((uint32_t)0x001EE000)
/**@} */

/** @defgroup CLK_SRC_FLAG Clk reference source
 * @brief       Clk reference source definitions
 * @{ */
#define CLK_DivideSource_Ahb              (1)
#define CLK_DivideSource_Apb1             (2)
/**@} */

/** @defgroup CLK_SRC_FLAG Clk reference source
 * @brief       Clk reference source definitions
 * @{ */
#define CLK_RCL_SELECT                    (0)    /*!< Clk source reference set as rcl */
#define CLK_RCH_SELECT                    (1)    /*!< Clk source reference set as rch */
#define CLK_XTL_SELECT                    (2)    /*!< Clk source reference set as xtl */
#define CLK_XTH_SELECT                    (3)    /*!< Clk source reference set as xth */
#define CLK_DPLL_SELECT                   (4)    /*!< Clk source reference set as dpll */
/**@} */

#define CLK_STABLE_STATUS_Pos             (24)  /*!< Clk stable register position */
#define CLK_STABLE_STATUS_Msk             (0x1ul << CLK_STABLE_STATUS_Pos)    /*!< Clk stable register mask value */

#define CLKTRIM_CALC_CLK_SEL_32K		  (0)
#define CLKTRIM_CALC_CLK_SEL_32M		  (1)

#define CLKTRIM_QDEC_CLK_SEL_APB		  (0)       /*!< Qdec clk source selecet apb clk */
#define CLKTRIM_QDEC_CLK_SEL_32K		  (1)       /*!< Qdec clk source selecet 32K */

#define CLKTRIM_KSCAN_CLK_SEL_APB		  (0)
#define CLKTRIM_KSCAN_CLK_SEL_32K		  (1)


/** @defgroup CLK_RST_MODULE Module Reset Control Register Constants
 * @brief Module Reset Control Register constant definitions
 * @{
 */
#define DMA_RST     ((0x0<<24) | CLK_IPRST0_DMARST_Pos      ) /*!< DMA_RST is one of the CLK_ResetModule() parameter */
#define LL_RST      ((0x0<<24) | CLK_IPRST0_RFRST_Pos       ) /*!< LL_RST is one of the CLK_ResetModule() parameter */
#define EFUSE_RST   ((0x0<<24) | CLK_IPRST0_EFUSERST_Pos    ) /*!< EFUSE_RST is one of the CLK_ResetModule() parameter */
#define USB_RST     ((0x0<<24) | CLK_IPRST0_USBRST_Pos      ) /*!< USB_RST is one of the CLK_ResetModule() parameter */
#define MDMSTB_RST  ((0x0<<24) | CLK_IPRST0_MDMSTDBYRST_Pos ) /*!< MDMSTB_RST is one of the CLK_ResetModule() parameter */
#define MDM_RST     ((0x0<<24) | CLK_IPRST0_MDMRST_Pos      ) /*!< MDM_RST is one of the CLK_ResetModule() parameter */
#define I2C0_RST    ((0x4<<24) | CLK_IPRST1_I2C0RST_Pos     ) /*!< I2C0_RST is one of the CLK_ResetModule() parameter */
#define SPI0_RST    ((0x4<<24) | CLK_IPRST1_SPI0RST_Pos     ) /*!< SPI0_RST is one of the CLK_ResetModule() parameter */
#define SPI1_RST    ((0x4<<24) | CLK_IPRST1_SPI1RST_Pos     ) /*!< SPI0_RST is one of the CLK_ResetModule() parameter */
#define UART0_RST   ((0x4<<24) | CLK_IPRST1_UART0RST_Pos    ) /*!< UART0_RST is one of the CLK_ResetModule() parameter */
#define UART1_RST   ((0x4<<24) | CLK_IPRST1_UART1RST_Pos    ) /*!< UART0_RST is one of the CLK_ResetModule() parameter */
#define PWM0_RST    ((0x4<<24) | CLK_IPRST1_PWM0RST_Pos     ) /*!< PWM0_RST is one of the CLK_ResetModule() parameter */
#define ADC_RST     ((0x4<<24) | CLK_IPRST1_ADCRST_Pos      ) /*!< ADC_RST is one of the CLK_ResetModule() parameter */
#define WDT_RST     ((0x4<<24) | CLK_IPRST1_WDTRST_Pos      ) /*!< WDT_RST is one of the CLK_ResetModule() parameter */
#define WWDT_RST    ((0x4<<24) | CLK_IPRST1_WWDTRST_Pos     ) /*!< WWDT_RST is one of the CLK_ResetModule() parameter */
#define TMR0_RST    ((0x4<<24) | CLK_IPRST1_TMR0RST_Pos     ) /*!< TMR0_RST is one of the CLK_ResetModule() parameter */
#define TMR1_RST    ((0x4<<24) | CLK_IPRST1_TMR1RST_Pos     ) /*!< TMR1_RST is one of the CLK_ResetModule() parameter */
#define TMR2_RST    ((0x4<<24) | CLK_IPRST1_TMR2RST_Pos     ) /*!< TMR2_RST is one of the CLK_ResetModule() parameter */
#define GPIO_RST    ((0x4<<24) | CLK_IPRST1_GPIORST_Pos     ) /*!< GPIO_RST is one of the CLK_ResetModule() parameter */
#define CLKTRIM_RST ((0x4<<24) | CLK_IPRST1_TRIMRST_Pos     ) /*!< CLKTRIM_RST is one of the CLK_ResetModule() parameter */
/** @} */ // End of CLK_RST_MODULE

/**
 * @defgroup BOD_Threshold_Selection Brown Out Detector Threshold Voltage Selection Constants
 * @brief Constants for configuring the Brown Out Detector threshold voltage selection.
 * @{
 */
#define CLK_BODCTL_BOD_INT_EN           (0UL<<BLD_CTRL_BOD_RST_EN_Pos)  /*!< Brown-out Interrupt Enable */
#define CLK_BODCTL_BOD_RST_EN           (1UL<<BLD_CTRL_BOD_RST_EN_Pos)  /*!< Brown-out Reset Enable */
#define CLK_BODCTL_BODSEL_1_75V         (0UL<<BLD_CTRL_PMU_BOD_VSEL_AON_Msk)    /*!< Setting Brown Out Detector Threshold Voltage as 1.75V */
#define CLK_BODCTL_BODSEL_1_85V         (1UL<<BLD_CTRL_PMU_BOD_VSEL_AON_Msk)    /*!< Setting Brown Out Detector Threshold Voltage as 1.75V */
#define CLK_BODCTL_BODSEL_1_95V         (2UL<<BLD_CTRL_PMU_BOD_VSEL_AON_Msk)    /*!< Setting Brown Out Detector Threshold Voltage as 1.95V */
#define CLK_BODCTL_BODSEL_2_05V         (3UL<<BLD_CTRL_PMU_BOD_VSEL_AON_Msk)    /*!< Setting Brown Out Detector Threshold Voltage as 1.75V */
#define CLK_BODCTL_BODSEL_2_15V         (4UL<<BLD_CTRL_PMU_BOD_VSEL_AON_Msk)    /*!< Setting Brown Out Detector Threshold Voltage as 2.15V */
#define CLK_BODCTL_BODSEL_2_25V         (5UL<<BLD_CTRL_PMU_BOD_VSEL_AON_Msk)    /*!< Setting Brown Out Detector Threshold Voltage as 1.75V */
#define CLK_BODCTL_BODSEL_2_35V         (6UL<<BLD_CTRL_PMU_BOD_VSEL_AON_Msk)    /*!< Setting Brown Out Detector Threshold Voltage as 2.35V */
#define CLK_BODCTL_BODSEL_2_45V         (7UL<<BLD_CTRL_PMU_BOD_VSEL_V2_AON_Pos)    /*!< Setting Brown Out Detector Threshold Voltage as 1.75V */
#define CLK_BODCTL_BODSEL_2_55V         (8UL<<BLD_CTRL_PMU_BOD_VSEL_V2_AON_Pos)    /*!< Setting Brown Out Detector Threshold Voltage as 2.55V */
#define CLK_BODCTL_BODSEL_2_65V         (9UL<<BLD_CTRL_PMU_BOD_VSEL_V2_AON_Pos)    /*!< Setting Brown Out Detector Threshold Voltage as 1.75V */
#define CLK_BODCTL_BODSEL_2_75V         (10UL<<BLD_CTRL_PMU_BOD_VSEL_V2_AON_Pos)    /*!< Setting Brown Out Detector Threshold Voltage as 2.75V */
#define CLK_BODCTL_BODSEL_2_85V         (11UL<<BLD_CTRL_PMU_BOD_VSEL_V2_AON_Pos)    /*!< Setting Brown Out Detector Threshold Voltage as 1.75V */
#define CLK_BODCTL_BODSEL_2_95V         (12UL<<BLD_CTRL_PMU_BOD_VSEL_V2_AON_Pos)    /*!< Setting Brown Out Detector Threshold Voltage as 2.95V */
#define CLK_BODCTL_BODSEL_3_05V         (13UL<<BLD_CTRL_PMU_BOD_VSEL_V2_AON_Pos)    /*!< Setting Brown Out Detector Threshold Voltage as 1.75V */
/** @} */ // End of BOD_Threshold_Selection


/**
 * @defgroup BOD_Debounce_Selection Brown Out Detector Debounce Selection Constants
 * @brief Constants for configuring the debounce selection for the Brown Out Detector.
 * @{
 */
#define CLK_BLDBCTL_BODDBSEL_2POW0      0x00000001UL
#define CLK_BLDBCTL_BODDBSEL_2POW1      0x00000002UL
#define CLK_BLDBCTL_BODDBSEL_2POW2      0x00000004UL
#define CLK_BLDBCTL_BODDBSEL_2POW3      0x00000008UL
#define CLK_BLDBCTL_BODDBSEL_2POW4      0x00000010UL
#define CLK_BLDBCTL_BODDBSEL_2POW5      0x00000020UL
/**@} */

/**
 * @defgroup LVR_Debounce_Selection Low Voltage Reset Debounce Selection Constants
 * @brief Constants for configuring the debounce selection for the Low Voltage Reset.
 * @{
 */
#define CLK_BLDBCTL_LVRDBSEL_2POW0      0x00800000UL
#define CLK_BLDBCTL_LVRDBSEL_2POW1      0x01000000UL
#define CLK_BLDBCTL_LVRDBSEL_2POW2      0x02000000UL
#define CLK_BLDBCTL_LVRDBSEL_2POW3      0x04000000UL
#define CLK_BLDBCTL_LVRDBSEL_2POW4      0x08000000UL
#define CLK_BLDBCTL_LVRDBSEL_2POW5      0x10000000UL
/**@} */

/**
  * @brief  Configures the xth clock.
  * @retval None
  */
__STATIC_INLINE void CLK_XthStartupConfig(void)
{
    PMU->XTH_CTRL |= (XTH_CTRL_FSYNXO_STARTUP_FAST_AON_Msk);
}

/**
  * @brief  Configures the Low Speed AHB clock (HCLK).
  * @param  u32ClkDiv: defines the AHB clock divider. This clock is derived from 
  *         the AHB clock (HCLK).
  *          This parameter can be 0~15,hclk = hclk / (u32ClkDiv + 1):
  * @retval None
  */
__STATIC_INLINE void CLK_HCLKConfig(uint32_t u32ClkDiv)
{
    CLK->AHB_CLK_CTRL = (CLK->AHB_CLK_CTRL & (~AHB_CLK_CTRL_AHB_DIV_Msk)) | (u32ClkDiv << AHB_CLK_CTRL_AHB_DIV_Pos);
}

/**
  * @brief  Configures the Low Speed APB clock (PCLK1).
  * @param  u32ClkDiv: defines the APB1 clock divider. This clock is derived from 
  *         the AHB clock (HCLK).
  * 		This parameter can be 0~15,PCLK1 = hclk / (u32ClkDiv * 2):
  * @retval None
  */
__STATIC_INLINE void CLK_PCLK1Config(uint32_t u32ClkDiv )
{
    CLK->AHB_CLK_CTRL = (CLK->AHB_CLK_CTRL & (~AHB_CLK_CTRL_APB1_DIV_Msk))| (u32ClkDiv << AHB_CLK_CTRL_APB1_DIV_Pos);
}


/**
* @brief  This function set DPLL frequence
* @param[in]  freq is target frequency,it could be:
*							\ref CLK_DPLL_OUT_48M
*							\ref CLK_DPLL_OUT_32M
* @return   none
*
*/
__STATIC_INLINE void CLK_SetDpllOutputFreq(uint32_t freq)
{
    PMU->DPLL_CTRL &= ~DPLL_CTRL_CLK_DPLL_FREQ_SEL_AON_Msk;
    PMU->DPLL_CTRL |= freq;
}


/**
* @brief  This function used to set keyscan divider
* @param[in]  div is keyscan sclk dividor ratio,9 bit width, clk = apbclk / (2 * div)
* @return   none
*
*/
__STATIC_INLINE void CLK_SetKeyscanDiv(uint32_t div)
{
    CLK->APB1_CLK_CTRL0 = (CLK->APB1_CLK_CTRL0 & ~APB1_CLK_CTRL0_KEYSCAN_DIV_Msk) | (div << APB1_CLK_CTRL0_KEYSCAN_DIV_Pos);
}
/**
* @brief  This function used to get keyscan divider
* @return keyscan divider value
*
*/
__STATIC_INLINE uint32_t CLK_GetKeyscanDiv(void)
{
    return ((CLK->APB1_CLK_CTRL0 & APB1_CLK_CTRL0_KEYSCAN_DIV_Msk) >> APB1_CLK_CTRL0_KEYSCAN_DIV_Pos);
}
/**
* @brief  This function used to set keyscan clk source
* @param[in]  src is keyscan sclk source,apb or 32k 
* @return   none
*
*/
__STATIC_INLINE void CLK_SetKeyscanClkSrc(uint32_t src)
{
    CLK->APB1_CLK_CTRL0 = (CLK->APB1_CLK_CTRL0 & ~APB1_CLK_CTRL0_KEYSCAN_CLK_SEL_Msk) | (src << APB1_CLK_CTRL0_KEYSCAN_CLK_SEL_Pos);
}


/**
  * @brief  This function get HCLK frequency. The frequency unit is Hz.
  * @return HCLK frequency
  */
uint32_t CLK_GetHCLKFreq(void);

 /**
   * @brief  This function get CPU frequency. The frequency unit is Hz.
   * @return CPU frequency
   */
uint32_t CLK_GetCPUFreq(void);

/**
  * @brief  This function get APB1 frequency. The frequency unit is Hz.
  * @return HCLK frequency
  */
uint32_t CLK_GetPCLK1Freq(void);


 /**
    * @brief  This function set 16M ref clock source
    * @param[in]  u32ClkSrc is HCLK clock source. Including :
    *                  - \ref CLK_SYS_SRCSEL_RCH 
    *                  - \ref CLK_SYS_SRCSEL_XTH 
    *                  - \ref CLK_SYS_SRCSEL_DPLL
    * @return None
    */
void CLK_RefClkSrcConfig(uint32_t u32ClkSrc);
/**
  * @brief  This function set HCLK clock source
  * @param[in]  u32ClkSrc is HCLK clock source. Including :
  *                  - \ref CLK_DPLL_REF_CLKSEL_RCH
  *                  - \ref CLK_DPLL_REF_CLKSEL_XTH
  * @param[in]  freq_out is output frequence clock
  * @return None
  */
void CLK_SYSCLKConfig(uint32_t u32ClkSrc,uint32_t freq_out);
/**
  * @brief  This function set CPU frequency divider. The frequency unit is Hz.
  * @param[in]  u32ClkDiv is ahb clock division
  * @return HCLK frequency
  */
void CLK_HCLKConfig(uint32_t u32ClkDiv);
/**
  * @brief  This function set APB1 frequency divider. The frequency unit is Hz.
  * @param[in]  u32ClkDiv is is APB1 clock division
  * @return HCLK frequency
  */
void CLK_PCLK1Config(uint32_t u32ClkDiv );
/**
  * @brief  This function set APB2 frequency divider. The frequency unit is Hz.
  * @param[in]  u32ClkDiv is is APB2 clock division  
  * @return HCLK frequency
  */
void CLK_PCLK2Config(uint32_t u32ClkDiv);

/**
  * @brief  Enables or disables the AHB peripheral clock.
  * @note   After reset, the peripheral clock (used for registers read/write access)
  *         is disabled and the application software has to enable this clock before 
  *         using it. 
  * @param  CLK_AHBPeriph: specifies the AHB2 peripheral to gates its clock.
  *          This parameter can be any combination of the following values:
  *             \ref AHB_PERIPHERAL_CLK_FLAG
  * @param  NewState    : new state of the specified peripheral clock.
  *          This parameter can be: ENABLE or DISABLE.
  * @retval None
  */
void CLK_AHBPeriphClockCmd(uint32_t CLK_AHBPeriph, FunctionalState NewState);

/**
  * @brief  Enables or disables the Low Speed APB (APB1) peripheral clock.
  * @note   After reset, the peripheral clock (used for registers read/write access)
  *         is disabled and the application software has to enable this clock before 
  *         using it. 
  * @param  CLK_APB1Periph: specifies the APB1 peripheral to gates its clock.
  *          This parameter can be any combination of the following values:
  *            \ref APB1_PERIPHERAL_CLK_FLAG
  * @param  NewState  : new state of the specified peripheral clock.
  *          This parameter can be: ENABLE or DISABLE.
  * @retval None
  */
void CLK_APB1PeriphClockCmd(uint32_t CLK_APB1Periph, FunctionalState NewState);

/**
  * @brief  Enables or disables the High Speed APB (APB2) peripheral clock.
  * @note   After reset, the peripheral clock (used for registers read/write access)
  *         is disabled and the application software has to enable this clock before 
  *         using it.
  * @param  CLK_APB2Periph: specifies the APB2 peripheral to gates its clock.
  *          This parameter can be any combination of the following values:
  *            \ref APB2_PERIPHERAL_CLK_FLAG 
  * @param  NewState: new state of the specified peripheral clock.
  *          This parameter can be: ENABLE or DISABLE.
  * @retval None
  */
void CLK_APB2PeriphClockCmd(uint32_t CLK_APB2Periph, FunctionalState NewState);

/**
  * @brief  This function check selected clock source status
  * @param[in]  u32ClkMask is selected clock source. Including
  *                  - \ref CLK_RCL_SELECT
  *                  - \ref CLK_RCH_SELECT
  *                  - \ref CLK_DPLL_SELECT
  * @retval   0  clock is not stable
  * @retval   1  clock is stable
  *
  * @details  To wait for clock ready by specified CLKSTATUS bit or timeout (~5ms)
  */
uint32_t CLK_WaitClockReady(uint32_t u32ClkMask);

/**
  * @brief  This function wait sync 3v clock locale stable
  * @param[in]  none
  * @retval   0  clock sync is not stable
  * @retval   1  clock sync is stable
  *
  * @details  To wait for clock ready by specified CLKSTATUS bit or timeout (~5ms)
  */
uint32_t CLK_Wait3vSyncReady(void);

/**
* @brief  This function wait sync 3v clock locale stable by hardware
* @return   none
*/
void CLK_Set3vSyncAuto(void);
/**
  * @brief This API is used to select wdt clock source
  * @param[in] u32clksel wdt clock selection. Could be 
  *             - \ref CLK_APB1_WDTSEL_MILLI_PULSE, \ref CLK_APB1_WDTSEL_RCL32K
  * @return none
  * @note This API is only used to select wdt clock source
  */
void CLK_SetWdtClkSrc(uint32_t u32clksel);

/**
  * @brief This API is used to select wwdt clock source
  * @param[in] u32clksel wwdt clock selection. Could be 
  *             - \ref CLK_APB1_WWDTSEL_MILLI_PULSE, \ref CLK_APB1_WWDTSEL_RCL32K
  * @return none
  * @note This API is only used to select wwdt clock source
  */
void CLK_SetWwdtClkSrc(uint32_t u32clksel);

/**
  * @brief This API is used to SELECT timer clock source
  * @param[in] u32clksel timer clock selection. Could be 
  *             - \ref CLK_APB_TMR0SEL_APB1CLK, \ref CLK_APB_TMR0SEL_LIRC,\ref CLK_APB_TMR0SEL_TM0
  *             - \ref CLK_APB_TMR1SEL_APB2CLK, \ref CLK_APB_TMR1SEL_LIRC,\ref CLK_APB_TMR0SEL_TM1
  *             - \ref CLK_APB_TMR2SEL_APB2CLK, \ref CLK_APB_TMR2SEL_LIRC,\ref CLK_APB_TMR0SEL_TM2
  * @return none
  * @note This API is only used to select timer clock source
  */
void CLK_SetTmrClkSrc(uint32_t u32clksel);


/**
  * @brief This API is used to select timer capture clock source
  * @param[in] u32clksel timer clock selection. Could be 
  *             - \ref SYS_CTRL_CAP_SRC_SEL_MAC_ACC 
  *             - \ref SYS_CTRL_CAP_SRC_SEL_USB_SOF
  * @return none
  * @note This API is only used to select timer capture clock source
  */
void CLK_SetTmrCaptureSrc(uint32_t u32clksel);

/**
  * @brief This API is used to select pwm counting clock source
  * @param[in] u32ChNum   pwm channel. Could be 0 ~ 7
  * @param[in] u32clksel  pwm counting clock selection. Could be
  *                       - \ref CLK_APB1_PWM_CH01_SEL_APB, \ref CLK_APB1_PWM_CH01_SEL_CLK32K
  *                       - \ref CLK_APB1_PWM_CH23_SEL_APB, \ref CLK_APB1_PWM_CH23_SEL_CLK32K
  *                       - \ref CLK_APB1_PWM_CH45_SEL_APB, \ref CLK_APB1_PWM_CH45_SEL_CLK32K
  *                       - \ref CLK_APB1_PWM_CH67_SEL_APB, \ref CLK_APB1_PWM_CH67_SEL_CLK32K
  * @return none
  * @note PWM channel 0/1, 2/3, 4/5, 6/7 share same clock source.
  */
void CLK_SetPwmClkSrc(uint32_t u32ChNum, uint32_t u32clksel);

/**
  * @brief This API is used to get peripheral clk frequence
  * @param[in] Peripheral The base address of peripheral module
  * @return none
  * @note This API is only used to select timer clock source
  */
uint32_t CLK_GetPeripheralFreq(void * Peripheral);

/**
 * @brief Reset the entire chip.
 *
 * This function resets the entire chip.
 *
 * @return None
 */
void CLK_ResetChip(void);

/**
 * @brief Reset the system to ROM mode.
 *
 * This function resets the system to ROM mode.
 *
 * @return None
 */
void CLK_ResetSystemToRomMode(void);

/**
 * @brief Reset a specified module.
 *
 * This function resets a specified module based on the provided module index.
 *
 * @param[in] u32ModuleIndex The index of the module to reset.
 *
 * @return None
 */
void CLK_ResetModule(uint32_t u32ModuleIndex);

/**
 * @brief Configure the BOD (Brown-Out Detector) with the specified mode and BOD level.
 *
 * This function configures the BOD with the specified mode and BOD level.
 *
 * @param[in] i32Mode The BOD mode.
 * @param[in] u32BODLevel The BOD level.
 *
 * @return None
 */
void CLK_ConfigureBOD(int32_t i32Mode, uint32_t u32BODLevel);

/**
 * @brief Enable the BOD (Brown-Out Detector).
 *
 * This function enables the BOD.
 *
 * @return None
 */
void CLK_EnableBOD(void);

/**
 * @brief Disable the BOD (Brown-Out Detector).
 *
 * This function disables the BOD.
 *
 * @return None
 */
void CLK_DisableBOD(void);

/**
 * @brief Get the BOD (Brown-Out Detector) status.
 *
 * This function returns the status of the BOD.
 *
 * @return The BOD status.
 */
uint32_t CLK_GetBODStatus(void);

/**
 * @brief Get the BOD (Brown-Out Detector) interrupt flag.
 *
 * This function returns the BOD interrupt flag.
 *
 * @return The BOD interrupt flag.
 */
uint32_t CLK_GetBODIntFlag(void);

/**
 * @brief Clear the BOD (Brown-Out Detector) interrupt flag.
 *
 * This function clears the BOD interrupt flag.
 *
 * @return None
 */
void CLK_ClearBODIntFlag(void);

/**
 * @brief Set the BOD (Brown-Out Detector) debounce time.
 *
 * This function sets the BOD debounce time.
 *
 * @param[in] u32BodDBTime The BOD debounce time to set.
 *
 * @return None
 */
void CLK_SetBODDebounceTime(uint32_t u32BodDBTime);

/**
 * @brief Enable the LVR (Low-Voltage Reset).
 *
 * This function enables the LVR.
 *
 * @return None
 */
void CLK_EnableLVR(void);

/**
 * @brief Disable the LVR (Low-Voltage Reset).
 *
 * This function disables the LVR.
 *
 * @return None
 */
void CLK_DisableLVR(void);

/**
 * @brief Set the LVR (Low-Voltage Reset) debounce time.
 *
 * This function sets the LVR debounce time.
 *
 * @param[in] u32LvrDBTime The LVR debounce time to set.
 *
 * @return None
 */
void CLK_SetLVRDebounceTime(uint32_t u32LvrDBTime);



/**@} */

#ifdef __cplusplus
}
#endif

#endif //__PAN_CLK_H__

/*** (C) COPYRIGHT 2023 Panchip Technology Corp. ***/
