/************************************************************************//**
 * @file     pan271x.h
 * @version  V1.00
 * $Revision: 0 $
 * $Date: 2023/11/08  $
 * @brief    Header file for the Panchip Series library.
 *
 * @note     This library provides functions and definitions specific to
 *           the Panchip Series microcontroller family.
 *           Copyright (C) 2023 Panchip Technology Corp. All rights reserved.
 *****************************************************************************/
#ifndef __PAN271X_H__
#define __PAN271X_H__

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
/**
 * @brief Panchip Series Interface
 * @defgroup panseries_interface Panchip Series Interface
 * @{
 */

/**
 * @brief Enumeration of Cortex-M0 Processor Exceptions and Device-specific Interrupt Numbers.
 */
typedef enum IRQn
{
    /******  Cortex-M0 Processor Exceptions Numbers ***************************************************/
    NonMaskableInt_IRQn         = -14,    /*!<  2 Cortex-M0 Non Maskable Interrupt                  */
    HardFault_IRQn              = -13,    /*!<  3 Cortex-M0 Hard Fault Interrupt                    */
    SVCall_IRQn                 = -5,     /*!< 11 Cortex-M0 SV Call Interrupt                       */
    PendSV_IRQn                 = -2,     /*!< 14 Cortex-M0 Pend SV Interrupt                       */
    SysTick_IRQn                = -1,     /*!< 15 Cortex-M0 System Tick Interrupt                   */

    /******  Device-specific Interrupt Numbers ********************************************************/
    I2C0_IRQn                   = 0,      /*!< I2C0 Interrupt */
    SPI0_IRQn                   = 1,      /*!< SPI0 Interrupt */
    UART0_IRQn                  = 3,      /*!< UART0 Interrupt */
    PWM_IRQn                    = 4,      /*!< PWM Interrupt */
    ADC_IRQn                    = 5,      /*!< ADC Interrupt */
    WDT_IRQn                    = 6,      /*!< Watchdog Timer Interrupt */
    TRIM_IRQn                   = 7,      /*!< Windowed Watchdog Timer Interrupt */
    TMR0_IRQn                   = 8,      /*!< Timer0 Interrupt */
    KSCAN_IRQn                  = 9,     /*!< SPI1 Interrupt */
    UART1_IRQn                  = 10,     /*!< UART1 Interrupt */
    LL_IRQn                     = 11,     /*!< LinLayer Interrupt */
    GPIO0_IRQn                  = 15,     /*!< GPIO Port 0 Interrupt */
    GPIO1_IRQn                  = 16,     /*!< GPIO Port 1 Interrupt */
    GPIO2_IRQn                  = 17,     /*!< GPIO Port 2 Interrupt */
    USB_IRQn                    = 22,     /*!< USB Interrupt */
    BOD_IRQn                    = 28,     /*!< Brown Out Detector Interrupt */
    SLPTMR_IRQn                 = 29,     /*!< Sleep Timer Interrupt */
    LP_IRQn                     = 30,     /*!< Low Power Interrupt */
    SLPTMR1_IRQn                = 31,     /*!< Sleep Timer1 Interrupt */

} IRQn_Type;



/*
 * ==========================================================================
 * ----------- Processor and Core Peripheral Section ------------------------
 * ==========================================================================
 */

/* Configuration of the Cortex-M0 Processor and Core Peripherals */
#define __CM0_REV                 0x0000    /*!< Core revision r0p0                               */
#define __MPU_PRESENT             0         /*!< MPU present or not                               */
#define __NVIC_PRIO_BITS          2         /*!< Number of Bits used for Priority Levels          */
#define __Vendor_SysTickConfig    0         /*!< Set to 1 if different SysTick Config is used     */
#define __FPU_PRESENT             0         /*!< FPU present or not                               */

#include <core_cm0.h>                       /* Cortex-M0 processor and core peripherals           */
#include "system_pan271x.h"             /* Panchip Series System include file                   */



#include <stdbool.h>

typedef enum {RESET = 0, SET = !RESET} FlagStatus, ITStatus;

typedef enum {DISABLE = 0, ENABLE = !DISABLE} FunctionalState;
#define IS_FUNCTIONAL_STATE(STATE) (((STATE) == DISABLE) || ((STATE) == ENABLE))

//typedef enum {ERROR = 0, SUCCESS = !ERROR} ErrorStatus;


/*---------------------- Timer Controller -------------------------*/
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
/**
    @defgroup TIMER Timer Controller(TIMER)
    Memory Mapped Structure for TIMER Controller
@{ */

typedef struct {
	__IO uint32_t TIMER_CTL;
	__IO uint32_t TIMER_CAPTURE;
} TIMER_T;


/**@} */
/**
    @addtogroup TIMER_CONST TIMER Bit Field Definition
    Constant Definitions for TIMER Controller
@{ */


#define TIMER_CTL_INT_EN_Pos             (31)
#define TIMER_CTL_INT_EN_Msk             (0x1ul << TIMER_CTL_INT_EN_Pos)

#define TIMER_CTL_WORK_MODE_Pos          (30)
#define TIMER_CTL_WORK_MODE_Msk          (0x1ul << TIMER_CTL_WORK_MODE_Pos)

#define TIMER_CTL_CNT_EN_Pos             (29)
#define TIMER_CTL_CNT_EN_Msk             (0x1ul << TIMER_CTL_CNT_EN_Pos)

#define TIMER_CTL_TIMER_FLAG_Pos         (28)
#define TIMER_CTL_TIMER_FLAG_Msk         (0x1ul << TIMER_CTL_TIMER_FLAG_Pos)

#define TIMER_CTL_CAPTURE_INT_EN_Pos     (27)
#define TIMER_CTL_CAPTURE_INT_EN_Msk     (0x1ul << TIMER_CTL_CAPTURE_INT_EN_Pos)

#define TIMER_CTL_CAPTURE_FLAG_Pos       (26)
#define TIMER_CTL_CAPTURE_FLAG_Msk       (0x1ul << TIMER_CTL_CAPTURE_FLAG_Pos)

#define TIMER_CTL_SOFTWARE_CNT_CLEAR_Pos (25)
#define TIMER_CTL_SOFTWARE_CNT_CLEAR_Msk (0x1ul << TIMER_CTL_SOFTWARE_CNT_CLEAR_Pos)

#define TIMER_CTL_CMPDATA_Pos            (0)
#define TIMER_CTL_CMPDATA_Msk            (0x1fffffful << TIMER_CTL_CMPDATA_Pos)

#define TIMER_CAPTURE_CAPTURE_EN_Pos     (27)
#define TIMER_CAPTURE_CAPTURE_EN_Msk     (0x1ul << TIMER_CAPTURE_CAPTURE_EN_Pos)

#define TIMER_CAPTURE_CAPTURE_SEL_Pos    (25)
#define TIMER_CAPTURE_CAPTURE_SEL_Msk    (0x3ul << TIMER_CAPTURE_CAPTURE_SEL_Pos)

#define TIMER_CAPTURE_CAP_CNT_Pos        (0)
#define TIMER_CAPTURE_CAP_CNT_Msk        (0x1fffffful << TIMER_CAPTURE_CAP_CNT_Pos)


/**@}*/ /* TIMER_CONST */
/**@}*/ /* end of TIMER register group */


/*clk trim*/
typedef struct {                                  //offset
    __IO uint32_t  CLKMEAS_CTL                ;   //0x00
    __IO uint32_t  CLKMEAS_CNT                ;   //0x04
} CLKMEAS_T;

/*CLKMEAS_CTL*/
#define CLKMEAS_MEASURE_EN_Pos                        (0)           
#define CLKMEAS_MEASURE_EN_Msk                        (0x1ul << CLKMEAS_MEASURE_EN_Pos)
#define CLKMEAS_INT_EN_Pos                            (1)           
#define CLKMEAS_INT_EN_Msk                            (0x1ul << CLKMEAS_INT_EN_Pos)
#define CLKMEAS_CLK_SEL_Pos                           (2)           
#define CLKMEAS_CLK_SEL_Msk                           (0x1ul << CLKMEAS_CLK_SEL_Pos)
#define CLKMEAS_MODE_SEL_Pos                          (3)           
#define CLKMEAS_MODE_SEL_Msk                          (0x3ul << CLKMEAS_MODE_SEL_Pos)
#define CLKMEAS_SAMP_NUM_Pos                          (5)           
#define CLKMEAS_SAMP_NUM_Msk                          (0x7ul << CLKMEAS_SAMP_NUM_Pos)
#define CLKMEAS_MEASURE_FLAG_Pos                      (8)           
#define CLKMEAS_MEASURE_FLAG_Msk                      (0x1ul << CLKMEAS_MEASURE_FLAG_Pos)

/*CLKMEAS_CNT  */
#define CLKMEAS_MEASURE_CNT_Pos                        	(0)           
#define CLKMEAS_MEASURE_CNT_Msk                        	(0x3FFFFFFul << CLKMEAS_MEASURE_CNT_Pos)







/*---------------------- General Purpose Input/Output Controller -------------------------*/
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
/**
    @addtogroup GP General Purpose Input/Output Controller(GP)
    Memory Mapped Structure for GP Controller
@{ */

typedef struct {


    /**
     * MODE
     * ===================================================================================================
     * Offset: 0x00  P0 I/O Mode Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |MODE0     |Port 0-5 I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Px.n is in Quasi-bidirectional mode.
     * |        |          |analog input mode after chip powered on.
     * |        |          |Note1:
     * |        |          |Max. n=7 for port 0
     * |        |          |Max. n=7 for port 1
     * |        |          |Max. n=7 for port 2
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[3:2]   |MODE1     |Port 0-5 I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Px.n is in Quasi-bidirectional mode.
     * |        |          |analog input mode after chip powered on.
     * |        |          |Note1:
     * |        |          |Max. n=7 for port 0
     * |        |          |Max. n=7 for port 1
     * |        |          |Max. n=7 for port 2
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.

     * |[5:4]   |MODE2     |Port 0-5 I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Px.n is in Quasi-bidirectional mode.
     * |        |          |analog input mode after chip powered on.
     * |        |          |Note1:
     * |        |          |Max. n=7 for port 0
     * |        |          |Max. n=7 for port 1
     * |        |          |Max. n=7 for port 2
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.

     * |[7:6]   |MODE3     |Port 0-5 I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Px.n is in Quasi-bidirectional mode.
     * |        |          |analog input mode after chip powered on.
     * |        |          |Note1:
     * |        |          |Max. n=7 for port 0
     * |        |          |Max. n=7 for port 1
     * |        |          |Max. n=7 for port 2
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.

     * |[9:8]   |MODE4     |Port 0-5 I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Px.n is in Quasi-bidirectional mode.
     * |        |          |analog input mode after chip powered on.
     * |        |          |Note1:
     * |        |          |Max. n=7 for port 0
     * |        |          |Max. n=7 for port 1
     * |        |          |Max. n=7 for port 2
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.

     * |[11:10] |MODE5     |Port 0-5 I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Px.n is in Quasi-bidirectional mode.
     * |        |          |analog input mode after chip powered on.
     * |        |          |Note1:
     * |        |          |Max. n=7 for port 0
     * |        |          |Max. n=7 for port 1
     * |        |          |Max. n=7 for port 2
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.

     * |[13:12] |MODE6     |Port 0-5 I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |11 = Px.n is in Quasi-bidirectional mode.
     * |        |          |analog input mode after chip powered on.
     * |        |          |Note1:
     * |        |          |Max. n=7 for port 0
     * |        |          |Max. n=7 for port 1
     * |        |          |Max. n=7 for port 2
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.

     * |[15:14] |MODE7     |Port 0-5 I/O Pin[N] Mode Control
     * |        |          |Determine each I/O mode of Px.n pins.
     * |        |          |00 = Px.n is in Input mode.
     * |        |          |01 = Px.n is in Push-pull Output mode.
     * |        |          |10 = Px.n is in Open-drain Output mode.
     * |        |          |analog input mode after chip powered on.
     * |        |          |Note1:
     * |        |          |Max. n=7 for port 0
     * |        |          |Max. n=7 for port 1
     * |        |          |Max. n=7 for port 2
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5..
    */
    __IO uint32_t MODE;

    /**
     * DINOFF
     * ===================================================================================================
     * Offset: 0x04  P0 Digital Input Path Disable Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[16]    |DINOFF0   |Port 0-5 Pin[N] Digital Input Path Disable Control
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled.
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[17]    |DINOFF1   |Port 0-5 Pin[N] Digital Input Path Disable Control
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled.
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[18]    |DINOFF2   |Port 0-5 Pin[N] Digital Input Path Disable Control
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled.
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[19]    |DINOFF3   |Port 0-5 Pin[N] Digital Input Path Disable Control
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled.
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[20]    |DINOFF4   |Port 0-5 Pin[N] Digital Input Path Disable Control
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled.
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[21]    |DINOFF5   |Port 0-5 Pin[N] Digital Input Path Disable Control
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled.
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[22]    |DINOFF6   |Port 0-5 Pin[N] Digital Input Path Disable Control
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled.
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[23]    |DINOFF7   |Port 0-5 Pin[N] Digital Input Path Disable Control
     * |        |          |Each of these bits is used to control if the digital input path of corresponding Px.n pin is disabled.
     * |        |          |If input is analog signal, users can disable Px.n digital input path to avoid input current leakage.
     * |        |          |0 = Px.n digital input path Enabled.
     * |        |          |1 = Px.n digital input path Disabled (digital input tied to low).
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
    */
    __IO uint32_t DINOFF;

    /**
     * DOUT
     * ===================================================================================================
     * Offset: 0x08  P0 Data Output Value
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |DOUT0     |Port 0-5 Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output or Quasi-bidirectional mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output or Quasi-bidirectional mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output or Quasi-bidirectional mode.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[1]     |DOUT1     |Port 0-5 Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output or Quasi-bidirectional mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output or Quasi-bidirectional mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output or Quasi-bidirectional mode.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[2]     |DOUT2     |Port 0-5 Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output or Quasi-bidirectional mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output or Quasi-bidirectional mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output or Quasi-bidirectional mode.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[3]     |DOUT3     |Port 0-5 Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output or Quasi-bidirectional mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output or Quasi-bidirectional mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output or Quasi-bidirectional mode.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[4]     |DOUT4     |Port 0-5 Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output or Quasi-bidirectional mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output or Quasi-bidirectional mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output or Quasi-bidirectional mode.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[5]     |DOUT5     |Port 0-5 Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output or Quasi-bidirectional mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output or Quasi-bidirectional mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output or Quasi-bidirectional mode.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[6]     |DOUT6     |Port 0-5 Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output or Quasi-bidirectional mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output or Quasi-bidirectional mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output or Quasi-bidirectional mode.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[7]     |DOUT7     |Port 0-5 Pin[N] Output Value
     * |        |          |Each of these bits controls the status of a Px.n pin when the Px.n is configured as Push-pull output, Open-drain output or Quasi-bidirectional mode.
     * |        |          |0 = Px.n will drive Low if the Px.n pin is configured as Push-pull output, Open-drain output or Quasi-bidirectional mode.
     * |        |          |1 = Px.n will drive High if the Px.n pin is configured as Push-pull output or Quasi-bidirectional mode.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
    */
    __IO uint32_t DOUT;

    /**
     * Reserved
     * ===================================================================================================
     * Offset: 0x0C
     * ---------------------------------------------------------------------------------------------------
    */
    __IO uint32_t RSV1;

    /**
     * PIN
     * ===================================================================================================
     * Offset: 0x10  P0 Pin Value
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |PIN0      |Port 0-5 Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[1]     |PIN1      |Port 0-5 Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[2]     |PIN2      |Port 0-5 Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[3]     |PIN3      |Port 0-5 Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[4]     |PIN4      |Port 0-5 Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[5]     |PIN5      |Port 0-5 Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[6]     |PIN6      |Port 0-5 Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[7]     |PIN7      |Port 0-5 Pin[N] Pin Value
     * |        |          |Each bit of the register reflects the actual status of the respective Px.n pin.
     * |        |          |If the bit is 1, it indicates the corresponding pin status is high; else the pin status is low.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
    */
    __IO  uint32_t PIN;

    /**
     * DBEN
     * ===================================================================================================
     * Offset: 0x14  P0 De-bounce Enable Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |DBEN0     |Port 0-5 Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit.
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt.
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt.
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignore.
     * |        |          |Note1: If Px.n pin is chosen as Power-down wake-up source, user should be disable the de-bounce function before entering Power-down mode to avoid the second interrupt event occurred after system waken up which caused by Px.n de-bounce function.
     * |        |          |Note2:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[1]     |DBEN1     |Port 0-5 Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit.
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt.
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt.
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignore.
     * |        |          |Note1: If Px.n pin is chosen as Power-down wake-up source, user should be disable the de-bounce function before entering Power-down mode to avoid the second interrupt event occurred after system waken up which caused by Px.n de-bounce function.
     * |        |          |Note2:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[2]     |DBEN2     |Port 0-5 Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit.
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt.
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt.
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignore.
     * |        |          |Note1: If Px.n pin is chosen as Power-down wake-up source, user should be disable the de-bounce function before entering Power-down mode to avoid the second interrupt event occurred after system waken up which caused by Px.n de-bounce function.
     * |        |          |Note2:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[3]     |DBEN3     |Port 0-5 Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit.
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt.
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt.
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignore.
     * |        |          |Note1: If Px.n pin is chosen as Power-down wake-up source, user should be disable the de-bounce function before entering Power-down mode to avoid the second interrupt event occurred after system waken up which caused by Px.n de-bounce function.
     * |        |          |Note2:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[4]     |DBEN4     |Port 0-5 Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit.
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt.
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt.
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignore.
     * |        |          |Note1: If Px.n pin is chosen as Power-down wake-up source, user should be disable the de-bounce function before entering Power-down mode to avoid the second interrupt event occurred after system waken up which caused by Px.n de-bounce function.
     * |        |          |Note2:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[5]     |DBEN5     |Port 0-5 Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit.
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt.
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt.
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignore.
     * |        |          |Note1: If Px.n pin is chosen as Power-down wake-up source, user should be disable the de-bounce function before entering Power-down mode to avoid the second interrupt event occurred after system waken up which caused by Px.n de-bounce function.
     * |        |          |Note2:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[6]     |DBEN6     |Port 0-5 Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit.
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt.
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt.
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignore.
     * |        |          |Note1: If Px.n pin is chosen as Power-down wake-up source, user should be disable the de-bounce function before entering Power-down mode to avoid the second interrupt event occurred after system waken up which caused by Px.n de-bounce function.
     * |        |          |Note2:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[7]     |DBEN7     |Port 0-5 Pin[N] Input Signal De-bounce Enable Bit
     * |        |          |The DBEN[n] bit is used to enable the de-bounce function for each corresponding bit.
     * |        |          |If the input signal pulse width cannot be sampled by continuous two de-bounce sample cycle, the input signal transition is seen as the signal bounce and will not trigger the interrupt.
     * |        |          |The de-bounce clock source is controlled by DBCLKSRC (GPIO_DBCTL [4]), one de-bounce sample cycle period is controlled by DBCLKSEL (GPIO_DBCTL [3:0]).
     * |        |          |0 = Px.n de-bounce function Disabled.
     * |        |          |1 = Px.n de-bounce function Enabled.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt.
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignore.
     * |        |          |Note1: If Px.n pin is chosen as Power-down wake-up source, user should be disable the de-bounce function before entering Power-down mode to avoid the second interrupt event occurred after system waken up which caused by Px.n de-bounce function.
     * |        |          |Note2:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
    */
    __IO uint32_t DBEN;

    /**
     * INTTYPE
     * ===================================================================================================
     * Offset: 0x18  P0 Interrupt Mode Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |TYPE0     |Port 0-5 Pin[N] Edge Or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger.
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce.
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n]).
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt.
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignore.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[1]     |TYPE1     |Port 0-5 Pin[N] Edge Or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger.
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce.
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n]).
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt.
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignore.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[2]     |TYPE2     |Port 0-5 Pin[N] Edge Or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger.
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce.
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n]).
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt.
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignore.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[3]     |TYPE3     |Port 0-5 Pin[N] Edge Or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger.
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce.
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n]).
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt.
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignore.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[4]     |TYPE4     |Port 0-5 Pin[N] Edge Or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger.
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce.
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n]).
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt.
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignore.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[5]     |TYPE5     |Port 0-5 Pin[N] Edge Or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger.
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce.
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n]).
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt.
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignore.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[6]     |TYPE6     |Port 0-5 Pin[N] Edge Or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger.
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce.
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n]).
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt.
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignore.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[7]     |TYPE7     |Port 0-5 Pin[N] Edge Or Level Detection Interrupt Trigger Type Control
     * |        |          |TYPE (Px_INTTYPE[n]) bit is used to control the triggered interrupt is by level trigger or by edge trigger.
     * |        |          |If the interrupt is by edge trigger, the trigger source can be controlled by de-bounce.
     * |        |          |If the interrupt is by level trigger, the input source is sampled by one HCLK clock and generates the interrupt.
     * |        |          |0 = Edge trigger interrupt.
     * |        |          |1 = Level trigger interrupt.
     * |        |          |If the pin is set as the level trigger interrupt, only one level can be set on the registers RHIEN (Px_INTEN[n+16])/FLIEN (Px_INTEN[n]).
     * |        |          |If both levels to trigger interrupt are set, the setting is ignored and no interrupt will occur.
     * |        |          |The de-bounce function is valid only for edge triggered interrupt.
     * |        |          |If the interrupt mode is level triggered, the de-bounce enable bit is ignore.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
    */
    __IO uint32_t INTTYPE;

    /**
     * INTEN
     * ===================================================================================================
     * Offset: 0x1C  P0 Interrupt Enable Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |FLIEN0    |Port 0-5 Pin[N] Falling Edge Or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1:
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[1]     |FLIEN1    |Port 0-5 Pin[N] Falling Edge Or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1:
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[2]     |FLIEN2    |Port 0-5 Pin[N] Falling Edge Or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1:
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[3]     |FLIEN3    |Port 0-5 Pin[N] Falling Edge Or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1:
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[4]     |FLIEN4    |Port 0-5 Pin[N] Falling Edge Or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1:
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[5]     |FLIEN5    |Port 0-5 Pin[N] Falling Edge Or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1:
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[6]     |FLIEN6    |Port 0-5 Pin[N] Falling Edge Or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1:
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[7]     |FLIEN7    |Port 0-5 Pin[N] Falling Edge Or Low Level Interrupt Trigger Type Enable Bit
     * |        |          |The FLIEN (Px_INTEN[n]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the FLIEN (Px_INTEN[n]) bit to 1function
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at low level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from high to low.
     * |        |          |0 = Px.n level low or high to low interrupt Disabled.
     * |        |          |1 = Px.n level low or high to low interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[16]    |RHIEN0    |Port 0-5 Pin[N] Rising Edge Or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1:
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[17]    |RHIEN1    |Port 0-5 Pin[N] Rising Edge Or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1:
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[18]    |RHIEN2    |Port 0-5 Pin[N] Rising Edge Or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1:
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[19]    |RHIEN3    |Port 0-5 Pin[N] Rising Edge Or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1:
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[20]    |RHIEN4    |Port 0-5 Pin[N] Rising Edge Or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1:
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[21]    |RHIEN5    |Port 0-5 Pin[N] Rising Edge Or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1:
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[22]    |RHIEN6    |Port 0-5 Pin[N] Rising Edge Or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1:
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[23]    |RHIEN7    |Port 0-5 Pin[N] Rising Edge Or High Level Interrupt Trigger Type Enable Bit
     * |        |          |The RHIEN (Px_INTEN[n+16]) bit is used to enable the interrupt for each of the corresponding input Px.n pin.
     * |        |          |Set bit to 1 also enable the pin wake-up function.
     * |        |          |When setting the RHIEN (Px_INTEN[n+16]) bit to 1:
     * |        |          |If the interrupt is level trigger (TYPE (Px_INTTYPE[n]) bit is set to 1), the input Px.n pin will generate the interrupt while this pin state is at high level.
     * |        |          |If the interrupt is edge trigger (TYPE (Px_INTTYPE[n]) bit is set to 0), the input Px.n pin will generate the interrupt while this pin state changed from low to high.
     * |        |          |0 = Px.n level high or low to high interrupt Disabled.
     * |        |          |1 = Px.n level high or low to high interrupt Enabled.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
    */
    __IO uint32_t INTEN;

    /**
     * INTSRC
     * ===================================================================================================
     * Offset: 0x20  P0 Interrupt Source Flag
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |INTSRC0   |Port 0-5 Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[1]     |INTSRC1   |Port 0-5 Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[2]     |INTSRC2   |Port 0-5 Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[3]     |INTSRC3   |Port 0-5 Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[4]     |INTSRC4   |Port 0-5 Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[5]     |INTSRC5   |Port 0-5 Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[6]     |INTSRC6   |Port 0-5 Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[7]     |INTSRC7   |Port 0-5 Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[8]     |INTSRC8   |Port 0-5 Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[9]     |INTSRC9   |Port 0-5 Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[10]    |INTSRC10  |Port 0-5 Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[11]    |INTSRC11  |Port 0-5 Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[12]    |INTSRC12  |Port 0-5 Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[13]    |INTSRC13  |Port 0-5 Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[14]    |INTSRC14  |Port 0-5 Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
     * |[15]    |INTSRC15  |Port 0-5 Pin[N] Interrupt Source Flag
     * |        |          |Write Operation:
     * |        |          |0 = No action.
     * |        |          |1 = Clear the corresponding pending interrupt.
     * |        |          |Read Operation:
     * |        |          |0 = No interrupt at Px.n.
     * |        |          |1 = Px.n generates an interrupt.
     * |        |          |Note:
     * |        |          |Max. n=7 for port 0.
     * |        |          |Max. n=7 for port 1.
     * |        |          |Max. n=7 for port 2.
     * |        |          |Max. n=7 for port 3, n=3, n=7 are reserved.
     * |        |          |Max. n=7 for port 4, n=0,.5 are reserved.
     * |        |          |Max. n=7 for port 5.
    */
    __IO uint32_t INTSRC;

} GPIO_T;

typedef struct {

    /**
     * DBCTL
     * ===================================================================================================
     * Offset: 0x180  De-bounce Cycle Control
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[3:0]   |DBCLKSEL  |De-bounce Sampling Cycle Selection
     * |        |          |0000 = Sample interrupt input once per 1 clock.
     * |        |          |0001 = Sample interrupt input once per 2 clocks.
     * |        |          |0010 = Sample interrupt input once per 4 clocks.
     * |        |          |0011 = Sample interrupt input once per 8 clocks.
     * |        |          |0100 = Sample interrupt input once per 16 clocks.
     * |        |          |0101 = Sample interrupt input once per 32 clocks.
     * |        |          |0110 = Sample interrupt input once per 64 clocks.
     * |        |          |0111 = Sample interrupt input once per 128 clocks.
     * |        |          |1000 = Sample interrupt input once per 256 clocks.
     * |        |          |1001 = Sample interrupt input once per 2*256 clocks.
     * |        |          |1010 = Sample interrupt input once per 4*256 clocks.
     * |        |          |1011 = Sample interrupt input once per 8*256 clocks.
     * |        |          |1100 = Sample interrupt input once per 16*256 clocks.
     * |        |          |1101 = Sample interrupt input once per 32*256 clocks.
     * |        |          |1110 = Sample interrupt input once per 64*256 clocks.
     * |        |          |1111 = Sample interrupt input once per 128*256 clocks.
     * |[4]     |DBCLKSRC  |De-bounce Counter Clock Source Selection
     * |        |          |0 = De-bounce counter clock source is HCLK.
     * |        |          |1 = De-bounce counter clock source is 10 kHz internal low speed RC oscillator (LIRC).
    */
    __IO uint32_t DBCTL;

} GPIO_DB_T;


/**
    @addtogroup GP_CONST GP Bit Field Definition
    Constant Definitions for GP Controller
@{ */

#define GP_MODE_MODE0_Pos                (0)                                               /*!< GP_T::MODE: MODE0 Position                */
#define GP_MODE_MODE0_Msk                (0x3ul << GP_MODE_MODE0_Pos)                      /*!< GP_T::MODE: MODE0 Mask                    */

#define GP_MODE_MODE1_Pos                (2)                                               /*!< GP_T::MODE: MODE1 Position                */
#define GP_MODE_MODE1_Msk                (0x3ul << GP_MODE_MODE1_Pos)                      /*!< GP_T::MODE: MODE1 Mask                    */

#define GP_MODE_MODE2_Pos                (4)                                               /*!< GP_T::MODE: MODE2 Position                */
#define GP_MODE_MODE2_Msk                (0x3ul << GP_MODE_MODE2_Pos)                      /*!< GP_T::MODE: MODE2 Mask                    */

#define GP_MODE_MODE3_Pos                (6)                                               /*!< GP_T::MODE: MODE3 Position                */
#define GP_MODE_MODE3_Msk                (0x3ul << GP_MODE_MODE3_Pos)                      /*!< GP_T::MODE: MODE3 Mask                    */

#define GP_MODE_MODE4_Pos                (8)                                               /*!< GP_T::MODE: MODE4 Position                */
#define GP_MODE_MODE4_Msk                (0x3ul << GP_MODE_MODE4_Pos)                      /*!< GP_T::MODE: MODE4 Mask                    */

#define GP_MODE_MODE5_Pos                (10)                                              /*!< GP_T::MODE: MODE5 Position                */
#define GP_MODE_MODE5_Msk                (0x3ul << GP_MODE_MODE5_Pos)                      /*!< GP_T::MODE: MODE5 Mask                    */

#define GP_DINOFF_DINOFF0_Pos            (16)                                              /*!< GP_T::DINOFF: DINOFF0 Position            */
#define GP_DINOFF_DINOFF0_Msk            (0x1ul << GP_DINOFF_DINOFF0_Pos)                  /*!< GP_T::DINOFF: DINOFF0 Mask                */

#define GP_DINOFF_DINOFF1_Pos            (17)                                              /*!< GP_T::DINOFF: DINOFF1 Position            */
#define GP_DINOFF_DINOFF1_Msk            (0x1ul << GP_DINOFF_DINOFF1_Pos)                  /*!< GP_T::DINOFF: DINOFF1 Mask                */

#define GP_DINOFF_DINOFF2_Pos            (18)                                              /*!< GP_T::DINOFF: DINOFF2 Position            */
#define GP_DINOFF_DINOFF2_Msk            (0x1ul << GP_DINOFF_DINOFF2_Pos)                  /*!< GP_T::DINOFF: DINOFF2 Mask                */

#define GP_DINOFF_DINOFF3_Pos            (19)                                              /*!< GP_T::DINOFF: DINOFF3 Position            */
#define GP_DINOFF_DINOFF3_Msk            (0x1ul << GP_DINOFF_DINOFF3_Pos)                  /*!< GP_T::DINOFF: DINOFF3 Mask                */

#define GP_DINOFF_DINOFF4_Pos            (20)                                              /*!< GP_T::DINOFF: DINOFF4 Position            */
#define GP_DINOFF_DINOFF4_Msk            (0x1ul << GP_DINOFF_DINOFF4_Pos)                  /*!< GP_T::DINOFF: DINOFF4 Mask                */

#define GP_DINOFF_DINOFF5_Pos            (21)                                              /*!< GP_T::DINOFF: DINOFF5 Position            */
#define GP_DINOFF_DINOFF5_Msk            (0x1ul << GP_DINOFF_DINOFF5_Pos)                  /*!< GP_T::DINOFF: DINOFF5 Mask                */

#define GP_DINOFF_DINOFF6_Pos            (22)                                              /*!< GP_T::DINOFF: DINOFF6 Position            */
#define GP_DINOFF_DINOFF6_Msk            (0x1ul << GP_DINOFF_DINOFF6_Pos)                  /*!< GP_T::DINOFF: DINOFF6 Mask                */

#define GP_DINOFF_DINOFF7_Pos            (23)                                              /*!< GP_T::DINOFF: DINOFF7 Position            */
#define GP_DINOFF_DINOFF7_Msk            (0x1ul << GP_DINOFF_DINOFF7_Pos)                  /*!< GP_T::DINOFF: DINOFF7 Mask                */

#define GP_DOUT_DOUT0_Pos                (0)                                               /*!< GP_T::DOUT: DOUT0 Position                */
#define GP_DOUT_DOUT0_Msk                (0x1ul << GP_DOUT_DOUT0_Pos)                      /*!< GP_T::DOUT: DOUT0 Mask                    */

#define GP_DOUT_DOUT1_Pos                (1)                                               /*!< GP_T::DOUT: DOUT1 Position                */
#define GP_DOUT_DOUT1_Msk                (0x1ul << GP_DOUT_DOUT1_Pos)                      /*!< GP_T::DOUT: DOUT1 Mask                    */

#define GP_DOUT_DOUT2_Pos                (2)                                               /*!< GP_T::DOUT: DOUT2 Position                */
#define GP_DOUT_DOUT2_Msk                (0x1ul << GP_DOUT_DOUT2_Pos)                      /*!< GP_T::DOUT: DOUT2 Mask                    */

#define GP_DOUT_DOUT3_Pos                (3)                                               /*!< GP_T::DOUT: DOUT3 Position                */
#define GP_DOUT_DOUT3_Msk                (0x1ul << GP_DOUT_DOUT3_Pos)                      /*!< GP_T::DOUT: DOUT3 Mask                    */

#define GP_DOUT_DOUT4_Pos                (4)                                               /*!< GP_T::DOUT: DOUT4 Position                */
#define GP_DOUT_DOUT4_Msk                (0x1ul << GP_DOUT_DOUT4_Pos)                      /*!< GP_T::DOUT: DOUT4 Mask                    */

#define GP_DOUT_DOUT5_Pos                (5)                                               /*!< GP_T::DOUT: DOUT5 Position                */
#define GP_DOUT_DOUT5_Msk                (0x1ul << GP_DOUT_DOUT5_Pos)                      /*!< GP_T::DOUT: DOUT5 Mask                    */

#define GP_DOUT_DOUT6_Pos                (6)                                               /*!< GP_T::DOUT: DOUT6 Position                */
#define GP_DOUT_DOUT6_Msk                (0x1ul << GP_DOUT_DOUT6_Pos)                      /*!< GP_T::DOUT: DOUT6 Mask                    */

#define GP_DOUT_DOUT7_Pos                (7)                                               /*!< GP_T::DOUT: DOUT7 Position                */
#define GP_DOUT_DOUT7_Msk                (0x1ul << GP_DOUT_DOUT7_Pos)                      /*!< GP_T::DOUT: DOUT7 Mask                    */

#define GP_DATMSK_DATMSK0_Pos            (0)                                               /*!< GP_T::DATMSK: DATMSK0 Position            */
#define GP_DATMSK_DATMSK0_Msk            (0x1ul << GP_DATMSK_DATMSK0_Pos)                  /*!< GP_T::DATMSK: DATMSK0 Mask                */

#define GP_DATMSK_DATMSK1_Pos            (1)                                               /*!< GP_T::DATMSK: DATMSK1 Position            */
#define GP_DATMSK_DATMSK1_Msk            (0x1ul << GP_DATMSK_DATMSK1_Pos)                  /*!< GP_T::DATMSK: DATMSK1 Mask                */

#define GP_DATMSK_DATMSK2_Pos            (2)                                               /*!< GP_T::DATMSK: DATMSK2 Position            */
#define GP_DATMSK_DATMSK2_Msk            (0x1ul << GP_DATMSK_DATMSK2_Pos)                  /*!< GP_T::DATMSK: DATMSK2 Mask                */

#define GP_DATMSK_DATMSK3_Pos            (3)                                               /*!< GP_T::DATMSK: DATMSK3 Position            */
#define GP_DATMSK_DATMSK3_Msk            (0x1ul << GP_DATMSK_DATMSK3_Pos)                  /*!< GP_T::DATMSK: DATMSK3 Mask                */

#define GP_DATMSK_DATMSK4_Pos            (4)                                               /*!< GP_T::DATMSK: DATMSK4 Position            */
#define GP_DATMSK_DATMSK4_Msk            (0x1ul << GP_DATMSK_DATMSK4_Pos)                  /*!< GP_T::DATMSK: DATMSK4 Mask                */

#define GP_DATMSK_DATMSK5_Pos            (5)                                               /*!< GP_T::DATMSK: DATMSK5 Position            */
#define GP_DATMSK_DATMSK5_Msk            (0x1ul << GP_DATMSK_DATMSK5_Pos)                  /*!< GP_T::DATMSK: DATMSK5 Mask                */

#define GP_DATMSK_DATMSK6_Pos            (6)                                               /*!< GP_T::DATMSK: DATMSK6 Position            */
#define GP_DATMSK_DATMSK6_Msk            (0x1ul << GP_DATMSK_DATMSK6_Pos)                  /*!< GP_T::DATMSK: DATMSK6 Mask                */

#define GP_DATMSK_DATMSK7_Pos            (7)                                               /*!< GP_T::DATMSK: DATMSK7 Position            */
#define GP_DATMSK_DATMSK7_Msk            (0x1ul << GP_DATMSK_DATMSK7_Pos)                  /*!< GP_T::DATMSK: DATMSK7 Mask                */

#define GP_PIN_PIN0_Pos                  (0)                                               /*!< GP_T::PIN: PIN0 Position                  */
#define GP_PIN_PIN0_Msk                  (0x1ul << GP_PIN_PIN0_Pos)                        /*!< GP_T::PIN: PIN0 Mask                      */

#define GP_PIN_PIN1_Pos                  (1)                                               /*!< GP_T::PIN: PIN1 Position                  */
#define GP_PIN_PIN1_Msk                  (0x1ul << GP_PIN_PIN1_Pos)                        /*!< GP_T::PIN: PIN1 Mask                      */

#define GP_PIN_PIN2_Pos                  (2)                                               /*!< GP_T::PIN: PIN2 Position                  */
#define GP_PIN_PIN2_Msk                  (0x1ul << GP_PIN_PIN2_Pos)                        /*!< GP_T::PIN: PIN2 Mask                      */

#define GP_PIN_PIN3_Pos                  (3)                                               /*!< GP_T::PIN: PIN3 Position                  */
#define GP_PIN_PIN3_Msk                  (0x1ul << GP_PIN_PIN3_Pos)                        /*!< GP_T::PIN: PIN3 Mask                      */

#define GP_PIN_PIN4_Pos                  (4)                                               /*!< GP_T::PIN: PIN4 Position                  */
#define GP_PIN_PIN4_Msk                  (0x1ul << GP_PIN_PIN4_Pos)                        /*!< GP_T::PIN: PIN4 Mask                      */

#define GP_PIN_PIN5_Pos                  (5)                                               /*!< GP_T::PIN: PIN5 Position                  */
#define GP_PIN_PIN5_Msk                  (0x1ul << GP_PIN_PIN5_Pos)                        /*!< GP_T::PIN: PIN5 Mask                      */

#define GP_PIN_PIN6_Pos                  (6)                                               /*!< GP_T::PIN: PIN6 Position                  */
#define GP_PIN_PIN6_Msk                  (0x1ul << GP_PIN_PIN6_Pos)                        /*!< GP_T::PIN: PIN6 Mask                      */

#define GP_PIN_PIN7_Pos                  (7)                                               /*!< GP_T::PIN: PIN7 Position                  */
#define GP_PIN_PIN7_Msk                  (0x1ul << GP_PIN_PIN7_Pos)                        /*!< GP_T::PIN: PIN7 Mask                      */

#define GP_DBEN_DBEN0_Pos                (0)                                               /*!< GP_T::DBEN: DBEN0 Position                */
#define GP_DBEN_DBEN0_Msk                (0x1ul << GP_DBEN_DBEN0_Pos)                      /*!< GP_T::DBEN: DBEN0 Mask                    */

#define GP_DBEN_DBEN1_Pos                (1)                                               /*!< GP_T::DBEN: DBEN1 Position                */
#define GP_DBEN_DBEN1_Msk                (0x1ul << GP_DBEN_DBEN1_Pos)                      /*!< GP_T::DBEN: DBEN1 Mask                    */

#define GP_DBEN_DBEN2_Pos                (2)                                               /*!< GP_T::DBEN: DBEN2 Position                */
#define GP_DBEN_DBEN2_Msk                (0x1ul << GP_DBEN_DBEN2_Pos)                      /*!< GP_T::DBEN: DBEN2 Mask                    */

#define GP_DBEN_DBEN3_Pos                (3)                                               /*!< GP_T::DBEN: DBEN3 Position                */
#define GP_DBEN_DBEN3_Msk                (0x1ul << GP_DBEN_DBEN3_Pos)                      /*!< GP_T::DBEN: DBEN3 Mask                    */

#define GP_DBEN_DBEN4_Pos                (4)                                               /*!< GP_T::DBEN: DBEN4 Position                */
#define GP_DBEN_DBEN4_Msk                (0x1ul << GP_DBEN_DBEN4_Pos)                      /*!< GP_T::DBEN: DBEN4 Mask                    */

#define GP_DBEN_DBEN5_Pos                (5)                                               /*!< GP_T::DBEN: DBEN5 Position                */
#define GP_DBEN_DBEN5_Msk                (0x1ul << GP_DBEN_DBEN5_Pos)                      /*!< GP_T::DBEN: DBEN5 Mask                    */

#define GP_DBEN_DBEN6_Pos                (6)                                               /*!< GP_T::DBEN: DBEN6 Position                */
#define GP_DBEN_DBEN6_Msk                (0x1ul << GP_DBEN_DBEN6_Pos)                      /*!< GP_T::DBEN: DBEN6 Mask                    */

#define GP_DBEN_DBEN7_Pos                (7)                                               /*!< GP_T::DBEN: DBEN7 Position                */
#define GP_DBEN_DBEN7_Msk                (0x1ul << GP_DBEN_DBEN7_Pos)                      /*!< GP_T::DBEN: DBEN7 Mask                    */

#define GP_INTTYPE_TYPE0_Pos             (0)                                               /*!< GP_T::INTTYPE: TYPE0 Position             */
#define GP_INTTYPE_TYPE0_Msk             (0x1ul << GP_INTTYPE_TYPE0_Pos)                   /*!< GP_T::INTTYPE: TYPE0 Mask                 */

#define GP_INTTYPE_TYPE1_Pos             (1)                                               /*!< GP_T::INTTYPE: TYPE1 Position             */
#define GP_INTTYPE_TYPE1_Msk             (0x1ul << GP_INTTYPE_TYPE1_Pos)                   /*!< GP_T::INTTYPE: TYPE1 Mask                 */

#define GP_INTTYPE_TYPE2_Pos             (2)                                               /*!< GP_T::INTTYPE: TYPE2 Position             */
#define GP_INTTYPE_TYPE2_Msk             (0x1ul << GP_INTTYPE_TYPE2_Pos)                   /*!< GP_T::INTTYPE: TYPE2 Mask                 */

#define GP_INTTYPE_TYPE3_Pos             (3)                                               /*!< GP_T::INTTYPE: TYPE3 Position             */
#define GP_INTTYPE_TYPE3_Msk             (0x1ul << GP_INTTYPE_TYPE3_Pos)                   /*!< GP_T::INTTYPE: TYPE3 Mask                 */

#define GP_INTTYPE_TYPE4_Pos             (4)                                               /*!< GP_T::INTTYPE: TYPE4 Position             */
#define GP_INTTYPE_TYPE4_Msk             (0x1ul << GP_INTTYPE_TYPE4_Pos)                   /*!< GP_T::INTTYPE: TYPE4 Mask                 */

#define GP_INTTYPE_TYPE5_Pos             (5)                                               /*!< GP_T::INTTYPE: TYPE5 Position             */
#define GP_INTTYPE_TYPE5_Msk             (0x1ul << GP_INTTYPE_TYPE5_Pos)                   /*!< GP_T::INTTYPE: TYPE5 Mask                 */

#define GP_INTTYPE_TYPE6_Pos             (6)                                               /*!< GP_T::INTTYPE: TYPE6 Position             */
#define GP_INTTYPE_TYPE6_Msk             (0x1ul << GP_INTTYPE_TYPE6_Pos)                   /*!< GP_T::INTTYPE: TYPE6 Mask                 */

#define GP_INTTYPE_TYPE7_Pos             (7)                                               /*!< GP_T::INTTYPE: TYPE7 Position             */
#define GP_INTTYPE_TYPE7_Msk             (0x1ul << GP_INTTYPE_TYPE7_Pos)                   /*!< GP_T::INTTYPE: TYPE7 Mask                 */

#define GP_INTEN_FLIEN0_Pos              (0)                                               /*!< GP_T::INTEN: FLIEN0 Position              */
#define GP_INTEN_FLIEN0_Msk              (0x1ul << GP_INTEN_FLIEN0_Pos)                    /*!< GP_T::INTEN: FLIEN0 Mask                  */

#define GP_INTEN_FLIEN1_Pos              (1)                                               /*!< GP_T::INTEN: FLIEN1 Position              */
#define GP_INTEN_FLIEN1_Msk              (0x1ul << GP_INTEN_FLIEN1_Pos)                    /*!< GP_T::INTEN: FLIEN1 Mask                  */

#define GP_INTEN_FLIEN2_Pos              (2)                                               /*!< GP_T::INTEN: FLIEN2 Position              */
#define GP_INTEN_FLIEN2_Msk              (0x1ul << GP_INTEN_FLIEN2_Pos)                    /*!< GP_T::INTEN: FLIEN2 Mask                  */

#define GP_INTEN_FLIEN3_Pos              (3)                                               /*!< GP_T::INTEN: FLIEN3 Position              */
#define GP_INTEN_FLIEN3_Msk              (0x1ul << GP_INTEN_FLIEN3_Pos)                    /*!< GP_T::INTEN: FLIEN3 Mask                  */

#define GP_INTEN_FLIEN4_Pos              (4)                                               /*!< GP_T::INTEN: FLIEN4 Position              */
#define GP_INTEN_FLIEN4_Msk              (0x1ul << GP_INTEN_FLIEN4_Pos)                    /*!< GP_T::INTEN: FLIEN4 Mask                  */

#define GP_INTEN_FLIEN5_Pos              (5)                                               /*!< GP_T::INTEN: FLIEN5 Position              */
#define GP_INTEN_FLIEN5_Msk              (0x1ul << GP_INTEN_FLIEN5_Pos)                    /*!< GP_T::INTEN: FLIEN5 Mask                  */

#define GP_INTEN_FLIEN6_Pos              (6)                                               /*!< GP_T::INTEN: FLIEN6 Position              */
#define GP_INTEN_FLIEN6_Msk              (0x1ul << GP_INTEN_FLIEN6_Pos)                    /*!< GP_T::INTEN: FLIEN6 Mask                  */

#define GP_INTEN_FLIEN7_Pos              (7)                                               /*!< GP_T::INTEN: FLIEN7 Position              */
#define GP_INTEN_FLIEN7_Msk              (0x1ul << GP_INTEN_FLIEN7_Pos)                    /*!< GP_T::INTEN: FLIEN7 Mask                  */

#define GP_INTEN_RHIEN0_Pos              (16)                                              /*!< GP_T::INTEN: RHIEN0 Position              */
#define GP_INTEN_RHIEN0_Msk              (0x1ul << GP_INTEN_RHIEN0_Pos)                    /*!< GP_T::INTEN: RHIEN0 Mask                  */

#define GP_INTEN_RHIEN1_Pos              (17)                                              /*!< GP_T::INTEN: RHIEN1 Position              */
#define GP_INTEN_RHIEN1_Msk              (0x1ul << GP_INTEN_RHIEN1_Pos)                    /*!< GP_T::INTEN: RHIEN1 Mask                  */

#define GP_INTEN_RHIEN2_Pos              (18)                                              /*!< GP_T::INTEN: RHIEN2 Position              */
#define GP_INTEN_RHIEN2_Msk              (0x1ul << GP_INTEN_RHIEN2_Pos)                    /*!< GP_T::INTEN: RHIEN2 Mask                  */

#define GP_INTEN_RHIEN3_Pos              (19)                                              /*!< GP_T::INTEN: RHIEN3 Position              */
#define GP_INTEN_RHIEN3_Msk              (0x1ul << GP_INTEN_RHIEN3_Pos)                    /*!< GP_T::INTEN: RHIEN3 Mask                  */

#define GP_INTEN_RHIEN4_Pos              (20)                                              /*!< GP_T::INTEN: RHIEN4 Position              */
#define GP_INTEN_RHIEN4_Msk              (0x1ul << GP_INTEN_RHIEN4_Pos)                    /*!< GP_T::INTEN: RHIEN4 Mask                  */

#define GP_INTEN_RHIEN5_Pos              (21)                                              /*!< GP_T::INTEN: RHIEN5 Position              */
#define GP_INTEN_RHIEN5_Msk              (0x1ul << GP_INTEN_RHIEN5_Pos)                    /*!< GP_T::INTEN: RHIEN5 Mask                  */

#define GP_INTEN_RHIEN6_Pos              (22)                                              /*!< GP_T::INTEN: RHIEN6 Position              */
#define GP_INTEN_RHIEN6_Msk              (0x1ul << GP_INTEN_RHIEN6_Pos)                    /*!< GP_T::INTEN: RHIEN6 Mask                  */

#define GP_INTEN_RHIEN7_Pos              (23)                                              /*!< GP_T::INTEN: RHIEN7 Position              */
#define GP_INTEN_RHIEN7_Msk              (0x1ul << GP_INTEN_RHIEN7_Pos)                    /*!< GP_T::INTEN: RHIEN7 Mask                  */

#define GP_INTSRC_INTSRC0_Pos            (0)                                               /*!< GP_T::INTSRC: INTSRC0 Position            */
#define GP_INTSRC_INTSRC0_Msk            (0x1ul << GP_INTSRC_INTSRC0_Pos)                  /*!< GP_T::INTSRC: INTSRC0 Mask                */

#define GP_INTSRC_INTSRC1_Pos            (1)                                               /*!< GP_T::INTSRC: INTSRC1 Position            */
#define GP_INTSRC_INTSRC1_Msk            (0x1ul << GP_INTSRC_INTSRC1_Pos)                  /*!< GP_T::INTSRC: INTSRC1 Mask                */

#define GP_INTSRC_INTSRC2_Pos            (2)                                               /*!< GP_T::INTSRC: INTSRC2 Position            */
#define GP_INTSRC_INTSRC2_Msk            (0x1ul << GP_INTSRC_INTSRC2_Pos)                  /*!< GP_T::INTSRC: INTSRC2 Mask                */

#define GP_INTSRC_INTSRC3_Pos            (3)                                               /*!< GP_T::INTSRC: INTSRC3 Position            */
#define GP_INTSRC_INTSRC3_Msk            (0x1ul << GP_INTSRC_INTSRC3_Pos)                  /*!< GP_T::INTSRC: INTSRC3 Mask                */

#define GP_INTSRC_INTSRC4_Pos            (4)                                               /*!< GP_T::INTSRC: INTSRC4 Position            */
#define GP_INTSRC_INTSRC4_Msk            (0x1ul << GP_INTSRC_INTSRC4_Pos)                  /*!< GP_T::INTSRC: INTSRC4 Mask                */

#define GP_INTSRC_INTSRC5_Pos            (5)                                               /*!< GP_T::INTSRC: INTSRC5 Position            */
#define GP_INTSRC_INTSRC5_Msk            (0x1ul << GP_INTSRC_INTSRC5_Pos)                  /*!< GP_T::INTSRC: INTSRC5 Mask                */

#define GP_INTSRC_INTSRC6_Pos            (6)                                               /*!< GP_T::INTSRC: INTSRC6 Position            */
#define GP_INTSRC_INTSRC6_Msk            (0x1ul << GP_INTSRC_INTSRC6_Pos)                  /*!< GP_T::INTSRC: INTSRC6 Mask                */

#define GP_INTSRC_INTSRC7_Pos            (7)                                               /*!< GP_T::INTSRC: INTSRC7 Position            */
#define GP_INTSRC_INTSRC7_Msk            (0x1ul << GP_INTSRC_INTSRC7_Pos)                  /*!< GP_T::INTSRC: INTSRC7 Mask                */

#define GP_INTSRC_INTSRC8_Pos            (8)                                               /*!< GP_T::INTSRC: INTSRC8 Position            */
#define GP_INTSRC_INTSRC8_Msk            (0x1ul << GP_INTSRC_INTSRC8_Pos)                  /*!< GP_T::INTSRC: INTSRC8 Mask                */

#define GP_INTSRC_INTSRC9_Pos            (9)                                               /*!< GP_T::INTSRC: INTSRC9 Position            */
#define GP_INTSRC_INTSRC9_Msk            (0x1ul << GP_INTSRC_INTSRC9_Pos)                  /*!< GP_T::INTSRC: INTSRC9 Mask                */

#define GP_INTSRC_INTSRC10_Pos           (10)                                              /*!< GP_T::INTSRC: INTSRC10 Position           */
#define GP_INTSRC_INTSRC10_Msk           (0x1ul << GP_INTSRC_INTSRC10_Pos)                 /*!< GP_T::INTSRC: INTSRC10 Mask               */

#define GP_INTSRC_INTSRC11_Pos           (11)                                              /*!< GP_T::INTSRC: INTSRC11 Position           */
#define GP_INTSRC_INTSRC11_Msk           (0x1ul << GP_INTSRC_INTSRC11_Pos)                 /*!< GP_T::INTSRC: INTSRC11 Mask               */

#define GP_INTSRC_INTSRC12_Pos           (12)                                              /*!< GP_T::INTSRC: INTSRC12 Position           */
#define GP_INTSRC_INTSRC12_Msk           (0x1ul << GP_INTSRC_INTSRC12_Pos)                 /*!< GP_T::INTSRC: INTSRC12 Mask               */

#define GP_INTSRC_INTSRC13_Pos           (13)                                              /*!< GP_T::INTSRC: INTSRC13 Position           */
#define GP_INTSRC_INTSRC13_Msk           (0x1ul << GP_INTSRC_INTSRC13_Pos)                 /*!< GP_T::INTSRC: INTSRC13 Mask               */

#define GP_INTSRC_INTSRC14_Pos           (14)                                              /*!< GP_T::INTSRC: INTSRC14 Position           */
#define GP_INTSRC_INTSRC14_Msk           (0x1ul << GP_INTSRC_INTSRC14_Pos)                 /*!< GP_T::INTSRC: INTSRC14 Mask               */

#define GP_INTSRC_INTSRC15_Pos           (15)                                              /*!< GP_T::INTSRC: INTSRC15 Position           */
#define GP_INTSRC_INTSRC15_Msk           (0x1ul << GP_INTSRC_INTSRC15_Pos)                 /*!< GP_T::INTSRC: INTSRC15 Mask               */

#define GP_MODE_MODE0_Pos                (0)                                               /*!< GP_T::MODE: MODE0 Position                */
#define GP_MODE_MODE0_Msk                (0x3ul << GP_MODE_MODE0_Pos)                      /*!< GP_T::MODE: MODE0 Mask                    */

#define GP_MODE_MODE1_Pos                (2)                                               /*!< GP_T::MODE: MODE1 Position                */
#define GP_MODE_MODE1_Msk                (0x3ul << GP_MODE_MODE1_Pos)                      /*!< GP_T::MODE: MODE1 Mask                    */

#define GP_MODE_MODE2_Pos                (4)                                               /*!< GP_T::MODE: MODE2 Position                */
#define GP_MODE_MODE2_Msk                (0x3ul << GP_MODE_MODE2_Pos)                      /*!< GP_T::MODE: MODE2 Mask                    */

#define GP_MODE_MODE3_Pos                (6)                                               /*!< GP_T::MODE: MODE3 Position                */
#define GP_MODE_MODE3_Msk                (0x3ul << GP_MODE_MODE3_Pos)                      /*!< GP_T::MODE: MODE3 Mask                    */

#define GP_MODE_MODE4_Pos                (8)                                               /*!< GP_T::MODE: MODE4 Position                */
#define GP_MODE_MODE4_Msk                (0x3ul << GP_MODE_MODE4_Pos)                      /*!< GP_T::MODE: MODE4 Mask                    */

#define GP_MODE_MODE5_Pos                (10)                                              /*!< GP_T::MODE: MODE5 Position                */
#define GP_MODE_MODE5_Msk                (0x3ul << GP_MODE_MODE5_Pos)                      /*!< GP_T::MODE: MODE5 Mask                    */

#define GP_DINOFF_DINOFF0_Pos            (16)                                              /*!< GP_T::DINOFF: DINOFF0 Position            */
#define GP_DINOFF_DINOFF0_Msk            (0x1ul << GP_DINOFF_DINOFF0_Pos)                  /*!< GP_T::DINOFF: DINOFF0 Mask                */

#define GP_DINOFF_DINOFF1_Pos            (17)                                              /*!< GP_T::DINOFF: DINOFF1 Position            */
#define GP_DINOFF_DINOFF1_Msk            (0x1ul << GP_DINOFF_DINOFF1_Pos)                  /*!< GP_T::DINOFF: DINOFF1 Mask                */

#define GP_DINOFF_DINOFF2_Pos            (18)                                              /*!< GP_T::DINOFF: DINOFF2 Position            */
#define GP_DINOFF_DINOFF2_Msk            (0x1ul << GP_DINOFF_DINOFF2_Pos)                  /*!< GP_T::DINOFF: DINOFF2 Mask                */

#define GP_DINOFF_DINOFF3_Pos            (19)                                              /*!< GP_T::DINOFF: DINOFF3 Position            */
#define GP_DINOFF_DINOFF3_Msk            (0x1ul << GP_DINOFF_DINOFF3_Pos)                  /*!< GP_T::DINOFF: DINOFF3 Mask                */

#define GP_DINOFF_DINOFF4_Pos            (20)                                              /*!< GP_T::DINOFF: DINOFF4 Position            */
#define GP_DINOFF_DINOFF4_Msk            (0x1ul << GP_DINOFF_DINOFF4_Pos)                  /*!< GP_T::DINOFF: DINOFF4 Mask                */

#define GP_DINOFF_DINOFF5_Pos            (21)                                              /*!< GP_T::DINOFF: DINOFF5 Position            */
#define GP_DINOFF_DINOFF5_Msk            (0x1ul << GP_DINOFF_DINOFF5_Pos)                  /*!< GP_T::DINOFF: DINOFF5 Mask                */

#define GP_DINOFF_DINOFF6_Pos            (22)                                              /*!< GP_T::DINOFF: DINOFF6 Position            */
#define GP_DINOFF_DINOFF6_Msk            (0x1ul << GP_DINOFF_DINOFF6_Pos)                  /*!< GP_T::DINOFF: DINOFF6 Mask                */

#define GP_DINOFF_DINOFF7_Pos            (23)                                              /*!< GP_T::DINOFF: DINOFF7 Position            */
#define GP_DINOFF_DINOFF7_Msk            (0x1ul << GP_DINOFF_DINOFF7_Pos)                  /*!< GP_T::DINOFF: DINOFF7 Mask                */

#define GP_DOUT_DOUT0_Pos                (0)                                               /*!< GP_T::DOUT: DOUT0 Position                */
#define GP_DOUT_DOUT0_Msk                (0x1ul << GP_DOUT_DOUT0_Pos)                      /*!< GP_T::DOUT: DOUT0 Mask                    */

#define GP_DOUT_DOUT1_Pos                (1)                                               /*!< GP_T::DOUT: DOUT1 Position                */
#define GP_DOUT_DOUT1_Msk                (0x1ul << GP_DOUT_DOUT1_Pos)                      /*!< GP_T::DOUT: DOUT1 Mask                    */

#define GP_DOUT_DOUT2_Pos                (2)                                               /*!< GP_T::DOUT: DOUT2 Position                */
#define GP_DOUT_DOUT2_Msk                (0x1ul << GP_DOUT_DOUT2_Pos)                      /*!< GP_T::DOUT: DOUT2 Mask                    */

#define GP_DOUT_DOUT3_Pos                (3)                                               /*!< GP_T::DOUT: DOUT3 Position                */
#define GP_DOUT_DOUT3_Msk                (0x1ul << GP_DOUT_DOUT3_Pos)                      /*!< GP_T::DOUT: DOUT3 Mask                    */

#define GP_DOUT_DOUT4_Pos                (4)                                               /*!< GP_T::DOUT: DOUT4 Position                */
#define GP_DOUT_DOUT4_Msk                (0x1ul << GP_DOUT_DOUT4_Pos)                      /*!< GP_T::DOUT: DOUT4 Mask                    */

#define GP_DOUT_DOUT5_Pos                (5)                                               /*!< GP_T::DOUT: DOUT5 Position                */
#define GP_DOUT_DOUT5_Msk                (0x1ul << GP_DOUT_DOUT5_Pos)                      /*!< GP_T::DOUT: DOUT5 Mask                    */

#define GP_DOUT_DOUT6_Pos                (6)                                               /*!< GP_T::DOUT: DOUT6 Position                */
#define GP_DOUT_DOUT6_Msk                (0x1ul << GP_DOUT_DOUT6_Pos)                      /*!< GP_T::DOUT: DOUT6 Mask                    */

#define GP_DOUT_DOUT7_Pos                (7)                                               /*!< GP_T::DOUT: DOUT7 Position                */
#define GP_DOUT_DOUT7_Msk                (0x1ul << GP_DOUT_DOUT7_Pos)                      /*!< GP_T::DOUT: DOUT7 Mask                    */

#define GP_DATMSK_DATMSK0_Pos            (0)                                               /*!< GP_T::DATMSK: DATMSK0 Position            */
#define GP_DATMSK_DATMSK0_Msk            (0x1ul << GP_DATMSK_DATMSK0_Pos)                  /*!< GP_T::DATMSK: DATMSK0 Mask                */

#define GP_DATMSK_DATMSK1_Pos            (1)                                               /*!< GP_T::DATMSK: DATMSK1 Position            */
#define GP_DATMSK_DATMSK1_Msk            (0x1ul << GP_DATMSK_DATMSK1_Pos)                  /*!< GP_T::DATMSK: DATMSK1 Mask                */

#define GP_DATMSK_DATMSK2_Pos            (2)                                               /*!< GP_T::DATMSK: DATMSK2 Position            */
#define GP_DATMSK_DATMSK2_Msk            (0x1ul << GP_DATMSK_DATMSK2_Pos)                  /*!< GP_T::DATMSK: DATMSK2 Mask                */

#define GP_DATMSK_DATMSK3_Pos            (3)                                               /*!< GP_T::DATMSK: DATMSK3 Position            */
#define GP_DATMSK_DATMSK3_Msk            (0x1ul << GP_DATMSK_DATMSK3_Pos)                  /*!< GP_T::DATMSK: DATMSK3 Mask                */

#define GP_DATMSK_DATMSK4_Pos            (4)                                               /*!< GP_T::DATMSK: DATMSK4 Position            */
#define GP_DATMSK_DATMSK4_Msk            (0x1ul << GP_DATMSK_DATMSK4_Pos)                  /*!< GP_T::DATMSK: DATMSK4 Mask                */

#define GP_DATMSK_DATMSK5_Pos            (5)                                               /*!< GP_T::DATMSK: DATMSK5 Position            */
#define GP_DATMSK_DATMSK5_Msk            (0x1ul << GP_DATMSK_DATMSK5_Pos)                  /*!< GP_T::DATMSK: DATMSK5 Mask                */

#define GP_DATMSK_DATMSK6_Pos            (6)                                               /*!< GP_T::DATMSK: DATMSK6 Position            */
#define GP_DATMSK_DATMSK6_Msk            (0x1ul << GP_DATMSK_DATMSK6_Pos)                  /*!< GP_T::DATMSK: DATMSK6 Mask                */

#define GP_DATMSK_DATMSK7_Pos            (7)                                               /*!< GP_T::DATMSK: DATMSK7 Position            */
#define GP_DATMSK_DATMSK7_Msk            (0x1ul << GP_DATMSK_DATMSK7_Pos)                  /*!< GP_T::DATMSK: DATMSK7 Mask                */

#define GP_PIN_PIN0_Pos                  (0)                                               /*!< GP_T::PIN: PIN0 Position                  */
#define GP_PIN_PIN0_Msk                  (0x1ul << GP_PIN_PIN0_Pos)                        /*!< GP_T::PIN: PIN0 Mask                      */

#define GP_PIN_PIN1_Pos                  (1)                                               /*!< GP_T::PIN: PIN1 Position                  */
#define GP_PIN_PIN1_Msk                  (0x1ul << GP_PIN_PIN1_Pos)                        /*!< GP_T::PIN: PIN1 Mask                      */

#define GP_PIN_PIN2_Pos                  (2)                                               /*!< GP_T::PIN: PIN2 Position                  */
#define GP_PIN_PIN2_Msk                  (0x1ul << GP_PIN_PIN2_Pos)                        /*!< GP_T::PIN: PIN2 Mask                      */

#define GP_PIN_PIN3_Pos                  (3)                                               /*!< GP_T::PIN: PIN3 Position                  */
#define GP_PIN_PIN3_Msk                  (0x1ul << GP_PIN_PIN3_Pos)                        /*!< GP_T::PIN: PIN3 Mask                      */

#define GP_PIN_PIN4_Pos                  (4)                                               /*!< GP_T::PIN: PIN4 Position                  */
#define GP_PIN_PIN4_Msk                  (0x1ul << GP_PIN_PIN4_Pos)                        /*!< GP_T::PIN: PIN4 Mask                      */

#define GP_PIN_PIN5_Pos                  (5)                                               /*!< GP_T::PIN: PIN5 Position                  */
#define GP_PIN_PIN5_Msk                  (0x1ul << GP_PIN_PIN5_Pos)                        /*!< GP_T::PIN: PIN5 Mask                      */

#define GP_PIN_PIN6_Pos                  (6)                                               /*!< GP_T::PIN: PIN6 Position                  */
#define GP_PIN_PIN6_Msk                  (0x1ul << GP_PIN_PIN6_Pos)                        /*!< GP_T::PIN: PIN6 Mask                      */

#define GP_PIN_PIN7_Pos                  (7)                                               /*!< GP_T::PIN: PIN7 Position                  */
#define GP_PIN_PIN7_Msk                  (0x1ul << GP_PIN_PIN7_Pos)                        /*!< GP_T::PIN: PIN7 Mask                      */

#define GP_DBEN_DBEN0_Pos                (0)                                               /*!< GP_T::DBEN: DBEN0 Position                */
#define GP_DBEN_DBEN0_Msk                (0x1ul << GP_DBEN_DBEN0_Pos)                      /*!< GP_T::DBEN: DBEN0 Mask                    */

#define GP_DBEN_DBEN1_Pos                (1)                                               /*!< GP_T::DBEN: DBEN1 Position                */
#define GP_DBEN_DBEN1_Msk                (0x1ul << GP_DBEN_DBEN1_Pos)                      /*!< GP_T::DBEN: DBEN1 Mask                    */

#define GP_DBEN_DBEN2_Pos                (2)                                               /*!< GP_T::DBEN: DBEN2 Position                */
#define GP_DBEN_DBEN2_Msk                (0x1ul << GP_DBEN_DBEN2_Pos)                      /*!< GP_T::DBEN: DBEN2 Mask                    */

#define GP_DBEN_DBEN3_Pos                (3)                                               /*!< GP_T::DBEN: DBEN3 Position                */
#define GP_DBEN_DBEN3_Msk                (0x1ul << GP_DBEN_DBEN3_Pos)                      /*!< GP_T::DBEN: DBEN3 Mask                    */

#define GP_DBEN_DBEN4_Pos                (4)                                               /*!< GP_T::DBEN: DBEN4 Position                */
#define GP_DBEN_DBEN4_Msk                (0x1ul << GP_DBEN_DBEN4_Pos)                      /*!< GP_T::DBEN: DBEN4 Mask                    */

#define GP_DBEN_DBEN5_Pos                (5)                                               /*!< GP_T::DBEN: DBEN5 Position                */
#define GP_DBEN_DBEN5_Msk                (0x1ul << GP_DBEN_DBEN5_Pos)                      /*!< GP_T::DBEN: DBEN5 Mask                    */

#define GP_DBEN_DBEN6_Pos                (6)                                               /*!< GP_T::DBEN: DBEN6 Position                */
#define GP_DBEN_DBEN6_Msk                (0x1ul << GP_DBEN_DBEN6_Pos)                      /*!< GP_T::DBEN: DBEN6 Mask                    */

#define GP_DBEN_DBEN7_Pos                (7)                                               /*!< GP_T::DBEN: DBEN7 Position                */
#define GP_DBEN_DBEN7_Msk                (0x1ul << GP_DBEN_DBEN7_Pos)                      /*!< GP_T::DBEN: DBEN7 Mask                    */

#define GP_INTTYPE_TYPE0_Pos             (0)                                               /*!< GP_T::INTTYPE: TYPE0 Position             */
#define GP_INTTYPE_TYPE0_Msk             (0x1ul << GP_INTTYPE_TYPE0_Pos)                   /*!< GP_T::INTTYPE: TYPE0 Mask                 */

#define GP_INTTYPE_TYPE1_Pos             (1)                                               /*!< GP_T::INTTYPE: TYPE1 Position             */
#define GP_INTTYPE_TYPE1_Msk             (0x1ul << GP_INTTYPE_TYPE1_Pos)                   /*!< GP_T::INTTYPE: TYPE1 Mask                 */

#define GP_INTTYPE_TYPE2_Pos             (2)                                               /*!< GP_T::INTTYPE: TYPE2 Position             */
#define GP_INTTYPE_TYPE2_Msk             (0x1ul << GP_INTTYPE_TYPE2_Pos)                   /*!< GP_T::INTTYPE: TYPE2 Mask                 */

#define GP_INTTYPE_TYPE3_Pos             (3)                                               /*!< GP_T::INTTYPE: TYPE3 Position             */
#define GP_INTTYPE_TYPE3_Msk             (0x1ul << GP_INTTYPE_TYPE3_Pos)                   /*!< GP_T::INTTYPE: TYPE3 Mask                 */

#define GP_INTTYPE_TYPE4_Pos             (4)                                               /*!< GP_T::INTTYPE: TYPE4 Position             */
#define GP_INTTYPE_TYPE4_Msk             (0x1ul << GP_INTTYPE_TYPE4_Pos)                   /*!< GP_T::INTTYPE: TYPE4 Mask                 */

#define GP_INTTYPE_TYPE5_Pos             (5)                                               /*!< GP_T::INTTYPE: TYPE5 Position             */
#define GP_INTTYPE_TYPE5_Msk             (0x1ul << GP_INTTYPE_TYPE5_Pos)                   /*!< GP_T::INTTYPE: TYPE5 Mask                 */

#define GP_INTTYPE_TYPE6_Pos             (6)                                               /*!< GP_T::INTTYPE: TYPE6 Position             */
#define GP_INTTYPE_TYPE6_Msk             (0x1ul << GP_INTTYPE_TYPE6_Pos)                   /*!< GP_T::INTTYPE: TYPE6 Mask                 */

#define GP_INTTYPE_TYPE7_Pos             (7)                                               /*!< GP_T::INTTYPE: TYPE7 Position             */
#define GP_INTTYPE_TYPE7_Msk             (0x1ul << GP_INTTYPE_TYPE7_Pos)                   /*!< GP_T::INTTYPE: TYPE7 Mask                 */

#define GP_INTEN_FLIEN0_Pos              (0)                                               /*!< GP_T::INTEN: FLIEN0 Position              */
#define GP_INTEN_FLIEN0_Msk              (0x1ul << GP_INTEN_FLIEN0_Pos)                    /*!< GP_T::INTEN: FLIEN0 Mask                  */

#define GP_INTEN_FLIEN1_Pos              (1)                                               /*!< GP_T::INTEN: FLIEN1 Position              */
#define GP_INTEN_FLIEN1_Msk              (0x1ul << GP_INTEN_FLIEN1_Pos)                    /*!< GP_T::INTEN: FLIEN1 Mask                  */

#define GP_INTEN_FLIEN2_Pos              (2)                                               /*!< GP_T::INTEN: FLIEN2 Position              */
#define GP_INTEN_FLIEN2_Msk              (0x1ul << GP_INTEN_FLIEN2_Pos)                    /*!< GP_T::INTEN: FLIEN2 Mask                  */

#define GP_INTEN_FLIEN3_Pos              (3)                                               /*!< GP_T::INTEN: FLIEN3 Position              */
#define GP_INTEN_FLIEN3_Msk              (0x1ul << GP_INTEN_FLIEN3_Pos)                    /*!< GP_T::INTEN: FLIEN3 Mask                  */

#define GP_INTEN_FLIEN4_Pos              (4)                                               /*!< GP_T::INTEN: FLIEN4 Position              */
#define GP_INTEN_FLIEN4_Msk              (0x1ul << GP_INTEN_FLIEN4_Pos)                    /*!< GP_T::INTEN: FLIEN4 Mask                  */

#define GP_INTEN_FLIEN5_Pos              (5)                                               /*!< GP_T::INTEN: FLIEN5 Position              */
#define GP_INTEN_FLIEN5_Msk              (0x1ul << GP_INTEN_FLIEN5_Pos)                    /*!< GP_T::INTEN: FLIEN5 Mask                  */

#define GP_INTEN_FLIEN6_Pos              (6)                                               /*!< GP_T::INTEN: FLIEN6 Position              */
#define GP_INTEN_FLIEN6_Msk              (0x1ul << GP_INTEN_FLIEN6_Pos)                    /*!< GP_T::INTEN: FLIEN6 Mask                  */

#define GP_INTEN_FLIEN7_Pos              (7)                                               /*!< GP_T::INTEN: FLIEN7 Position              */
#define GP_INTEN_FLIEN7_Msk              (0x1ul << GP_INTEN_FLIEN7_Pos)                    /*!< GP_T::INTEN: FLIEN7 Mask                  */

#define GP_INTEN_RHIEN0_Pos              (16)                                              /*!< GP_T::INTEN: RHIEN0 Position              */
#define GP_INTEN_RHIEN0_Msk              (0x1ul << GP_INTEN_RHIEN0_Pos)                    /*!< GP_T::INTEN: RHIEN0 Mask                  */

#define GP_INTEN_RHIEN1_Pos              (17)                                              /*!< GP_T::INTEN: RHIEN1 Position              */
#define GP_INTEN_RHIEN1_Msk              (0x1ul << GP_INTEN_RHIEN1_Pos)                    /*!< GP_T::INTEN: RHIEN1 Mask                  */

#define GP_INTEN_RHIEN2_Pos              (18)                                              /*!< GP_T::INTEN: RHIEN2 Position              */
#define GP_INTEN_RHIEN2_Msk              (0x1ul << GP_INTEN_RHIEN2_Pos)                    /*!< GP_T::INTEN: RHIEN2 Mask                  */

#define GP_INTEN_RHIEN3_Pos              (19)                                              /*!< GP_T::INTEN: RHIEN3 Position              */
#define GP_INTEN_RHIEN3_Msk              (0x1ul << GP_INTEN_RHIEN3_Pos)                    /*!< GP_T::INTEN: RHIEN3 Mask                  */

#define GP_INTEN_RHIEN4_Pos              (20)                                              /*!< GP_T::INTEN: RHIEN4 Position              */
#define GP_INTEN_RHIEN4_Msk              (0x1ul << GP_INTEN_RHIEN4_Pos)                    /*!< GP_T::INTEN: RHIEN4 Mask                  */

#define GP_INTEN_RHIEN5_Pos              (21)                                              /*!< GP_T::INTEN: RHIEN5 Position              */
#define GP_INTEN_RHIEN5_Msk              (0x1ul << GP_INTEN_RHIEN5_Pos)                    /*!< GP_T::INTEN: RHIEN5 Mask                  */

#define GP_INTEN_RHIEN6_Pos              (22)                                              /*!< GP_T::INTEN: RHIEN6 Position              */
#define GP_INTEN_RHIEN6_Msk              (0x1ul << GP_INTEN_RHIEN6_Pos)                    /*!< GP_T::INTEN: RHIEN6 Mask                  */

#define GP_INTEN_RHIEN7_Pos              (23)                                              /*!< GP_T::INTEN: RHIEN7 Position              */
#define GP_INTEN_RHIEN7_Msk              (0x1ul << GP_INTEN_RHIEN7_Pos)                    /*!< GP_T::INTEN: RHIEN7 Mask                  */

#define GP_INTSRC_INTSRC0_Pos            (0)                                               /*!< GP_T::INTSRC: INTSRC0 Position            */
#define GP_INTSRC_INTSRC0_Msk            (0x1ul << GP_INTSRC_INTSRC0_Pos)                  /*!< GP_T::INTSRC: INTSRC0 Mask                */

#define GP_INTSRC_INTSRC1_Pos            (1)                                               /*!< GP_T::INTSRC: INTSRC1 Position            */
#define GP_INTSRC_INTSRC1_Msk            (0x1ul << GP_INTSRC_INTSRC1_Pos)                  /*!< GP_T::INTSRC: INTSRC1 Mask                */

#define GP_INTSRC_INTSRC2_Pos            (2)                                               /*!< GP_T::INTSRC: INTSRC2 Position            */
#define GP_INTSRC_INTSRC2_Msk            (0x1ul << GP_INTSRC_INTSRC2_Pos)                  /*!< GP_T::INTSRC: INTSRC2 Mask                */

#define GP_INTSRC_INTSRC3_Pos            (3)                                               /*!< GP_T::INTSRC: INTSRC3 Position            */
#define GP_INTSRC_INTSRC3_Msk            (0x1ul << GP_INTSRC_INTSRC3_Pos)                  /*!< GP_T::INTSRC: INTSRC3 Mask                */

#define GP_INTSRC_INTSRC4_Pos            (4)                                               /*!< GP_T::INTSRC: INTSRC4 Position            */
#define GP_INTSRC_INTSRC4_Msk            (0x1ul << GP_INTSRC_INTSRC4_Pos)                  /*!< GP_T::INTSRC: INTSRC4 Mask                */

#define GP_INTSRC_INTSRC5_Pos            (5)                                               /*!< GP_T::INTSRC: INTSRC5 Position            */
#define GP_INTSRC_INTSRC5_Msk            (0x1ul << GP_INTSRC_INTSRC5_Pos)                  /*!< GP_T::INTSRC: INTSRC5 Mask                */

#define GP_INTSRC_INTSRC6_Pos            (6)                                               /*!< GP_T::INTSRC: INTSRC6 Position            */
#define GP_INTSRC_INTSRC6_Msk            (0x1ul << GP_INTSRC_INTSRC6_Pos)                  /*!< GP_T::INTSRC: INTSRC6 Mask                */

#define GP_INTSRC_INTSRC7_Pos            (7)                                               /*!< GP_T::INTSRC: INTSRC7 Position            */
#define GP_INTSRC_INTSRC7_Msk            (0x1ul << GP_INTSRC_INTSRC7_Pos)                  /*!< GP_T::INTSRC: INTSRC7 Mask                */

#define GP_INTSRC_INTSRC8_Pos            (8)                                               /*!< GP_T::INTSRC: INTSRC8 Position            */
#define GP_INTSRC_INTSRC8_Msk            (0x1ul << GP_INTSRC_INTSRC8_Pos)                  /*!< GP_T::INTSRC: INTSRC8 Mask                */

#define GP_INTSRC_INTSRC9_Pos            (9)                                               /*!< GP_T::INTSRC: INTSRC9 Position            */
#define GP_INTSRC_INTSRC9_Msk            (0x1ul << GP_INTSRC_INTSRC9_Pos)                  /*!< GP_T::INTSRC: INTSRC9 Mask                */

#define GP_INTSRC_INTSRC10_Pos           (10)                                              /*!< GP_T::INTSRC: INTSRC10 Position           */
#define GP_INTSRC_INTSRC10_Msk           (0x1ul << GP_INTSRC_INTSRC10_Pos)                 /*!< GP_T::INTSRC: INTSRC10 Mask               */

#define GP_INTSRC_INTSRC11_Pos           (11)                                              /*!< GP_T::INTSRC: INTSRC11 Position           */
#define GP_INTSRC_INTSRC11_Msk           (0x1ul << GP_INTSRC_INTSRC11_Pos)                 /*!< GP_T::INTSRC: INTSRC11 Mask               */

#define GP_INTSRC_INTSRC12_Pos           (12)                                              /*!< GP_T::INTSRC: INTSRC12 Position           */
#define GP_INTSRC_INTSRC12_Msk           (0x1ul << GP_INTSRC_INTSRC12_Pos)                 /*!< GP_T::INTSRC: INTSRC12 Mask               */

#define GP_INTSRC_INTSRC13_Pos           (13)                                              /*!< GP_T::INTSRC: INTSRC13 Position           */
#define GP_INTSRC_INTSRC13_Msk           (0x1ul << GP_INTSRC_INTSRC13_Pos)                 /*!< GP_T::INTSRC: INTSRC13 Mask               */

#define GP_INTSRC_INTSRC14_Pos           (14)                                              /*!< GP_T::INTSRC: INTSRC14 Position           */
#define GP_INTSRC_INTSRC14_Msk           (0x1ul << GP_INTSRC_INTSRC14_Pos)                 /*!< GP_T::INTSRC: INTSRC14 Mask               */

#define GP_INTSRC_INTSRC15_Pos           (15)                                              /*!< GP_T::INTSRC: INTSRC15 Position           */
#define GP_INTSRC_INTSRC15_Msk           (0x1ul << GP_INTSRC_INTSRC15_Pos)                 /*!< GP_T::INTSRC: INTSRC15 Mask               */

#define GP_MODE_MODE0_Pos                (0)                                               /*!< GP_T::MODE: MODE0 Position                */
#define GP_MODE_MODE0_Msk                (0x3ul << GP_MODE_MODE0_Pos)                      /*!< GP_T::MODE: MODE0 Mask                    */

#define GP_MODE_MODE1_Pos                (2)                                               /*!< GP_T::MODE: MODE1 Position                */
#define GP_MODE_MODE1_Msk                (0x3ul << GP_MODE_MODE1_Pos)                      /*!< GP_T::MODE: MODE1 Mask                    */

#define GP_MODE_MODE2_Pos                (4)                                               /*!< GP_T::MODE: MODE2 Position                */
#define GP_MODE_MODE2_Msk                (0x3ul << GP_MODE_MODE2_Pos)                      /*!< GP_T::MODE: MODE2 Mask                    */

#define GP_MODE_MODE3_Pos                (6)                                               /*!< GP_T::MODE: MODE3 Position                */
#define GP_MODE_MODE3_Msk                (0x3ul << GP_MODE_MODE3_Pos)                      /*!< GP_T::MODE: MODE3 Mask                    */

#define GP_MODE_MODE4_Pos                (8)                                               /*!< GP_T::MODE: MODE4 Position                */
#define GP_MODE_MODE4_Msk                (0x3ul << GP_MODE_MODE4_Pos)                      /*!< GP_T::MODE: MODE4 Mask                    */

#define GP_MODE_MODE5_Pos                (10)                                              /*!< GP_T::MODE: MODE5 Position                */
#define GP_MODE_MODE5_Msk                (0x3ul << GP_MODE_MODE5_Pos)                      /*!< GP_T::MODE: MODE5 Mask                    */

#define GP_DINOFF_DINOFF0_Pos            (16)                                              /*!< GP_T::DINOFF: DINOFF0 Position            */
#define GP_DINOFF_DINOFF0_Msk            (0x1ul << GP_DINOFF_DINOFF0_Pos)                  /*!< GP_T::DINOFF: DINOFF0 Mask                */

#define GP_DINOFF_DINOFF1_Pos            (17)                                              /*!< GP_T::DINOFF: DINOFF1 Position            */
#define GP_DINOFF_DINOFF1_Msk            (0x1ul << GP_DINOFF_DINOFF1_Pos)                  /*!< GP_T::DINOFF: DINOFF1 Mask                */

#define GP_DINOFF_DINOFF2_Pos            (18)                                              /*!< GP_T::DINOFF: DINOFF2 Position            */
#define GP_DINOFF_DINOFF2_Msk            (0x1ul << GP_DINOFF_DINOFF2_Pos)                  /*!< GP_T::DINOFF: DINOFF2 Mask                */

#define GP_DINOFF_DINOFF3_Pos            (19)                                              /*!< GP_T::DINOFF: DINOFF3 Position            */
#define GP_DINOFF_DINOFF3_Msk            (0x1ul << GP_DINOFF_DINOFF3_Pos)                  /*!< GP_T::DINOFF: DINOFF3 Mask                */

#define GP_DINOFF_DINOFF4_Pos            (20)                                              /*!< GP_T::DINOFF: DINOFF4 Position            */
#define GP_DINOFF_DINOFF4_Msk            (0x1ul << GP_DINOFF_DINOFF4_Pos)                  /*!< GP_T::DINOFF: DINOFF4 Mask                */

#define GP_DINOFF_DINOFF5_Pos            (21)                                              /*!< GP_T::DINOFF: DINOFF5 Position            */
#define GP_DINOFF_DINOFF5_Msk            (0x1ul << GP_DINOFF_DINOFF5_Pos)                  /*!< GP_T::DINOFF: DINOFF5 Mask                */

#define GP_DINOFF_DINOFF6_Pos            (22)                                              /*!< GP_T::DINOFF: DINOFF6 Position            */
#define GP_DINOFF_DINOFF6_Msk            (0x1ul << GP_DINOFF_DINOFF6_Pos)                  /*!< GP_T::DINOFF: DINOFF6 Mask                */

#define GP_DINOFF_DINOFF7_Pos            (23)                                              /*!< GP_T::DINOFF: DINOFF7 Position            */
#define GP_DINOFF_DINOFF7_Msk            (0x1ul << GP_DINOFF_DINOFF7_Pos)                  /*!< GP_T::DINOFF: DINOFF7 Mask                */

#define GP_DOUT_DOUT0_Pos                (0)                                               /*!< GP_T::DOUT: DOUT0 Position                */
#define GP_DOUT_DOUT0_Msk                (0x1ul << GP_DOUT_DOUT0_Pos)                      /*!< GP_T::DOUT: DOUT0 Mask                    */

#define GP_DOUT_DOUT1_Pos                (1)                                               /*!< GP_T::DOUT: DOUT1 Position                */
#define GP_DOUT_DOUT1_Msk                (0x1ul << GP_DOUT_DOUT1_Pos)                      /*!< GP_T::DOUT: DOUT1 Mask                    */

#define GP_DOUT_DOUT2_Pos                (2)                                               /*!< GP_T::DOUT: DOUT2 Position                */
#define GP_DOUT_DOUT2_Msk                (0x1ul << GP_DOUT_DOUT2_Pos)                      /*!< GP_T::DOUT: DOUT2 Mask                    */

#define GP_DOUT_DOUT3_Pos                (3)                                               /*!< GP_T::DOUT: DOUT3 Position                */
#define GP_DOUT_DOUT3_Msk                (0x1ul << GP_DOUT_DOUT3_Pos)                      /*!< GP_T::DOUT: DOUT3 Mask                    */

#define GP_DOUT_DOUT4_Pos                (4)                                               /*!< GP_T::DOUT: DOUT4 Position                */
#define GP_DOUT_DOUT4_Msk                (0x1ul << GP_DOUT_DOUT4_Pos)                      /*!< GP_T::DOUT: DOUT4 Mask                    */

#define GP_DOUT_DOUT5_Pos                (5)                                               /*!< GP_T::DOUT: DOUT5 Position                */
#define GP_DOUT_DOUT5_Msk                (0x1ul << GP_DOUT_DOUT5_Pos)                      /*!< GP_T::DOUT: DOUT5 Mask                    */

#define GP_DOUT_DOUT6_Pos                (6)                                               /*!< GP_T::DOUT: DOUT6 Position                */
#define GP_DOUT_DOUT6_Msk                (0x1ul << GP_DOUT_DOUT6_Pos)                      /*!< GP_T::DOUT: DOUT6 Mask                    */

#define GP_DOUT_DOUT7_Pos                (7)                                               /*!< GP_T::DOUT: DOUT7 Position                */
#define GP_DOUT_DOUT7_Msk                (0x1ul << GP_DOUT_DOUT7_Pos)                      /*!< GP_T::DOUT: DOUT7 Mask                    */

#define GP_DATMSK_DATMSK0_Pos            (0)                                               /*!< GP_T::DATMSK: DATMSK0 Position            */
#define GP_DATMSK_DATMSK0_Msk            (0x1ul << GP_DATMSK_DATMSK0_Pos)                  /*!< GP_T::DATMSK: DATMSK0 Mask                */

#define GP_DATMSK_DATMSK1_Pos            (1)                                               /*!< GP_T::DATMSK: DATMSK1 Position            */
#define GP_DATMSK_DATMSK1_Msk            (0x1ul << GP_DATMSK_DATMSK1_Pos)                  /*!< GP_T::DATMSK: DATMSK1 Mask                */

#define GP_DATMSK_DATMSK2_Pos            (2)                                               /*!< GP_T::DATMSK: DATMSK2 Position            */
#define GP_DATMSK_DATMSK2_Msk            (0x1ul << GP_DATMSK_DATMSK2_Pos)                  /*!< GP_T::DATMSK: DATMSK2 Mask                */

#define GP_DATMSK_DATMSK3_Pos            (3)                                               /*!< GP_T::DATMSK: DATMSK3 Position            */
#define GP_DATMSK_DATMSK3_Msk            (0x1ul << GP_DATMSK_DATMSK3_Pos)                  /*!< GP_T::DATMSK: DATMSK3 Mask                */

#define GP_DATMSK_DATMSK4_Pos            (4)                                               /*!< GP_T::DATMSK: DATMSK4 Position            */
#define GP_DATMSK_DATMSK4_Msk            (0x1ul << GP_DATMSK_DATMSK4_Pos)                  /*!< GP_T::DATMSK: DATMSK4 Mask                */

#define GP_DATMSK_DATMSK5_Pos            (5)                                               /*!< GP_T::DATMSK: DATMSK5 Position            */
#define GP_DATMSK_DATMSK5_Msk            (0x1ul << GP_DATMSK_DATMSK5_Pos)                  /*!< GP_T::DATMSK: DATMSK5 Mask                */

#define GP_DATMSK_DATMSK6_Pos            (6)                                               /*!< GP_T::DATMSK: DATMSK6 Position            */
#define GP_DATMSK_DATMSK6_Msk            (0x1ul << GP_DATMSK_DATMSK6_Pos)                  /*!< GP_T::DATMSK: DATMSK6 Mask                */

#define GP_DATMSK_DATMSK7_Pos            (7)                                               /*!< GP_T::DATMSK: DATMSK7 Position            */
#define GP_DATMSK_DATMSK7_Msk            (0x1ul << GP_DATMSK_DATMSK7_Pos)                  /*!< GP_T::DATMSK: DATMSK7 Mask                */

#define GP_PIN_PIN0_Pos                  (0)                                               /*!< GP_T::PIN: PIN0 Position                  */
#define GP_PIN_PIN0_Msk                  (0x1ul << GP_PIN_PIN0_Pos)                        /*!< GP_T::PIN: PIN0 Mask                      */

#define GP_PIN_PIN1_Pos                  (1)                                               /*!< GP_T::PIN: PIN1 Position                  */
#define GP_PIN_PIN1_Msk                  (0x1ul << GP_PIN_PIN1_Pos)                        /*!< GP_T::PIN: PIN1 Mask                      */

#define GP_PIN_PIN2_Pos                  (2)                                               /*!< GP_T::PIN: PIN2 Position                  */
#define GP_PIN_PIN2_Msk                  (0x1ul << GP_PIN_PIN2_Pos)                        /*!< GP_T::PIN: PIN2 Mask                      */

#define GP_PIN_PIN3_Pos                  (3)                                               /*!< GP_T::PIN: PIN3 Position                  */
#define GP_PIN_PIN3_Msk                  (0x1ul << GP_PIN_PIN3_Pos)                        /*!< GP_T::PIN: PIN3 Mask                      */

#define GP_PIN_PIN4_Pos                  (4)                                               /*!< GP_T::PIN: PIN4 Position                  */
#define GP_PIN_PIN4_Msk                  (0x1ul << GP_PIN_PIN4_Pos)                        /*!< GP_T::PIN: PIN4 Mask                      */

#define GP_PIN_PIN5_Pos                  (5)                                               /*!< GP_T::PIN: PIN5 Position                  */
#define GP_PIN_PIN5_Msk                  (0x1ul << GP_PIN_PIN5_Pos)                        /*!< GP_T::PIN: PIN5 Mask                      */

#define GP_PIN_PIN6_Pos                  (6)                                               /*!< GP_T::PIN: PIN6 Position                  */
#define GP_PIN_PIN6_Msk                  (0x1ul << GP_PIN_PIN6_Pos)                        /*!< GP_T::PIN: PIN6 Mask                      */

#define GP_PIN_PIN7_Pos                  (7)                                               /*!< GP_T::PIN: PIN7 Position                  */
#define GP_PIN_PIN7_Msk                  (0x1ul << GP_PIN_PIN7_Pos)                        /*!< GP_T::PIN: PIN7 Mask                      */

#define GP_DBEN_DBEN0_Pos                (0)                                               /*!< GP_T::DBEN: DBEN0 Position                */
#define GP_DBEN_DBEN0_Msk                (0x1ul << GP_DBEN_DBEN0_Pos)                      /*!< GP_T::DBEN: DBEN0 Mask                    */

#define GP_DBEN_DBEN1_Pos                (1)                                               /*!< GP_T::DBEN: DBEN1 Position                */
#define GP_DBEN_DBEN1_Msk                (0x1ul << GP_DBEN_DBEN1_Pos)                      /*!< GP_T::DBEN: DBEN1 Mask                    */

#define GP_DBEN_DBEN2_Pos                (2)                                               /*!< GP_T::DBEN: DBEN2 Position                */
#define GP_DBEN_DBEN2_Msk                (0x1ul << GP_DBEN_DBEN2_Pos)                      /*!< GP_T::DBEN: DBEN2 Mask                    */

#define GP_DBEN_DBEN3_Pos                (3)                                               /*!< GP_T::DBEN: DBEN3 Position                */
#define GP_DBEN_DBEN3_Msk                (0x1ul << GP_DBEN_DBEN3_Pos)                      /*!< GP_T::DBEN: DBEN3 Mask                    */

#define GP_DBEN_DBEN4_Pos                (4)                                               /*!< GP_T::DBEN: DBEN4 Position                */
#define GP_DBEN_DBEN4_Msk                (0x1ul << GP_DBEN_DBEN4_Pos)                      /*!< GP_T::DBEN: DBEN4 Mask                    */

#define GP_DBEN_DBEN5_Pos                (5)                                               /*!< GP_T::DBEN: DBEN5 Position                */
#define GP_DBEN_DBEN5_Msk                (0x1ul << GP_DBEN_DBEN5_Pos)                      /*!< GP_T::DBEN: DBEN5 Mask                    */

#define GP_DBEN_DBEN6_Pos                (6)                                               /*!< GP_T::DBEN: DBEN6 Position                */
#define GP_DBEN_DBEN6_Msk                (0x1ul << GP_DBEN_DBEN6_Pos)                      /*!< GP_T::DBEN: DBEN6 Mask                    */

#define GP_DBEN_DBEN7_Pos                (7)                                               /*!< GP_T::DBEN: DBEN7 Position                */
#define GP_DBEN_DBEN7_Msk                (0x1ul << GP_DBEN_DBEN7_Pos)                      /*!< GP_T::DBEN: DBEN7 Mask                    */

#define GP_INTTYPE_TYPE0_Pos             (0)                                               /*!< GP_T::INTTYPE: TYPE0 Position             */
#define GP_INTTYPE_TYPE0_Msk             (0x1ul << GP_INTTYPE_TYPE0_Pos)                   /*!< GP_T::INTTYPE: TYPE0 Mask                 */

#define GP_INTTYPE_TYPE1_Pos             (1)                                               /*!< GP_T::INTTYPE: TYPE1 Position             */
#define GP_INTTYPE_TYPE1_Msk             (0x1ul << GP_INTTYPE_TYPE1_Pos)                   /*!< GP_T::INTTYPE: TYPE1 Mask                 */

#define GP_INTTYPE_TYPE2_Pos             (2)                                               /*!< GP_T::INTTYPE: TYPE2 Position             */
#define GP_INTTYPE_TYPE2_Msk             (0x1ul << GP_INTTYPE_TYPE2_Pos)                   /*!< GP_T::INTTYPE: TYPE2 Mask                 */

#define GP_INTTYPE_TYPE3_Pos             (3)                                               /*!< GP_T::INTTYPE: TYPE3 Position             */
#define GP_INTTYPE_TYPE3_Msk             (0x1ul << GP_INTTYPE_TYPE3_Pos)                   /*!< GP_T::INTTYPE: TYPE3 Mask                 */

#define GP_INTTYPE_TYPE4_Pos             (4)                                               /*!< GP_T::INTTYPE: TYPE4 Position             */
#define GP_INTTYPE_TYPE4_Msk             (0x1ul << GP_INTTYPE_TYPE4_Pos)                   /*!< GP_T::INTTYPE: TYPE4 Mask                 */

#define GP_INTTYPE_TYPE5_Pos             (5)                                               /*!< GP_T::INTTYPE: TYPE5 Position             */
#define GP_INTTYPE_TYPE5_Msk             (0x1ul << GP_INTTYPE_TYPE5_Pos)                   /*!< GP_T::INTTYPE: TYPE5 Mask                 */

#define GP_INTTYPE_TYPE6_Pos             (6)                                               /*!< GP_T::INTTYPE: TYPE6 Position             */
#define GP_INTTYPE_TYPE6_Msk             (0x1ul << GP_INTTYPE_TYPE6_Pos)                   /*!< GP_T::INTTYPE: TYPE6 Mask                 */

#define GP_INTTYPE_TYPE7_Pos             (7)                                               /*!< GP_T::INTTYPE: TYPE7 Position             */
#define GP_INTTYPE_TYPE7_Msk             (0x1ul << GP_INTTYPE_TYPE7_Pos)                   /*!< GP_T::INTTYPE: TYPE7 Mask                 */

#define GP_INTEN_FLIEN0_Pos              (0)                                               /*!< GP_T::INTEN: FLIEN0 Position              */
#define GP_INTEN_FLIEN0_Msk              (0x1ul << GP_INTEN_FLIEN0_Pos)                    /*!< GP_T::INTEN: FLIEN0 Mask                  */

#define GP_INTEN_FLIEN1_Pos              (1)                                               /*!< GP_T::INTEN: FLIEN1 Position              */
#define GP_INTEN_FLIEN1_Msk              (0x1ul << GP_INTEN_FLIEN1_Pos)                    /*!< GP_T::INTEN: FLIEN1 Mask                  */

#define GP_INTEN_FLIEN2_Pos              (2)                                               /*!< GP_T::INTEN: FLIEN2 Position              */
#define GP_INTEN_FLIEN2_Msk              (0x1ul << GP_INTEN_FLIEN2_Pos)                    /*!< GP_T::INTEN: FLIEN2 Mask                  */

#define GP_INTEN_FLIEN3_Pos              (3)                                               /*!< GP_T::INTEN: FLIEN3 Position              */
#define GP_INTEN_FLIEN3_Msk              (0x1ul << GP_INTEN_FLIEN3_Pos)                    /*!< GP_T::INTEN: FLIEN3 Mask                  */

#define GP_INTEN_FLIEN4_Pos              (4)                                               /*!< GP_T::INTEN: FLIEN4 Position              */
#define GP_INTEN_FLIEN4_Msk              (0x1ul << GP_INTEN_FLIEN4_Pos)                    /*!< GP_T::INTEN: FLIEN4 Mask                  */

#define GP_INTEN_FLIEN5_Pos              (5)                                               /*!< GP_T::INTEN: FLIEN5 Position              */
#define GP_INTEN_FLIEN5_Msk              (0x1ul << GP_INTEN_FLIEN5_Pos)                    /*!< GP_T::INTEN: FLIEN5 Mask                  */

#define GP_INTEN_FLIEN6_Pos              (6)                                               /*!< GP_T::INTEN: FLIEN6 Position              */
#define GP_INTEN_FLIEN6_Msk              (0x1ul << GP_INTEN_FLIEN6_Pos)                    /*!< GP_T::INTEN: FLIEN6 Mask                  */

#define GP_INTEN_FLIEN7_Pos              (7)                                               /*!< GP_T::INTEN: FLIEN7 Position              */
#define GP_INTEN_FLIEN7_Msk              (0x1ul << GP_INTEN_FLIEN7_Pos)                    /*!< GP_T::INTEN: FLIEN7 Mask                  */

#define GP_INTEN_RHIEN0_Pos              (16)                                              /*!< GP_T::INTEN: RHIEN0 Position              */
#define GP_INTEN_RHIEN0_Msk              (0x1ul << GP_INTEN_RHIEN0_Pos)                    /*!< GP_T::INTEN: RHIEN0 Mask                  */

#define GP_INTEN_RHIEN1_Pos              (17)                                              /*!< GP_T::INTEN: RHIEN1 Position              */
#define GP_INTEN_RHIEN1_Msk              (0x1ul << GP_INTEN_RHIEN1_Pos)                    /*!< GP_T::INTEN: RHIEN1 Mask                  */

#define GP_INTEN_RHIEN2_Pos              (18)                                              /*!< GP_T::INTEN: RHIEN2 Position              */
#define GP_INTEN_RHIEN2_Msk              (0x1ul << GP_INTEN_RHIEN2_Pos)                    /*!< GP_T::INTEN: RHIEN2 Mask                  */

#define GP_INTEN_RHIEN3_Pos              (19)                                              /*!< GP_T::INTEN: RHIEN3 Position              */
#define GP_INTEN_RHIEN3_Msk              (0x1ul << GP_INTEN_RHIEN3_Pos)                    /*!< GP_T::INTEN: RHIEN3 Mask                  */

#define GP_INTEN_RHIEN4_Pos              (20)                                              /*!< GP_T::INTEN: RHIEN4 Position              */
#define GP_INTEN_RHIEN4_Msk              (0x1ul << GP_INTEN_RHIEN4_Pos)                    /*!< GP_T::INTEN: RHIEN4 Mask                  */

#define GP_INTEN_RHIEN5_Pos              (21)                                              /*!< GP_T::INTEN: RHIEN5 Position              */
#define GP_INTEN_RHIEN5_Msk              (0x1ul << GP_INTEN_RHIEN5_Pos)                    /*!< GP_T::INTEN: RHIEN5 Mask                  */

#define GP_INTEN_RHIEN6_Pos              (22)                                              /*!< GP_T::INTEN: RHIEN6 Position              */
#define GP_INTEN_RHIEN6_Msk              (0x1ul << GP_INTEN_RHIEN6_Pos)                    /*!< GP_T::INTEN: RHIEN6 Mask                  */

#define GP_INTEN_RHIEN7_Pos              (23)                                              /*!< GP_T::INTEN: RHIEN7 Position              */
#define GP_INTEN_RHIEN7_Msk              (0x1ul << GP_INTEN_RHIEN7_Pos)                    /*!< GP_T::INTEN: RHIEN7 Mask                  */

#define GP_INTSRC_INTSRC0_Pos            (0)                                               /*!< GP_T::INTSRC: INTSRC0 Position            */
#define GP_INTSRC_INTSRC0_Msk            (0x1ul << GP_INTSRC_INTSRC0_Pos)                  /*!< GP_T::INTSRC: INTSRC0 Mask                */

#define GP_INTSRC_INTSRC1_Pos            (1)                                               /*!< GP_T::INTSRC: INTSRC1 Position            */
#define GP_INTSRC_INTSRC1_Msk            (0x1ul << GP_INTSRC_INTSRC1_Pos)                  /*!< GP_T::INTSRC: INTSRC1 Mask                */

#define GP_INTSRC_INTSRC2_Pos            (2)                                               /*!< GP_T::INTSRC: INTSRC2 Position            */
#define GP_INTSRC_INTSRC2_Msk            (0x1ul << GP_INTSRC_INTSRC2_Pos)                  /*!< GP_T::INTSRC: INTSRC2 Mask                */

#define GP_INTSRC_INTSRC3_Pos            (3)                                               /*!< GP_T::INTSRC: INTSRC3 Position            */
#define GP_INTSRC_INTSRC3_Msk            (0x1ul << GP_INTSRC_INTSRC3_Pos)                  /*!< GP_T::INTSRC: INTSRC3 Mask                */

#define GP_INTSRC_INTSRC4_Pos            (4)                                               /*!< GP_T::INTSRC: INTSRC4 Position            */
#define GP_INTSRC_INTSRC4_Msk            (0x1ul << GP_INTSRC_INTSRC4_Pos)                  /*!< GP_T::INTSRC: INTSRC4 Mask                */

#define GP_INTSRC_INTSRC5_Pos            (5)                                               /*!< GP_T::INTSRC: INTSRC5 Position            */
#define GP_INTSRC_INTSRC5_Msk            (0x1ul << GP_INTSRC_INTSRC5_Pos)                  /*!< GP_T::INTSRC: INTSRC5 Mask                */

#define GP_INTSRC_INTSRC6_Pos            (6)                                               /*!< GP_T::INTSRC: INTSRC6 Position            */
#define GP_INTSRC_INTSRC6_Msk            (0x1ul << GP_INTSRC_INTSRC6_Pos)                  /*!< GP_T::INTSRC: INTSRC6 Mask                */

#define GP_INTSRC_INTSRC7_Pos            (7)                                               /*!< GP_T::INTSRC: INTSRC7 Position            */
#define GP_INTSRC_INTSRC7_Msk            (0x1ul << GP_INTSRC_INTSRC7_Pos)                  /*!< GP_T::INTSRC: INTSRC7 Mask                */

#define GP_INTSRC_INTSRC8_Pos            (8)                                               /*!< GP_T::INTSRC: INTSRC8 Position            */
#define GP_INTSRC_INTSRC8_Msk            (0x1ul << GP_INTSRC_INTSRC8_Pos)                  /*!< GP_T::INTSRC: INTSRC8 Mask                */

#define GP_INTSRC_INTSRC9_Pos            (9)                                               /*!< GP_T::INTSRC: INTSRC9 Position            */
#define GP_INTSRC_INTSRC9_Msk            (0x1ul << GP_INTSRC_INTSRC9_Pos)                  /*!< GP_T::INTSRC: INTSRC9 Mask                */

#define GP_INTSRC_INTSRC10_Pos           (10)                                              /*!< GP_T::INTSRC: INTSRC10 Position           */
#define GP_INTSRC_INTSRC10_Msk           (0x1ul << GP_INTSRC_INTSRC10_Pos)                 /*!< GP_T::INTSRC: INTSRC10 Mask               */

#define GP_INTSRC_INTSRC11_Pos           (11)                                              /*!< GP_T::INTSRC: INTSRC11 Position           */
#define GP_INTSRC_INTSRC11_Msk           (0x1ul << GP_INTSRC_INTSRC11_Pos)                 /*!< GP_T::INTSRC: INTSRC11 Mask               */

#define GP_INTSRC_INTSRC12_Pos           (12)                                              /*!< GP_T::INTSRC: INTSRC12 Position           */
#define GP_INTSRC_INTSRC12_Msk           (0x1ul << GP_INTSRC_INTSRC12_Pos)                 /*!< GP_T::INTSRC: INTSRC12 Mask               */

#define GP_INTSRC_INTSRC13_Pos           (13)                                              /*!< GP_T::INTSRC: INTSRC13 Position           */
#define GP_INTSRC_INTSRC13_Msk           (0x1ul << GP_INTSRC_INTSRC13_Pos)                 /*!< GP_T::INTSRC: INTSRC13 Mask               */

#define GP_INTSRC_INTSRC14_Pos           (14)                                              /*!< GP_T::INTSRC: INTSRC14 Position           */
#define GP_INTSRC_INTSRC14_Msk           (0x1ul << GP_INTSRC_INTSRC14_Pos)                 /*!< GP_T::INTSRC: INTSRC14 Mask               */

#define GP_INTSRC_INTSRC15_Pos           (15)                                              /*!< GP_T::INTSRC: INTSRC15 Position           */
#define GP_INTSRC_INTSRC15_Msk           (0x1ul << GP_INTSRC_INTSRC15_Pos)                 /*!< GP_T::INTSRC: INTSRC15 Mask               */

#define GP_MODE_MODE0_Pos                (0)                                               /*!< GP_T::MODE: MODE0 Position                */
#define GP_MODE_MODE0_Msk                (0x3ul << GP_MODE_MODE0_Pos)                      /*!< GP_T::MODE: MODE0 Mask                    */

#define GP_MODE_MODE1_Pos                (2)                                               /*!< GP_T::MODE: MODE1 Position                */
#define GP_MODE_MODE1_Msk                (0x3ul << GP_MODE_MODE1_Pos)                      /*!< GP_T::MODE: MODE1 Mask                    */

#define GP_MODE_MODE2_Pos                (4)                                               /*!< GP_T::MODE: MODE2 Position                */
#define GP_MODE_MODE2_Msk                (0x3ul << GP_MODE_MODE2_Pos)                      /*!< GP_T::MODE: MODE2 Mask                    */

#define GP_MODE_MODE3_Pos                (6)                                               /*!< GP_T::MODE: MODE3 Position                */
#define GP_MODE_MODE3_Msk                (0x3ul << GP_MODE_MODE3_Pos)                      /*!< GP_T::MODE: MODE3 Mask                    */

#define GP_MODE_MODE4_Pos                (8)                                               /*!< GP_T::MODE: MODE4 Position                */
#define GP_MODE_MODE4_Msk                (0x3ul << GP_MODE_MODE4_Pos)                      /*!< GP_T::MODE: MODE4 Mask                    */

#define GP_MODE_MODE5_Pos                (10)                                              /*!< GP_T::MODE: MODE5 Position                */
#define GP_MODE_MODE5_Msk                (0x3ul << GP_MODE_MODE5_Pos)                      /*!< GP_T::MODE: MODE5 Mask                    */

#define GP_DINOFF_DINOFF0_Pos            (16)                                              /*!< GP_T::DINOFF: DINOFF0 Position            */
#define GP_DINOFF_DINOFF0_Msk            (0x1ul << GP_DINOFF_DINOFF0_Pos)                  /*!< GP_T::DINOFF: DINOFF0 Mask                */

#define GP_DINOFF_DINOFF1_Pos            (17)                                              /*!< GP_T::DINOFF: DINOFF1 Position            */
#define GP_DINOFF_DINOFF1_Msk            (0x1ul << GP_DINOFF_DINOFF1_Pos)                  /*!< GP_T::DINOFF: DINOFF1 Mask                */

#define GP_DINOFF_DINOFF2_Pos            (18)                                              /*!< GP_T::DINOFF: DINOFF2 Position            */
#define GP_DINOFF_DINOFF2_Msk            (0x1ul << GP_DINOFF_DINOFF2_Pos)                  /*!< GP_T::DINOFF: DINOFF2 Mask                */

#define GP_DINOFF_DINOFF3_Pos            (19)                                              /*!< GP_T::DINOFF: DINOFF3 Position            */
#define GP_DINOFF_DINOFF3_Msk            (0x1ul << GP_DINOFF_DINOFF3_Pos)                  /*!< GP_T::DINOFF: DINOFF3 Mask                */

#define GP_DINOFF_DINOFF4_Pos            (20)                                              /*!< GP_T::DINOFF: DINOFF4 Position            */
#define GP_DINOFF_DINOFF4_Msk            (0x1ul << GP_DINOFF_DINOFF4_Pos)                  /*!< GP_T::DINOFF: DINOFF4 Mask                */

#define GP_DINOFF_DINOFF5_Pos            (21)                                              /*!< GP_T::DINOFF: DINOFF5 Position            */
#define GP_DINOFF_DINOFF5_Msk            (0x1ul << GP_DINOFF_DINOFF5_Pos)                  /*!< GP_T::DINOFF: DINOFF5 Mask                */

#define GP_DINOFF_DINOFF6_Pos            (22)                                              /*!< GP_T::DINOFF: DINOFF6 Position            */
#define GP_DINOFF_DINOFF6_Msk            (0x1ul << GP_DINOFF_DINOFF6_Pos)                  /*!< GP_T::DINOFF: DINOFF6 Mask                */

#define GP_DINOFF_DINOFF7_Pos            (23)                                              /*!< GP_T::DINOFF: DINOFF7 Position            */
#define GP_DINOFF_DINOFF7_Msk            (0x1ul << GP_DINOFF_DINOFF7_Pos)                  /*!< GP_T::DINOFF: DINOFF7 Mask                */

#define GP_DOUT_DOUT0_Pos                (0)                                               /*!< GP_T::DOUT: DOUT0 Position                */
#define GP_DOUT_DOUT0_Msk                (0x1ul << GP_DOUT_DOUT0_Pos)                      /*!< GP_T::DOUT: DOUT0 Mask                    */

#define GP_DOUT_DOUT1_Pos                (1)                                               /*!< GP_T::DOUT: DOUT1 Position                */
#define GP_DOUT_DOUT1_Msk                (0x1ul << GP_DOUT_DOUT1_Pos)                      /*!< GP_T::DOUT: DOUT1 Mask                    */

#define GP_DOUT_DOUT2_Pos                (2)                                               /*!< GP_T::DOUT: DOUT2 Position                */
#define GP_DOUT_DOUT2_Msk                (0x1ul << GP_DOUT_DOUT2_Pos)                      /*!< GP_T::DOUT: DOUT2 Mask                    */

#define GP_DOUT_DOUT3_Pos                (3)                                               /*!< GP_T::DOUT: DOUT3 Position                */
#define GP_DOUT_DOUT3_Msk                (0x1ul << GP_DOUT_DOUT3_Pos)                      /*!< GP_T::DOUT: DOUT3 Mask                    */

#define GP_DOUT_DOUT4_Pos                (4)                                               /*!< GP_T::DOUT: DOUT4 Position                */
#define GP_DOUT_DOUT4_Msk                (0x1ul << GP_DOUT_DOUT4_Pos)                      /*!< GP_T::DOUT: DOUT4 Mask                    */

#define GP_DOUT_DOUT5_Pos                (5)                                               /*!< GP_T::DOUT: DOUT5 Position                */
#define GP_DOUT_DOUT5_Msk                (0x1ul << GP_DOUT_DOUT5_Pos)                      /*!< GP_T::DOUT: DOUT5 Mask                    */

#define GP_DOUT_DOUT6_Pos                (6)                                               /*!< GP_T::DOUT: DOUT6 Position                */
#define GP_DOUT_DOUT6_Msk                (0x1ul << GP_DOUT_DOUT6_Pos)                      /*!< GP_T::DOUT: DOUT6 Mask                    */

#define GP_DOUT_DOUT7_Pos                (7)                                               /*!< GP_T::DOUT: DOUT7 Position                */
#define GP_DOUT_DOUT7_Msk                (0x1ul << GP_DOUT_DOUT7_Pos)                      /*!< GP_T::DOUT: DOUT7 Mask                    */

#define GP_DATMSK_DATMSK0_Pos            (0)                                               /*!< GP_T::DATMSK: DATMSK0 Position            */
#define GP_DATMSK_DATMSK0_Msk            (0x1ul << GP_DATMSK_DATMSK0_Pos)                  /*!< GP_T::DATMSK: DATMSK0 Mask                */

#define GP_DATMSK_DATMSK1_Pos            (1)                                               /*!< GP_T::DATMSK: DATMSK1 Position            */
#define GP_DATMSK_DATMSK1_Msk            (0x1ul << GP_DATMSK_DATMSK1_Pos)                  /*!< GP_T::DATMSK: DATMSK1 Mask                */

#define GP_DATMSK_DATMSK2_Pos            (2)                                               /*!< GP_T::DATMSK: DATMSK2 Position            */
#define GP_DATMSK_DATMSK2_Msk            (0x1ul << GP_DATMSK_DATMSK2_Pos)                  /*!< GP_T::DATMSK: DATMSK2 Mask                */

#define GP_DATMSK_DATMSK3_Pos            (3)                                               /*!< GP_T::DATMSK: DATMSK3 Position            */
#define GP_DATMSK_DATMSK3_Msk            (0x1ul << GP_DATMSK_DATMSK3_Pos)                  /*!< GP_T::DATMSK: DATMSK3 Mask                */

#define GP_DATMSK_DATMSK4_Pos            (4)                                               /*!< GP_T::DATMSK: DATMSK4 Position            */
#define GP_DATMSK_DATMSK4_Msk            (0x1ul << GP_DATMSK_DATMSK4_Pos)                  /*!< GP_T::DATMSK: DATMSK4 Mask                */

#define GP_DATMSK_DATMSK5_Pos            (5)                                               /*!< GP_T::DATMSK: DATMSK5 Position            */
#define GP_DATMSK_DATMSK5_Msk            (0x1ul << GP_DATMSK_DATMSK5_Pos)                  /*!< GP_T::DATMSK: DATMSK5 Mask                */

#define GP_DATMSK_DATMSK6_Pos            (6)                                               /*!< GP_T::DATMSK: DATMSK6 Position            */
#define GP_DATMSK_DATMSK6_Msk            (0x1ul << GP_DATMSK_DATMSK6_Pos)                  /*!< GP_T::DATMSK: DATMSK6 Mask                */

#define GP_DATMSK_DATMSK7_Pos            (7)                                               /*!< GP_T::DATMSK: DATMSK7 Position            */
#define GP_DATMSK_DATMSK7_Msk            (0x1ul << GP_DATMSK_DATMSK7_Pos)                  /*!< GP_T::DATMSK: DATMSK7 Mask                */

#define GP_PIN_PIN0_Pos                  (0)                                               /*!< GP_T::PIN: PIN0 Position                  */
#define GP_PIN_PIN0_Msk                  (0x1ul << GP_PIN_PIN0_Pos)                        /*!< GP_T::PIN: PIN0 Mask                      */

#define GP_PIN_PIN1_Pos                  (1)                                               /*!< GP_T::PIN: PIN1 Position                  */
#define GP_PIN_PIN1_Msk                  (0x1ul << GP_PIN_PIN1_Pos)                        /*!< GP_T::PIN: PIN1 Mask                      */

#define GP_PIN_PIN2_Pos                  (2)                                               /*!< GP_T::PIN: PIN2 Position                  */
#define GP_PIN_PIN2_Msk                  (0x1ul << GP_PIN_PIN2_Pos)                        /*!< GP_T::PIN: PIN2 Mask                      */

#define GP_PIN_PIN3_Pos                  (3)                                               /*!< GP_T::PIN: PIN3 Position                  */
#define GP_PIN_PIN3_Msk                  (0x1ul << GP_PIN_PIN3_Pos)                        /*!< GP_T::PIN: PIN3 Mask                      */

#define GP_PIN_PIN4_Pos                  (4)                                               /*!< GP_T::PIN: PIN4 Position                  */
#define GP_PIN_PIN4_Msk                  (0x1ul << GP_PIN_PIN4_Pos)                        /*!< GP_T::PIN: PIN4 Mask                      */

#define GP_PIN_PIN5_Pos                  (5)                                               /*!< GP_T::PIN: PIN5 Position                  */
#define GP_PIN_PIN5_Msk                  (0x1ul << GP_PIN_PIN5_Pos)                        /*!< GP_T::PIN: PIN5 Mask                      */

#define GP_PIN_PIN6_Pos                  (6)                                               /*!< GP_T::PIN: PIN6 Position                  */
#define GP_PIN_PIN6_Msk                  (0x1ul << GP_PIN_PIN6_Pos)                        /*!< GP_T::PIN: PIN6 Mask                      */

#define GP_PIN_PIN7_Pos                  (7)                                               /*!< GP_T::PIN: PIN7 Position                  */
#define GP_PIN_PIN7_Msk                  (0x1ul << GP_PIN_PIN7_Pos)                        /*!< GP_T::PIN: PIN7 Mask                      */

#define GP_DBEN_DBEN0_Pos                (0)                                               /*!< GP_T::DBEN: DBEN0 Position                */
#define GP_DBEN_DBEN0_Msk                (0x1ul << GP_DBEN_DBEN0_Pos)                      /*!< GP_T::DBEN: DBEN0 Mask                    */

#define GP_DBEN_DBEN1_Pos                (1)                                               /*!< GP_T::DBEN: DBEN1 Position                */
#define GP_DBEN_DBEN1_Msk                (0x1ul << GP_DBEN_DBEN1_Pos)                      /*!< GP_T::DBEN: DBEN1 Mask                    */

#define GP_DBEN_DBEN2_Pos                (2)                                               /*!< GP_T::DBEN: DBEN2 Position                */
#define GP_DBEN_DBEN2_Msk                (0x1ul << GP_DBEN_DBEN2_Pos)                      /*!< GP_T::DBEN: DBEN2 Mask                    */

#define GP_DBEN_DBEN3_Pos                (3)                                               /*!< GP_T::DBEN: DBEN3 Position                */
#define GP_DBEN_DBEN3_Msk                (0x1ul << GP_DBEN_DBEN3_Pos)                      /*!< GP_T::DBEN: DBEN3 Mask                    */

#define GP_DBEN_DBEN4_Pos                (4)                                               /*!< GP_T::DBEN: DBEN4 Position                */
#define GP_DBEN_DBEN4_Msk                (0x1ul << GP_DBEN_DBEN4_Pos)                      /*!< GP_T::DBEN: DBEN4 Mask                    */

#define GP_DBEN_DBEN5_Pos                (5)                                               /*!< GP_T::DBEN: DBEN5 Position                */
#define GP_DBEN_DBEN5_Msk                (0x1ul << GP_DBEN_DBEN5_Pos)                      /*!< GP_T::DBEN: DBEN5 Mask                    */

#define GP_DBEN_DBEN6_Pos                (6)                                               /*!< GP_T::DBEN: DBEN6 Position                */
#define GP_DBEN_DBEN6_Msk                (0x1ul << GP_DBEN_DBEN6_Pos)                      /*!< GP_T::DBEN: DBEN6 Mask                    */

#define GP_DBEN_DBEN7_Pos                (7)                                               /*!< GP_T::DBEN: DBEN7 Position                */
#define GP_DBEN_DBEN7_Msk                (0x1ul << GP_DBEN_DBEN7_Pos)                      /*!< GP_T::DBEN: DBEN7 Mask                    */

#define GP_INTTYPE_TYPE0_Pos             (0)                                               /*!< GP_T::INTTYPE: TYPE0 Position             */
#define GP_INTTYPE_TYPE0_Msk             (0x1ul << GP_INTTYPE_TYPE0_Pos)                   /*!< GP_T::INTTYPE: TYPE0 Mask                 */

#define GP_INTTYPE_TYPE1_Pos             (1)                                               /*!< GP_T::INTTYPE: TYPE1 Position             */
#define GP_INTTYPE_TYPE1_Msk             (0x1ul << GP_INTTYPE_TYPE1_Pos)                   /*!< GP_T::INTTYPE: TYPE1 Mask                 */

#define GP_INTTYPE_TYPE2_Pos             (2)                                               /*!< GP_T::INTTYPE: TYPE2 Position             */
#define GP_INTTYPE_TYPE2_Msk             (0x1ul << GP_INTTYPE_TYPE2_Pos)                   /*!< GP_T::INTTYPE: TYPE2 Mask                 */

#define GP_INTTYPE_TYPE3_Pos             (3)                                               /*!< GP_T::INTTYPE: TYPE3 Position             */
#define GP_INTTYPE_TYPE3_Msk             (0x1ul << GP_INTTYPE_TYPE3_Pos)                   /*!< GP_T::INTTYPE: TYPE3 Mask                 */

#define GP_INTTYPE_TYPE4_Pos             (4)                                               /*!< GP_T::INTTYPE: TYPE4 Position             */
#define GP_INTTYPE_TYPE4_Msk             (0x1ul << GP_INTTYPE_TYPE4_Pos)                   /*!< GP_T::INTTYPE: TYPE4 Mask                 */

#define GP_INTTYPE_TYPE5_Pos             (5)                                               /*!< GP_T::INTTYPE: TYPE5 Position             */
#define GP_INTTYPE_TYPE5_Msk             (0x1ul << GP_INTTYPE_TYPE5_Pos)                   /*!< GP_T::INTTYPE: TYPE5 Mask                 */

#define GP_INTTYPE_TYPE6_Pos             (6)                                               /*!< GP_T::INTTYPE: TYPE6 Position             */
#define GP_INTTYPE_TYPE6_Msk             (0x1ul << GP_INTTYPE_TYPE6_Pos)                   /*!< GP_T::INTTYPE: TYPE6 Mask                 */

#define GP_INTTYPE_TYPE7_Pos             (7)                                               /*!< GP_T::INTTYPE: TYPE7 Position             */
#define GP_INTTYPE_TYPE7_Msk             (0x1ul << GP_INTTYPE_TYPE7_Pos)                   /*!< GP_T::INTTYPE: TYPE7 Mask                 */

#define GP_INTEN_FLIEN0_Pos              (0)                                               /*!< GP_T::INTEN: FLIEN0 Position              */
#define GP_INTEN_FLIEN0_Msk              (0x1ul << GP_INTEN_FLIEN0_Pos)                    /*!< GP_T::INTEN: FLIEN0 Mask                  */

#define GP_INTEN_FLIEN1_Pos              (1)                                               /*!< GP_T::INTEN: FLIEN1 Position              */
#define GP_INTEN_FLIEN1_Msk              (0x1ul << GP_INTEN_FLIEN1_Pos)                    /*!< GP_T::INTEN: FLIEN1 Mask                  */

#define GP_INTEN_FLIEN2_Pos              (2)                                               /*!< GP_T::INTEN: FLIEN2 Position              */
#define GP_INTEN_FLIEN2_Msk              (0x1ul << GP_INTEN_FLIEN2_Pos)                    /*!< GP_T::INTEN: FLIEN2 Mask                  */

#define GP_INTEN_FLIEN3_Pos              (3)                                               /*!< GP_T::INTEN: FLIEN3 Position              */
#define GP_INTEN_FLIEN3_Msk              (0x1ul << GP_INTEN_FLIEN3_Pos)                    /*!< GP_T::INTEN: FLIEN3 Mask                  */

#define GP_INTEN_FLIEN4_Pos              (4)                                               /*!< GP_T::INTEN: FLIEN4 Position              */
#define GP_INTEN_FLIEN4_Msk              (0x1ul << GP_INTEN_FLIEN4_Pos)                    /*!< GP_T::INTEN: FLIEN4 Mask                  */

#define GP_INTEN_FLIEN5_Pos              (5)                                               /*!< GP_T::INTEN: FLIEN5 Position              */
#define GP_INTEN_FLIEN5_Msk              (0x1ul << GP_INTEN_FLIEN5_Pos)                    /*!< GP_T::INTEN: FLIEN5 Mask                  */

#define GP_INTEN_FLIEN6_Pos              (6)                                               /*!< GP_T::INTEN: FLIEN6 Position              */
#define GP_INTEN_FLIEN6_Msk              (0x1ul << GP_INTEN_FLIEN6_Pos)                    /*!< GP_T::INTEN: FLIEN6 Mask                  */

#define GP_INTEN_FLIEN7_Pos              (7)                                               /*!< GP_T::INTEN: FLIEN7 Position              */
#define GP_INTEN_FLIEN7_Msk              (0x1ul << GP_INTEN_FLIEN7_Pos)                    /*!< GP_T::INTEN: FLIEN7 Mask                  */

#define GP_INTEN_RHIEN0_Pos              (16)                                              /*!< GP_T::INTEN: RHIEN0 Position              */
#define GP_INTEN_RHIEN0_Msk              (0x1ul << GP_INTEN_RHIEN0_Pos)                    /*!< GP_T::INTEN: RHIEN0 Mask                  */

#define GP_INTEN_RHIEN1_Pos              (17)                                              /*!< GP_T::INTEN: RHIEN1 Position              */
#define GP_INTEN_RHIEN1_Msk              (0x1ul << GP_INTEN_RHIEN1_Pos)                    /*!< GP_T::INTEN: RHIEN1 Mask                  */

#define GP_INTEN_RHIEN2_Pos              (18)                                              /*!< GP_T::INTEN: RHIEN2 Position              */
#define GP_INTEN_RHIEN2_Msk              (0x1ul << GP_INTEN_RHIEN2_Pos)                    /*!< GP_T::INTEN: RHIEN2 Mask                  */

#define GP_INTEN_RHIEN3_Pos              (19)                                              /*!< GP_T::INTEN: RHIEN3 Position              */
#define GP_INTEN_RHIEN3_Msk              (0x1ul << GP_INTEN_RHIEN3_Pos)                    /*!< GP_T::INTEN: RHIEN3 Mask                  */

#define GP_INTEN_RHIEN4_Pos              (20)                                              /*!< GP_T::INTEN: RHIEN4 Position              */
#define GP_INTEN_RHIEN4_Msk              (0x1ul << GP_INTEN_RHIEN4_Pos)                    /*!< GP_T::INTEN: RHIEN4 Mask                  */

#define GP_INTEN_RHIEN5_Pos              (21)                                              /*!< GP_T::INTEN: RHIEN5 Position              */
#define GP_INTEN_RHIEN5_Msk              (0x1ul << GP_INTEN_RHIEN5_Pos)                    /*!< GP_T::INTEN: RHIEN5 Mask                  */

#define GP_INTEN_RHIEN6_Pos              (22)                                              /*!< GP_T::INTEN: RHIEN6 Position              */
#define GP_INTEN_RHIEN6_Msk              (0x1ul << GP_INTEN_RHIEN6_Pos)                    /*!< GP_T::INTEN: RHIEN6 Mask                  */

#define GP_INTEN_RHIEN7_Pos              (23)                                              /*!< GP_T::INTEN: RHIEN7 Position              */
#define GP_INTEN_RHIEN7_Msk              (0x1ul << GP_INTEN_RHIEN7_Pos)                    /*!< GP_T::INTEN: RHIEN7 Mask                  */

#define GP_INTSRC_INTSRC0_Pos            (0)                                               /*!< GP_T::INTSRC: INTSRC0 Position            */
#define GP_INTSRC_INTSRC0_Msk            (0x1ul << GP_INTSRC_INTSRC0_Pos)                  /*!< GP_T::INTSRC: INTSRC0 Mask                */

#define GP_INTSRC_INTSRC1_Pos            (1)                                               /*!< GP_T::INTSRC: INTSRC1 Position            */
#define GP_INTSRC_INTSRC1_Msk            (0x1ul << GP_INTSRC_INTSRC1_Pos)                  /*!< GP_T::INTSRC: INTSRC1 Mask                */

#define GP_INTSRC_INTSRC2_Pos            (2)                                               /*!< GP_T::INTSRC: INTSRC2 Position            */
#define GP_INTSRC_INTSRC2_Msk            (0x1ul << GP_INTSRC_INTSRC2_Pos)                  /*!< GP_T::INTSRC: INTSRC2 Mask                */

#define GP_INTSRC_INTSRC3_Pos            (3)                                               /*!< GP_T::INTSRC: INTSRC3 Position            */
#define GP_INTSRC_INTSRC3_Msk            (0x1ul << GP_INTSRC_INTSRC3_Pos)                  /*!< GP_T::INTSRC: INTSRC3 Mask                */

#define GP_INTSRC_INTSRC4_Pos            (4)                                               /*!< GP_T::INTSRC: INTSRC4 Position            */
#define GP_INTSRC_INTSRC4_Msk            (0x1ul << GP_INTSRC_INTSRC4_Pos)                  /*!< GP_T::INTSRC: INTSRC4 Mask                */

#define GP_INTSRC_INTSRC5_Pos            (5)                                               /*!< GP_T::INTSRC: INTSRC5 Position            */
#define GP_INTSRC_INTSRC5_Msk            (0x1ul << GP_INTSRC_INTSRC5_Pos)                  /*!< GP_T::INTSRC: INTSRC5 Mask                */

#define GP_INTSRC_INTSRC6_Pos            (6)                                               /*!< GP_T::INTSRC: INTSRC6 Position            */
#define GP_INTSRC_INTSRC6_Msk            (0x1ul << GP_INTSRC_INTSRC6_Pos)                  /*!< GP_T::INTSRC: INTSRC6 Mask                */

#define GP_INTSRC_INTSRC7_Pos            (7)                                               /*!< GP_T::INTSRC: INTSRC7 Position            */
#define GP_INTSRC_INTSRC7_Msk            (0x1ul << GP_INTSRC_INTSRC7_Pos)                  /*!< GP_T::INTSRC: INTSRC7 Mask                */

#define GP_INTSRC_INTSRC8_Pos            (8)                                               /*!< GP_T::INTSRC: INTSRC8 Position            */
#define GP_INTSRC_INTSRC8_Msk            (0x1ul << GP_INTSRC_INTSRC8_Pos)                  /*!< GP_T::INTSRC: INTSRC8 Mask                */

#define GP_INTSRC_INTSRC9_Pos            (9)                                               /*!< GP_T::INTSRC: INTSRC9 Position            */
#define GP_INTSRC_INTSRC9_Msk            (0x1ul << GP_INTSRC_INTSRC9_Pos)                  /*!< GP_T::INTSRC: INTSRC9 Mask                */

#define GP_INTSRC_INTSRC10_Pos           (10)                                              /*!< GP_T::INTSRC: INTSRC10 Position           */
#define GP_INTSRC_INTSRC10_Msk           (0x1ul << GP_INTSRC_INTSRC10_Pos)                 /*!< GP_T::INTSRC: INTSRC10 Mask               */

#define GP_INTSRC_INTSRC11_Pos           (11)                                              /*!< GP_T::INTSRC: INTSRC11 Position           */
#define GP_INTSRC_INTSRC11_Msk           (0x1ul << GP_INTSRC_INTSRC11_Pos)                 /*!< GP_T::INTSRC: INTSRC11 Mask               */

#define GP_INTSRC_INTSRC12_Pos           (12)                                              /*!< GP_T::INTSRC: INTSRC12 Position           */
#define GP_INTSRC_INTSRC12_Msk           (0x1ul << GP_INTSRC_INTSRC12_Pos)                 /*!< GP_T::INTSRC: INTSRC12 Mask               */

#define GP_INTSRC_INTSRC13_Pos           (13)                                              /*!< GP_T::INTSRC: INTSRC13 Position           */
#define GP_INTSRC_INTSRC13_Msk           (0x1ul << GP_INTSRC_INTSRC13_Pos)                 /*!< GP_T::INTSRC: INTSRC13 Mask               */

#define GP_INTSRC_INTSRC14_Pos           (14)                                              /*!< GP_T::INTSRC: INTSRC14 Position           */
#define GP_INTSRC_INTSRC14_Msk           (0x1ul << GP_INTSRC_INTSRC14_Pos)                 /*!< GP_T::INTSRC: INTSRC14 Mask               */

#define GP_INTSRC_INTSRC15_Pos           (15)                                              /*!< GP_T::INTSRC: INTSRC15 Position           */
#define GP_INTSRC_INTSRC15_Msk           (0x1ul << GP_INTSRC_INTSRC15_Pos)                 /*!< GP_T::INTSRC: INTSRC15 Mask               */

#define GP_MODE_MODE0_Pos                (0)                                               /*!< GP_T::MODE: MODE0 Position                */
#define GP_MODE_MODE0_Msk                (0x3ul << GP_MODE_MODE0_Pos)                      /*!< GP_T::MODE: MODE0 Mask                    */

#define GP_MODE_MODE1_Pos                (2)                                               /*!< GP_T::MODE: MODE1 Position                */
#define GP_MODE_MODE1_Msk                (0x3ul << GP_MODE_MODE1_Pos)                      /*!< GP_T::MODE: MODE1 Mask                    */

#define GP_MODE_MODE2_Pos                (4)                                               /*!< GP_T::MODE: MODE2 Position                */
#define GP_MODE_MODE2_Msk                (0x3ul << GP_MODE_MODE2_Pos)                      /*!< GP_T::MODE: MODE2 Mask                    */

#define GP_MODE_MODE3_Pos                (6)                                               /*!< GP_T::MODE: MODE3 Position                */
#define GP_MODE_MODE3_Msk                (0x3ul << GP_MODE_MODE3_Pos)                      /*!< GP_T::MODE: MODE3 Mask                    */

#define GP_MODE_MODE4_Pos                (8)                                               /*!< GP_T::MODE: MODE4 Position                */
#define GP_MODE_MODE4_Msk                (0x3ul << GP_MODE_MODE4_Pos)                      /*!< GP_T::MODE: MODE4 Mask                    */

#define GP_MODE_MODE5_Pos                (10)                                              /*!< GP_T::MODE: MODE5 Position                */
#define GP_MODE_MODE5_Msk                (0x3ul << GP_MODE_MODE5_Pos)                      /*!< GP_T::MODE: MODE5 Mask                    */

#define GP_DINOFF_DINOFF0_Pos            (16)                                              /*!< GP_T::DINOFF: DINOFF0 Position            */
#define GP_DINOFF_DINOFF0_Msk            (0x1ul << GP_DINOFF_DINOFF0_Pos)                  /*!< GP_T::DINOFF: DINOFF0 Mask                */

#define GP_DINOFF_DINOFF1_Pos            (17)                                              /*!< GP_T::DINOFF: DINOFF1 Position            */
#define GP_DINOFF_DINOFF1_Msk            (0x1ul << GP_DINOFF_DINOFF1_Pos)                  /*!< GP_T::DINOFF: DINOFF1 Mask                */

#define GP_DINOFF_DINOFF2_Pos            (18)                                              /*!< GP_T::DINOFF: DINOFF2 Position            */
#define GP_DINOFF_DINOFF2_Msk            (0x1ul << GP_DINOFF_DINOFF2_Pos)                  /*!< GP_T::DINOFF: DINOFF2 Mask                */

#define GP_DINOFF_DINOFF3_Pos            (19)                                              /*!< GP_T::DINOFF: DINOFF3 Position            */
#define GP_DINOFF_DINOFF3_Msk            (0x1ul << GP_DINOFF_DINOFF3_Pos)                  /*!< GP_T::DINOFF: DINOFF3 Mask                */

#define GP_DINOFF_DINOFF4_Pos            (20)                                              /*!< GP_T::DINOFF: DINOFF4 Position            */
#define GP_DINOFF_DINOFF4_Msk            (0x1ul << GP_DINOFF_DINOFF4_Pos)                  /*!< GP_T::DINOFF: DINOFF4 Mask                */

#define GP_DINOFF_DINOFF5_Pos            (21)                                              /*!< GP_T::DINOFF: DINOFF5 Position            */
#define GP_DINOFF_DINOFF5_Msk            (0x1ul << GP_DINOFF_DINOFF5_Pos)                  /*!< GP_T::DINOFF: DINOFF5 Mask                */

#define GP_DINOFF_DINOFF6_Pos            (22)                                              /*!< GP_T::DINOFF: DINOFF6 Position            */
#define GP_DINOFF_DINOFF6_Msk            (0x1ul << GP_DINOFF_DINOFF6_Pos)                  /*!< GP_T::DINOFF: DINOFF6 Mask                */

#define GP_DINOFF_DINOFF7_Pos            (23)                                              /*!< GP_T::DINOFF: DINOFF7 Position            */
#define GP_DINOFF_DINOFF7_Msk            (0x1ul << GP_DINOFF_DINOFF7_Pos)                  /*!< GP_T::DINOFF: DINOFF7 Mask                */

#define GP_DOUT_DOUT0_Pos                (0)                                               /*!< GP_T::DOUT: DOUT0 Position                */
#define GP_DOUT_DOUT0_Msk                (0x1ul << GP_DOUT_DOUT0_Pos)                      /*!< GP_T::DOUT: DOUT0 Mask                    */

#define GP_DOUT_DOUT1_Pos                (1)                                               /*!< GP_T::DOUT: DOUT1 Position                */
#define GP_DOUT_DOUT1_Msk                (0x1ul << GP_DOUT_DOUT1_Pos)                      /*!< GP_T::DOUT: DOUT1 Mask                    */

#define GP_DOUT_DOUT2_Pos                (2)                                               /*!< GP_T::DOUT: DOUT2 Position                */
#define GP_DOUT_DOUT2_Msk                (0x1ul << GP_DOUT_DOUT2_Pos)                      /*!< GP_T::DOUT: DOUT2 Mask                    */

#define GP_DOUT_DOUT3_Pos                (3)                                               /*!< GP_T::DOUT: DOUT3 Position                */
#define GP_DOUT_DOUT3_Msk                (0x1ul << GP_DOUT_DOUT3_Pos)                      /*!< GP_T::DOUT: DOUT3 Mask                    */

#define GP_DOUT_DOUT4_Pos                (4)                                               /*!< GP_T::DOUT: DOUT4 Position                */
#define GP_DOUT_DOUT4_Msk                (0x1ul << GP_DOUT_DOUT4_Pos)                      /*!< GP_T::DOUT: DOUT4 Mask                    */

#define GP_DOUT_DOUT5_Pos                (5)                                               /*!< GP_T::DOUT: DOUT5 Position                */
#define GP_DOUT_DOUT5_Msk                (0x1ul << GP_DOUT_DOUT5_Pos)                      /*!< GP_T::DOUT: DOUT5 Mask                    */

#define GP_DOUT_DOUT6_Pos                (6)                                               /*!< GP_T::DOUT: DOUT6 Position                */
#define GP_DOUT_DOUT6_Msk                (0x1ul << GP_DOUT_DOUT6_Pos)                      /*!< GP_T::DOUT: DOUT6 Mask                    */

#define GP_DOUT_DOUT7_Pos                (7)                                               /*!< GP_T::DOUT: DOUT7 Position                */
#define GP_DOUT_DOUT7_Msk                (0x1ul << GP_DOUT_DOUT7_Pos)                      /*!< GP_T::DOUT: DOUT7 Mask                    */

#define GP_DATMSK_DATMSK0_Pos            (0)                                               /*!< GP_T::DATMSK: DATMSK0 Position            */
#define GP_DATMSK_DATMSK0_Msk            (0x1ul << GP_DATMSK_DATMSK0_Pos)                  /*!< GP_T::DATMSK: DATMSK0 Mask                */

#define GP_DATMSK_DATMSK1_Pos            (1)                                               /*!< GP_T::DATMSK: DATMSK1 Position            */
#define GP_DATMSK_DATMSK1_Msk            (0x1ul << GP_DATMSK_DATMSK1_Pos)                  /*!< GP_T::DATMSK: DATMSK1 Mask                */

#define GP_DATMSK_DATMSK2_Pos            (2)                                               /*!< GP_T::DATMSK: DATMSK2 Position            */
#define GP_DATMSK_DATMSK2_Msk            (0x1ul << GP_DATMSK_DATMSK2_Pos)                  /*!< GP_T::DATMSK: DATMSK2 Mask                */

#define GP_DATMSK_DATMSK3_Pos            (3)                                               /*!< GP_T::DATMSK: DATMSK3 Position            */
#define GP_DATMSK_DATMSK3_Msk            (0x1ul << GP_DATMSK_DATMSK3_Pos)                  /*!< GP_T::DATMSK: DATMSK3 Mask                */

#define GP_DATMSK_DATMSK4_Pos            (4)                                               /*!< GP_T::DATMSK: DATMSK4 Position            */
#define GP_DATMSK_DATMSK4_Msk            (0x1ul << GP_DATMSK_DATMSK4_Pos)                  /*!< GP_T::DATMSK: DATMSK4 Mask                */

#define GP_DATMSK_DATMSK5_Pos            (5)                                               /*!< GP_T::DATMSK: DATMSK5 Position            */
#define GP_DATMSK_DATMSK5_Msk            (0x1ul << GP_DATMSK_DATMSK5_Pos)                  /*!< GP_T::DATMSK: DATMSK5 Mask                */

#define GP_DATMSK_DATMSK6_Pos            (6)                                               /*!< GP_T::DATMSK: DATMSK6 Position            */
#define GP_DATMSK_DATMSK6_Msk            (0x1ul << GP_DATMSK_DATMSK6_Pos)                  /*!< GP_T::DATMSK: DATMSK6 Mask                */

#define GP_DATMSK_DATMSK7_Pos            (7)                                               /*!< GP_T::DATMSK: DATMSK7 Position            */
#define GP_DATMSK_DATMSK7_Msk            (0x1ul << GP_DATMSK_DATMSK7_Pos)                  /*!< GP_T::DATMSK: DATMSK7 Mask                */

#define GP_PIN_PIN0_Pos                  (0)                                               /*!< GP_T::PIN: PIN0 Position                  */
#define GP_PIN_PIN0_Msk                  (0x1ul << GP_PIN_PIN0_Pos)                        /*!< GP_T::PIN: PIN0 Mask                      */

#define GP_PIN_PIN1_Pos                  (1)                                               /*!< GP_T::PIN: PIN1 Position                  */
#define GP_PIN_PIN1_Msk                  (0x1ul << GP_PIN_PIN1_Pos)                        /*!< GP_T::PIN: PIN1 Mask                      */

#define GP_PIN_PIN2_Pos                  (2)                                               /*!< GP_T::PIN: PIN2 Position                  */
#define GP_PIN_PIN2_Msk                  (0x1ul << GP_PIN_PIN2_Pos)                        /*!< GP_T::PIN: PIN2 Mask                      */

#define GP_PIN_PIN3_Pos                  (3)                                               /*!< GP_T::PIN: PIN3 Position                  */
#define GP_PIN_PIN3_Msk                  (0x1ul << GP_PIN_PIN3_Pos)                        /*!< GP_T::PIN: PIN3 Mask                      */

#define GP_PIN_PIN4_Pos                  (4)                                               /*!< GP_T::PIN: PIN4 Position                  */
#define GP_PIN_PIN4_Msk                  (0x1ul << GP_PIN_PIN4_Pos)                        /*!< GP_T::PIN: PIN4 Mask                      */

#define GP_PIN_PIN5_Pos                  (5)                                               /*!< GP_T::PIN: PIN5 Position                  */
#define GP_PIN_PIN5_Msk                  (0x1ul << GP_PIN_PIN5_Pos)                        /*!< GP_T::PIN: PIN5 Mask                      */

#define GP_PIN_PIN6_Pos                  (6)                                               /*!< GP_T::PIN: PIN6 Position                  */
#define GP_PIN_PIN6_Msk                  (0x1ul << GP_PIN_PIN6_Pos)                        /*!< GP_T::PIN: PIN6 Mask                      */

#define GP_PIN_PIN7_Pos                  (7)                                               /*!< GP_T::PIN: PIN7 Position                  */
#define GP_PIN_PIN7_Msk                  (0x1ul << GP_PIN_PIN7_Pos)                        /*!< GP_T::PIN: PIN7 Mask                      */

#define GP_DBEN_DBEN0_Pos                (0)                                               /*!< GP_T::DBEN: DBEN0 Position                */
#define GP_DBEN_DBEN0_Msk                (0x1ul << GP_DBEN_DBEN0_Pos)                      /*!< GP_T::DBEN: DBEN0 Mask                    */

#define GP_DBEN_DBEN1_Pos                (1)                                               /*!< GP_T::DBEN: DBEN1 Position                */
#define GP_DBEN_DBEN1_Msk                (0x1ul << GP_DBEN_DBEN1_Pos)                      /*!< GP_T::DBEN: DBEN1 Mask                    */

#define GP_DBEN_DBEN2_Pos                (2)                                               /*!< GP_T::DBEN: DBEN2 Position                */
#define GP_DBEN_DBEN2_Msk                (0x1ul << GP_DBEN_DBEN2_Pos)                      /*!< GP_T::DBEN: DBEN2 Mask                    */

#define GP_DBEN_DBEN3_Pos                (3)                                               /*!< GP_T::DBEN: DBEN3 Position                */
#define GP_DBEN_DBEN3_Msk                (0x1ul << GP_DBEN_DBEN3_Pos)                      /*!< GP_T::DBEN: DBEN3 Mask                    */

#define GP_DBEN_DBEN4_Pos                (4)                                               /*!< GP_T::DBEN: DBEN4 Position                */
#define GP_DBEN_DBEN4_Msk                (0x1ul << GP_DBEN_DBEN4_Pos)                      /*!< GP_T::DBEN: DBEN4 Mask                    */

#define GP_DBEN_DBEN5_Pos                (5)                                               /*!< GP_T::DBEN: DBEN5 Position                */
#define GP_DBEN_DBEN5_Msk                (0x1ul << GP_DBEN_DBEN5_Pos)                      /*!< GP_T::DBEN: DBEN5 Mask                    */

#define GP_DBEN_DBEN6_Pos                (6)                                               /*!< GP_T::DBEN: DBEN6 Position                */
#define GP_DBEN_DBEN6_Msk                (0x1ul << GP_DBEN_DBEN6_Pos)                      /*!< GP_T::DBEN: DBEN6 Mask                    */

#define GP_DBEN_DBEN7_Pos                (7)                                               /*!< GP_T::DBEN: DBEN7 Position                */
#define GP_DBEN_DBEN7_Msk                (0x1ul << GP_DBEN_DBEN7_Pos)                      /*!< GP_T::DBEN: DBEN7 Mask                    */

#define GP_INTTYPE_TYPE0_Pos             (0)                                               /*!< GP_T::INTTYPE: TYPE0 Position             */
#define GP_INTTYPE_TYPE0_Msk             (0x1ul << GP_INTTYPE_TYPE0_Pos)                   /*!< GP_T::INTTYPE: TYPE0 Mask                 */

#define GP_INTTYPE_TYPE1_Pos             (1)                                               /*!< GP_T::INTTYPE: TYPE1 Position             */
#define GP_INTTYPE_TYPE1_Msk             (0x1ul << GP_INTTYPE_TYPE1_Pos)                   /*!< GP_T::INTTYPE: TYPE1 Mask                 */

#define GP_INTTYPE_TYPE2_Pos             (2)                                               /*!< GP_T::INTTYPE: TYPE2 Position             */
#define GP_INTTYPE_TYPE2_Msk             (0x1ul << GP_INTTYPE_TYPE2_Pos)                   /*!< GP_T::INTTYPE: TYPE2 Mask                 */

#define GP_INTTYPE_TYPE3_Pos             (3)                                               /*!< GP_T::INTTYPE: TYPE3 Position             */
#define GP_INTTYPE_TYPE3_Msk             (0x1ul << GP_INTTYPE_TYPE3_Pos)                   /*!< GP_T::INTTYPE: TYPE3 Mask                 */

#define GP_INTTYPE_TYPE4_Pos             (4)                                               /*!< GP_T::INTTYPE: TYPE4 Position             */
#define GP_INTTYPE_TYPE4_Msk             (0x1ul << GP_INTTYPE_TYPE4_Pos)                   /*!< GP_T::INTTYPE: TYPE4 Mask                 */

#define GP_INTTYPE_TYPE5_Pos             (5)                                               /*!< GP_T::INTTYPE: TYPE5 Position             */
#define GP_INTTYPE_TYPE5_Msk             (0x1ul << GP_INTTYPE_TYPE5_Pos)                   /*!< GP_T::INTTYPE: TYPE5 Mask                 */

#define GP_INTTYPE_TYPE6_Pos             (6)                                               /*!< GP_T::INTTYPE: TYPE6 Position             */
#define GP_INTTYPE_TYPE6_Msk             (0x1ul << GP_INTTYPE_TYPE6_Pos)                   /*!< GP_T::INTTYPE: TYPE6 Mask                 */

#define GP_INTTYPE_TYPE7_Pos             (7)                                               /*!< GP_T::INTTYPE: TYPE7 Position             */
#define GP_INTTYPE_TYPE7_Msk             (0x1ul << GP_INTTYPE_TYPE7_Pos)                   /*!< GP_T::INTTYPE: TYPE7 Mask                 */

#define GP_INTEN_FLIEN0_Pos              (0)                                               /*!< GP_T::INTEN: FLIEN0 Position              */
#define GP_INTEN_FLIEN0_Msk              (0x1ul << GP_INTEN_FLIEN0_Pos)                    /*!< GP_T::INTEN: FLIEN0 Mask                  */

#define GP_INTEN_FLIEN1_Pos              (1)                                               /*!< GP_T::INTEN: FLIEN1 Position              */
#define GP_INTEN_FLIEN1_Msk              (0x1ul << GP_INTEN_FLIEN1_Pos)                    /*!< GP_T::INTEN: FLIEN1 Mask                  */

#define GP_INTEN_FLIEN2_Pos              (2)                                               /*!< GP_T::INTEN: FLIEN2 Position              */
#define GP_INTEN_FLIEN2_Msk              (0x1ul << GP_INTEN_FLIEN2_Pos)                    /*!< GP_T::INTEN: FLIEN2 Mask                  */

#define GP_INTEN_FLIEN3_Pos              (3)                                               /*!< GP_T::INTEN: FLIEN3 Position              */
#define GP_INTEN_FLIEN3_Msk              (0x1ul << GP_INTEN_FLIEN3_Pos)                    /*!< GP_T::INTEN: FLIEN3 Mask                  */

#define GP_INTEN_FLIEN4_Pos              (4)                                               /*!< GP_T::INTEN: FLIEN4 Position              */
#define GP_INTEN_FLIEN4_Msk              (0x1ul << GP_INTEN_FLIEN4_Pos)                    /*!< GP_T::INTEN: FLIEN4 Mask                  */

#define GP_INTEN_FLIEN5_Pos              (5)                                               /*!< GP_T::INTEN: FLIEN5 Position              */
#define GP_INTEN_FLIEN5_Msk              (0x1ul << GP_INTEN_FLIEN5_Pos)                    /*!< GP_T::INTEN: FLIEN5 Mask                  */

#define GP_INTEN_FLIEN6_Pos              (6)                                               /*!< GP_T::INTEN: FLIEN6 Position              */
#define GP_INTEN_FLIEN6_Msk              (0x1ul << GP_INTEN_FLIEN6_Pos)                    /*!< GP_T::INTEN: FLIEN6 Mask                  */

#define GP_INTEN_FLIEN7_Pos              (7)                                               /*!< GP_T::INTEN: FLIEN7 Position              */
#define GP_INTEN_FLIEN7_Msk              (0x1ul << GP_INTEN_FLIEN7_Pos)                    /*!< GP_T::INTEN: FLIEN7 Mask                  */

#define GP_INTEN_RHIEN0_Pos              (16)                                              /*!< GP_T::INTEN: RHIEN0 Position              */
#define GP_INTEN_RHIEN0_Msk              (0x1ul << GP_INTEN_RHIEN0_Pos)                    /*!< GP_T::INTEN: RHIEN0 Mask                  */

#define GP_INTEN_RHIEN1_Pos              (17)                                              /*!< GP_T::INTEN: RHIEN1 Position              */
#define GP_INTEN_RHIEN1_Msk              (0x1ul << GP_INTEN_RHIEN1_Pos)                    /*!< GP_T::INTEN: RHIEN1 Mask                  */

#define GP_INTEN_RHIEN2_Pos              (18)                                              /*!< GP_T::INTEN: RHIEN2 Position              */
#define GP_INTEN_RHIEN2_Msk              (0x1ul << GP_INTEN_RHIEN2_Pos)                    /*!< GP_T::INTEN: RHIEN2 Mask                  */

#define GP_INTEN_RHIEN3_Pos              (19)                                              /*!< GP_T::INTEN: RHIEN3 Position              */
#define GP_INTEN_RHIEN3_Msk              (0x1ul << GP_INTEN_RHIEN3_Pos)                    /*!< GP_T::INTEN: RHIEN3 Mask                  */

#define GP_INTEN_RHIEN4_Pos              (20)                                              /*!< GP_T::INTEN: RHIEN4 Position              */
#define GP_INTEN_RHIEN4_Msk              (0x1ul << GP_INTEN_RHIEN4_Pos)                    /*!< GP_T::INTEN: RHIEN4 Mask                  */

#define GP_INTEN_RHIEN5_Pos              (21)                                              /*!< GP_T::INTEN: RHIEN5 Position              */
#define GP_INTEN_RHIEN5_Msk              (0x1ul << GP_INTEN_RHIEN5_Pos)                    /*!< GP_T::INTEN: RHIEN5 Mask                  */

#define GP_INTEN_RHIEN6_Pos              (22)                                              /*!< GP_T::INTEN: RHIEN6 Position              */
#define GP_INTEN_RHIEN6_Msk              (0x1ul << GP_INTEN_RHIEN6_Pos)                    /*!< GP_T::INTEN: RHIEN6 Mask                  */

#define GP_INTEN_RHIEN7_Pos              (23)                                              /*!< GP_T::INTEN: RHIEN7 Position              */
#define GP_INTEN_RHIEN7_Msk              (0x1ul << GP_INTEN_RHIEN7_Pos)                    /*!< GP_T::INTEN: RHIEN7 Mask                  */

#define GP_INTSRC_INTSRC0_Pos            (0)                                               /*!< GP_T::INTSRC: INTSRC0 Position            */
#define GP_INTSRC_INTSRC0_Msk            (0x1ul << GP_INTSRC_INTSRC0_Pos)                  /*!< GP_T::INTSRC: INTSRC0 Mask                */

#define GP_INTSRC_INTSRC1_Pos            (1)                                               /*!< GP_T::INTSRC: INTSRC1 Position            */
#define GP_INTSRC_INTSRC1_Msk            (0x1ul << GP_INTSRC_INTSRC1_Pos)                  /*!< GP_T::INTSRC: INTSRC1 Mask                */

#define GP_INTSRC_INTSRC2_Pos            (2)                                               /*!< GP_T::INTSRC: INTSRC2 Position            */
#define GP_INTSRC_INTSRC2_Msk            (0x1ul << GP_INTSRC_INTSRC2_Pos)                  /*!< GP_T::INTSRC: INTSRC2 Mask                */

#define GP_INTSRC_INTSRC3_Pos            (3)                                               /*!< GP_T::INTSRC: INTSRC3 Position            */
#define GP_INTSRC_INTSRC3_Msk            (0x1ul << GP_INTSRC_INTSRC3_Pos)                  /*!< GP_T::INTSRC: INTSRC3 Mask                */

#define GP_INTSRC_INTSRC4_Pos            (4)                                               /*!< GP_T::INTSRC: INTSRC4 Position            */
#define GP_INTSRC_INTSRC4_Msk            (0x1ul << GP_INTSRC_INTSRC4_Pos)                  /*!< GP_T::INTSRC: INTSRC4 Mask                */

#define GP_INTSRC_INTSRC5_Pos            (5)                                               /*!< GP_T::INTSRC: INTSRC5 Position            */
#define GP_INTSRC_INTSRC5_Msk            (0x1ul << GP_INTSRC_INTSRC5_Pos)                  /*!< GP_T::INTSRC: INTSRC5 Mask                */

#define GP_INTSRC_INTSRC6_Pos            (6)                                               /*!< GP_T::INTSRC: INTSRC6 Position            */
#define GP_INTSRC_INTSRC6_Msk            (0x1ul << GP_INTSRC_INTSRC6_Pos)                  /*!< GP_T::INTSRC: INTSRC6 Mask                */

#define GP_INTSRC_INTSRC7_Pos            (7)                                               /*!< GP_T::INTSRC: INTSRC7 Position            */
#define GP_INTSRC_INTSRC7_Msk            (0x1ul << GP_INTSRC_INTSRC7_Pos)                  /*!< GP_T::INTSRC: INTSRC7 Mask                */

#define GP_INTSRC_INTSRC8_Pos            (8)                                               /*!< GP_T::INTSRC: INTSRC8 Position            */
#define GP_INTSRC_INTSRC8_Msk            (0x1ul << GP_INTSRC_INTSRC8_Pos)                  /*!< GP_T::INTSRC: INTSRC8 Mask                */

#define GP_INTSRC_INTSRC9_Pos            (9)                                               /*!< GP_T::INTSRC: INTSRC9 Position            */
#define GP_INTSRC_INTSRC9_Msk            (0x1ul << GP_INTSRC_INTSRC9_Pos)                  /*!< GP_T::INTSRC: INTSRC9 Mask                */

#define GP_INTSRC_INTSRC10_Pos           (10)                                              /*!< GP_T::INTSRC: INTSRC10 Position           */
#define GP_INTSRC_INTSRC10_Msk           (0x1ul << GP_INTSRC_INTSRC10_Pos)                 /*!< GP_T::INTSRC: INTSRC10 Mask               */

#define GP_INTSRC_INTSRC11_Pos           (11)                                              /*!< GP_T::INTSRC: INTSRC11 Position           */
#define GP_INTSRC_INTSRC11_Msk           (0x1ul << GP_INTSRC_INTSRC11_Pos)                 /*!< GP_T::INTSRC: INTSRC11 Mask               */

#define GP_INTSRC_INTSRC12_Pos           (12)                                              /*!< GP_T::INTSRC: INTSRC12 Position           */
#define GP_INTSRC_INTSRC12_Msk           (0x1ul << GP_INTSRC_INTSRC12_Pos)                 /*!< GP_T::INTSRC: INTSRC12 Mask               */

#define GP_INTSRC_INTSRC13_Pos           (13)                                              /*!< GP_T::INTSRC: INTSRC13 Position           */
#define GP_INTSRC_INTSRC13_Msk           (0x1ul << GP_INTSRC_INTSRC13_Pos)                 /*!< GP_T::INTSRC: INTSRC13 Mask               */

#define GP_INTSRC_INTSRC14_Pos           (14)                                              /*!< GP_T::INTSRC: INTSRC14 Position           */
#define GP_INTSRC_INTSRC14_Msk           (0x1ul << GP_INTSRC_INTSRC14_Pos)                 /*!< GP_T::INTSRC: INTSRC14 Mask               */

#define GP_INTSRC_INTSRC15_Pos           (15)                                              /*!< GP_T::INTSRC: INTSRC15 Position           */
#define GP_INTSRC_INTSRC15_Msk           (0x1ul << GP_INTSRC_INTSRC15_Pos)                 /*!< GP_T::INTSRC: INTSRC15 Mask               */

#define GP_MODE_MODE0_Pos                (0)                                               /*!< GP_T::MODE: MODE0 Position                */
#define GP_MODE_MODE0_Msk                (0x3ul << GP_MODE_MODE0_Pos)                      /*!< GP_T::MODE: MODE0 Mask                    */

#define GP_MODE_MODE1_Pos                (2)                                               /*!< GP_T::MODE: MODE1 Position                */
#define GP_MODE_MODE1_Msk                (0x3ul << GP_MODE_MODE1_Pos)                      /*!< GP_T::MODE: MODE1 Mask                    */

#define GP_MODE_MODE2_Pos                (4)                                               /*!< GP_T::MODE: MODE2 Position                */
#define GP_MODE_MODE2_Msk                (0x3ul << GP_MODE_MODE2_Pos)                      /*!< GP_T::MODE: MODE2 Mask                    */

#define GP_MODE_MODE3_Pos                (6)                                               /*!< GP_T::MODE: MODE3 Position                */
#define GP_MODE_MODE3_Msk                (0x3ul << GP_MODE_MODE3_Pos)                      /*!< GP_T::MODE: MODE3 Mask                    */

#define GP_MODE_MODE4_Pos                (8)                                               /*!< GP_T::MODE: MODE4 Position                */
#define GP_MODE_MODE4_Msk                (0x3ul << GP_MODE_MODE4_Pos)                      /*!< GP_T::MODE: MODE4 Mask                    */

#define GP_MODE_MODE5_Pos                (10)                                              /*!< GP_T::MODE: MODE5 Position                */
#define GP_MODE_MODE5_Msk                (0x3ul << GP_MODE_MODE5_Pos)                      /*!< GP_T::MODE: MODE5 Mask                    */

#define GP_DINOFF_DINOFF0_Pos            (16)                                              /*!< GP_T::DINOFF: DINOFF0 Position            */
#define GP_DINOFF_DINOFF0_Msk            (0x1ul << GP_DINOFF_DINOFF0_Pos)                  /*!< GP_T::DINOFF: DINOFF0 Mask                */

#define GP_DINOFF_DINOFF1_Pos            (17)                                              /*!< GP_T::DINOFF: DINOFF1 Position            */
#define GP_DINOFF_DINOFF1_Msk            (0x1ul << GP_DINOFF_DINOFF1_Pos)                  /*!< GP_T::DINOFF: DINOFF1 Mask                */

#define GP_DINOFF_DINOFF2_Pos            (18)                                              /*!< GP_T::DINOFF: DINOFF2 Position            */
#define GP_DINOFF_DINOFF2_Msk            (0x1ul << GP_DINOFF_DINOFF2_Pos)                  /*!< GP_T::DINOFF: DINOFF2 Mask                */

#define GP_DINOFF_DINOFF3_Pos            (19)                                              /*!< GP_T::DINOFF: DINOFF3 Position            */
#define GP_DINOFF_DINOFF3_Msk            (0x1ul << GP_DINOFF_DINOFF3_Pos)                  /*!< GP_T::DINOFF: DINOFF3 Mask                */

#define GP_DINOFF_DINOFF4_Pos            (20)                                              /*!< GP_T::DINOFF: DINOFF4 Position            */
#define GP_DINOFF_DINOFF4_Msk            (0x1ul << GP_DINOFF_DINOFF4_Pos)                  /*!< GP_T::DINOFF: DINOFF4 Mask                */

#define GP_DINOFF_DINOFF5_Pos            (21)                                              /*!< GP_T::DINOFF: DINOFF5 Position            */
#define GP_DINOFF_DINOFF5_Msk            (0x1ul << GP_DINOFF_DINOFF5_Pos)                  /*!< GP_T::DINOFF: DINOFF5 Mask                */

#define GP_DINOFF_DINOFF6_Pos            (22)                                              /*!< GP_T::DINOFF: DINOFF6 Position            */
#define GP_DINOFF_DINOFF6_Msk            (0x1ul << GP_DINOFF_DINOFF6_Pos)                  /*!< GP_T::DINOFF: DINOFF6 Mask                */

#define GP_DINOFF_DINOFF7_Pos            (23)                                              /*!< GP_T::DINOFF: DINOFF7 Position            */
#define GP_DINOFF_DINOFF7_Msk            (0x1ul << GP_DINOFF_DINOFF7_Pos)                  /*!< GP_T::DINOFF: DINOFF7 Mask                */

#define GP_DOUT_DOUT0_Pos                (0)                                               /*!< GP_T::DOUT: DOUT0 Position                */
#define GP_DOUT_DOUT0_Msk                (0x1ul << GP_DOUT_DOUT0_Pos)                      /*!< GP_T::DOUT: DOUT0 Mask                    */

#define GP_DOUT_DOUT1_Pos                (1)                                               /*!< GP_T::DOUT: DOUT1 Position                */
#define GP_DOUT_DOUT1_Msk                (0x1ul << GP_DOUT_DOUT1_Pos)                      /*!< GP_T::DOUT: DOUT1 Mask                    */

#define GP_DOUT_DOUT2_Pos                (2)                                               /*!< GP_T::DOUT: DOUT2 Position                */
#define GP_DOUT_DOUT2_Msk                (0x1ul << GP_DOUT_DOUT2_Pos)                      /*!< GP_T::DOUT: DOUT2 Mask                    */

#define GP_DOUT_DOUT3_Pos                (3)                                               /*!< GP_T::DOUT: DOUT3 Position                */
#define GP_DOUT_DOUT3_Msk                (0x1ul << GP_DOUT_DOUT3_Pos)                      /*!< GP_T::DOUT: DOUT3 Mask                    */

#define GP_DOUT_DOUT4_Pos                (4)                                               /*!< GP_T::DOUT: DOUT4 Position                */
#define GP_DOUT_DOUT4_Msk                (0x1ul << GP_DOUT_DOUT4_Pos)                      /*!< GP_T::DOUT: DOUT4 Mask                    */

#define GP_DOUT_DOUT5_Pos                (5)                                               /*!< GP_T::DOUT: DOUT5 Position                */
#define GP_DOUT_DOUT5_Msk                (0x1ul << GP_DOUT_DOUT5_Pos)                      /*!< GP_T::DOUT: DOUT5 Mask                    */

#define GP_DOUT_DOUT6_Pos                (6)                                               /*!< GP_T::DOUT: DOUT6 Position                */
#define GP_DOUT_DOUT6_Msk                (0x1ul << GP_DOUT_DOUT6_Pos)                      /*!< GP_T::DOUT: DOUT6 Mask                    */

#define GP_DOUT_DOUT7_Pos                (7)                                               /*!< GP_T::DOUT: DOUT7 Position                */
#define GP_DOUT_DOUT7_Msk                (0x1ul << GP_DOUT_DOUT7_Pos)                      /*!< GP_T::DOUT: DOUT7 Mask                    */

#define GP_DATMSK_DATMSK0_Pos            (0)                                               /*!< GP_T::DATMSK: DATMSK0 Position            */
#define GP_DATMSK_DATMSK0_Msk            (0x1ul << GP_DATMSK_DATMSK0_Pos)                  /*!< GP_T::DATMSK: DATMSK0 Mask                */

#define GP_DATMSK_DATMSK1_Pos            (1)                                               /*!< GP_T::DATMSK: DATMSK1 Position            */
#define GP_DATMSK_DATMSK1_Msk            (0x1ul << GP_DATMSK_DATMSK1_Pos)                  /*!< GP_T::DATMSK: DATMSK1 Mask                */

#define GP_DATMSK_DATMSK2_Pos            (2)                                               /*!< GP_T::DATMSK: DATMSK2 Position            */
#define GP_DATMSK_DATMSK2_Msk            (0x1ul << GP_DATMSK_DATMSK2_Pos)                  /*!< GP_T::DATMSK: DATMSK2 Mask                */

#define GP_DATMSK_DATMSK3_Pos            (3)                                               /*!< GP_T::DATMSK: DATMSK3 Position            */
#define GP_DATMSK_DATMSK3_Msk            (0x1ul << GP_DATMSK_DATMSK3_Pos)                  /*!< GP_T::DATMSK: DATMSK3 Mask                */

#define GP_DATMSK_DATMSK4_Pos            (4)                                               /*!< GP_T::DATMSK: DATMSK4 Position            */
#define GP_DATMSK_DATMSK4_Msk            (0x1ul << GP_DATMSK_DATMSK4_Pos)                  /*!< GP_T::DATMSK: DATMSK4 Mask                */

#define GP_DATMSK_DATMSK5_Pos            (5)                                               /*!< GP_T::DATMSK: DATMSK5 Position            */
#define GP_DATMSK_DATMSK5_Msk            (0x1ul << GP_DATMSK_DATMSK5_Pos)                  /*!< GP_T::DATMSK: DATMSK5 Mask                */

#define GP_DATMSK_DATMSK6_Pos            (6)                                               /*!< GP_T::DATMSK: DATMSK6 Position            */
#define GP_DATMSK_DATMSK6_Msk            (0x1ul << GP_DATMSK_DATMSK6_Pos)                  /*!< GP_T::DATMSK: DATMSK6 Mask                */

#define GP_DATMSK_DATMSK7_Pos            (7)                                               /*!< GP_T::DATMSK: DATMSK7 Position            */
#define GP_DATMSK_DATMSK7_Msk            (0x1ul << GP_DATMSK_DATMSK7_Pos)                  /*!< GP_T::DATMSK: DATMSK7 Mask                */

#define GP_PIN_PIN0_Pos                  (0)                                               /*!< GP_T::PIN: PIN0 Position                  */
#define GP_PIN_PIN0_Msk                  (0x1ul << GP_PIN_PIN0_Pos)                        /*!< GP_T::PIN: PIN0 Mask                      */

#define GP_PIN_PIN1_Pos                  (1)                                               /*!< GP_T::PIN: PIN1 Position                  */
#define GP_PIN_PIN1_Msk                  (0x1ul << GP_PIN_PIN1_Pos)                        /*!< GP_T::PIN: PIN1 Mask                      */

#define GP_PIN_PIN2_Pos                  (2)                                               /*!< GP_T::PIN: PIN2 Position                  */
#define GP_PIN_PIN2_Msk                  (0x1ul << GP_PIN_PIN2_Pos)                        /*!< GP_T::PIN: PIN2 Mask                      */

#define GP_PIN_PIN3_Pos                  (3)                                               /*!< GP_T::PIN: PIN3 Position                  */
#define GP_PIN_PIN3_Msk                  (0x1ul << GP_PIN_PIN3_Pos)                        /*!< GP_T::PIN: PIN3 Mask                      */

#define GP_PIN_PIN4_Pos                  (4)                                               /*!< GP_T::PIN: PIN4 Position                  */
#define GP_PIN_PIN4_Msk                  (0x1ul << GP_PIN_PIN4_Pos)                        /*!< GP_T::PIN: PIN4 Mask                      */

#define GP_PIN_PIN5_Pos                  (5)                                               /*!< GP_T::PIN: PIN5 Position                  */
#define GP_PIN_PIN5_Msk                  (0x1ul << GP_PIN_PIN5_Pos)                        /*!< GP_T::PIN: PIN5 Mask                      */

#define GP_PIN_PIN6_Pos                  (6)                                               /*!< GP_T::PIN: PIN6 Position                  */
#define GP_PIN_PIN6_Msk                  (0x1ul << GP_PIN_PIN6_Pos)                        /*!< GP_T::PIN: PIN6 Mask                      */

#define GP_PIN_PIN7_Pos                  (7)                                               /*!< GP_T::PIN: PIN7 Position                  */
#define GP_PIN_PIN7_Msk                  (0x1ul << GP_PIN_PIN7_Pos)                        /*!< GP_T::PIN: PIN7 Mask                      */

#define GP_DBEN_DBEN0_Pos                (0)                                               /*!< GP_T::DBEN: DBEN0 Position                */
#define GP_DBEN_DBEN0_Msk                (0x1ul << GP_DBEN_DBEN0_Pos)                      /*!< GP_T::DBEN: DBEN0 Mask                    */

#define GP_DBEN_DBEN1_Pos                (1)                                               /*!< GP_T::DBEN: DBEN1 Position                */
#define GP_DBEN_DBEN1_Msk                (0x1ul << GP_DBEN_DBEN1_Pos)                      /*!< GP_T::DBEN: DBEN1 Mask                    */

#define GP_DBEN_DBEN2_Pos                (2)                                               /*!< GP_T::DBEN: DBEN2 Position                */
#define GP_DBEN_DBEN2_Msk                (0x1ul << GP_DBEN_DBEN2_Pos)                      /*!< GP_T::DBEN: DBEN2 Mask                    */

#define GP_DBEN_DBEN3_Pos                (3)                                               /*!< GP_T::DBEN: DBEN3 Position                */
#define GP_DBEN_DBEN3_Msk                (0x1ul << GP_DBEN_DBEN3_Pos)                      /*!< GP_T::DBEN: DBEN3 Mask                    */

#define GP_DBEN_DBEN4_Pos                (4)                                               /*!< GP_T::DBEN: DBEN4 Position                */
#define GP_DBEN_DBEN4_Msk                (0x1ul << GP_DBEN_DBEN4_Pos)                      /*!< GP_T::DBEN: DBEN4 Mask                    */

#define GP_DBEN_DBEN5_Pos                (5)                                               /*!< GP_T::DBEN: DBEN5 Position                */
#define GP_DBEN_DBEN5_Msk                (0x1ul << GP_DBEN_DBEN5_Pos)                      /*!< GP_T::DBEN: DBEN5 Mask                    */

#define GP_DBEN_DBEN6_Pos                (6)                                               /*!< GP_T::DBEN: DBEN6 Position                */
#define GP_DBEN_DBEN6_Msk                (0x1ul << GP_DBEN_DBEN6_Pos)                      /*!< GP_T::DBEN: DBEN6 Mask                    */

#define GP_DBEN_DBEN7_Pos                (7)                                               /*!< GP_T::DBEN: DBEN7 Position                */
#define GP_DBEN_DBEN7_Msk                (0x1ul << GP_DBEN_DBEN7_Pos)                      /*!< GP_T::DBEN: DBEN7 Mask                    */

#define GP_INTTYPE_TYPE0_Pos             (0)                                               /*!< GP_T::INTTYPE: TYPE0 Position             */
#define GP_INTTYPE_TYPE0_Msk             (0x1ul << GP_INTTYPE_TYPE0_Pos)                   /*!< GP_T::INTTYPE: TYPE0 Mask                 */

#define GP_INTTYPE_TYPE1_Pos             (1)                                               /*!< GP_T::INTTYPE: TYPE1 Position             */
#define GP_INTTYPE_TYPE1_Msk             (0x1ul << GP_INTTYPE_TYPE1_Pos)                   /*!< GP_T::INTTYPE: TYPE1 Mask                 */

#define GP_INTTYPE_TYPE2_Pos             (2)                                               /*!< GP_T::INTTYPE: TYPE2 Position             */
#define GP_INTTYPE_TYPE2_Msk             (0x1ul << GP_INTTYPE_TYPE2_Pos)                   /*!< GP_T::INTTYPE: TYPE2 Mask                 */

#define GP_INTTYPE_TYPE3_Pos             (3)                                               /*!< GP_T::INTTYPE: TYPE3 Position             */
#define GP_INTTYPE_TYPE3_Msk             (0x1ul << GP_INTTYPE_TYPE3_Pos)                   /*!< GP_T::INTTYPE: TYPE3 Mask                 */

#define GP_INTTYPE_TYPE4_Pos             (4)                                               /*!< GP_T::INTTYPE: TYPE4 Position             */
#define GP_INTTYPE_TYPE4_Msk             (0x1ul << GP_INTTYPE_TYPE4_Pos)                   /*!< GP_T::INTTYPE: TYPE4 Mask                 */

#define GP_INTTYPE_TYPE5_Pos             (5)                                               /*!< GP_T::INTTYPE: TYPE5 Position             */
#define GP_INTTYPE_TYPE5_Msk             (0x1ul << GP_INTTYPE_TYPE5_Pos)                   /*!< GP_T::INTTYPE: TYPE5 Mask                 */

#define GP_INTTYPE_TYPE6_Pos             (6)                                               /*!< GP_T::INTTYPE: TYPE6 Position             */
#define GP_INTTYPE_TYPE6_Msk             (0x1ul << GP_INTTYPE_TYPE6_Pos)                   /*!< GP_T::INTTYPE: TYPE6 Mask                 */

#define GP_INTTYPE_TYPE7_Pos             (7)                                               /*!< GP_T::INTTYPE: TYPE7 Position             */
#define GP_INTTYPE_TYPE7_Msk             (0x1ul << GP_INTTYPE_TYPE7_Pos)                   /*!< GP_T::INTTYPE: TYPE7 Mask                 */

#define GP_INTEN_FLIEN0_Pos              (0)                                               /*!< GP_T::INTEN: FLIEN0 Position              */
#define GP_INTEN_FLIEN0_Msk              (0x1ul << GP_INTEN_FLIEN0_Pos)                    /*!< GP_T::INTEN: FLIEN0 Mask                  */

#define GP_INTEN_FLIEN1_Pos              (1)                                               /*!< GP_T::INTEN: FLIEN1 Position              */
#define GP_INTEN_FLIEN1_Msk              (0x1ul << GP_INTEN_FLIEN1_Pos)                    /*!< GP_T::INTEN: FLIEN1 Mask                  */

#define GP_INTEN_FLIEN2_Pos              (2)                                               /*!< GP_T::INTEN: FLIEN2 Position              */
#define GP_INTEN_FLIEN2_Msk              (0x1ul << GP_INTEN_FLIEN2_Pos)                    /*!< GP_T::INTEN: FLIEN2 Mask                  */

#define GP_INTEN_FLIEN3_Pos              (3)                                               /*!< GP_T::INTEN: FLIEN3 Position              */
#define GP_INTEN_FLIEN3_Msk              (0x1ul << GP_INTEN_FLIEN3_Pos)                    /*!< GP_T::INTEN: FLIEN3 Mask                  */

#define GP_INTEN_FLIEN4_Pos              (4)                                               /*!< GP_T::INTEN: FLIEN4 Position              */
#define GP_INTEN_FLIEN4_Msk              (0x1ul << GP_INTEN_FLIEN4_Pos)                    /*!< GP_T::INTEN: FLIEN4 Mask                  */

#define GP_INTEN_FLIEN5_Pos              (5)                                               /*!< GP_T::INTEN: FLIEN5 Position              */
#define GP_INTEN_FLIEN5_Msk              (0x1ul << GP_INTEN_FLIEN5_Pos)                    /*!< GP_T::INTEN: FLIEN5 Mask                  */

#define GP_INTEN_FLIEN6_Pos              (6)                                               /*!< GP_T::INTEN: FLIEN6 Position              */
#define GP_INTEN_FLIEN6_Msk              (0x1ul << GP_INTEN_FLIEN6_Pos)                    /*!< GP_T::INTEN: FLIEN6 Mask                  */

#define GP_INTEN_FLIEN7_Pos              (7)                                               /*!< GP_T::INTEN: FLIEN7 Position              */
#define GP_INTEN_FLIEN7_Msk              (0x1ul << GP_INTEN_FLIEN7_Pos)                    /*!< GP_T::INTEN: FLIEN7 Mask                  */

#define GP_INTEN_RHIEN0_Pos              (16)                                              /*!< GP_T::INTEN: RHIEN0 Position              */
#define GP_INTEN_RHIEN0_Msk              (0x1ul << GP_INTEN_RHIEN0_Pos)                    /*!< GP_T::INTEN: RHIEN0 Mask                  */

#define GP_INTEN_RHIEN1_Pos              (17)                                              /*!< GP_T::INTEN: RHIEN1 Position              */
#define GP_INTEN_RHIEN1_Msk              (0x1ul << GP_INTEN_RHIEN1_Pos)                    /*!< GP_T::INTEN: RHIEN1 Mask                  */

#define GP_INTEN_RHIEN2_Pos              (18)                                              /*!< GP_T::INTEN: RHIEN2 Position              */
#define GP_INTEN_RHIEN2_Msk              (0x1ul << GP_INTEN_RHIEN2_Pos)                    /*!< GP_T::INTEN: RHIEN2 Mask                  */

#define GP_INTEN_RHIEN3_Pos              (19)                                              /*!< GP_T::INTEN: RHIEN3 Position              */
#define GP_INTEN_RHIEN3_Msk              (0x1ul << GP_INTEN_RHIEN3_Pos)                    /*!< GP_T::INTEN: RHIEN3 Mask                  */

#define GP_INTEN_RHIEN4_Pos              (20)                                              /*!< GP_T::INTEN: RHIEN4 Position              */
#define GP_INTEN_RHIEN4_Msk              (0x1ul << GP_INTEN_RHIEN4_Pos)                    /*!< GP_T::INTEN: RHIEN4 Mask                  */

#define GP_INTEN_RHIEN5_Pos              (21)                                              /*!< GP_T::INTEN: RHIEN5 Position              */
#define GP_INTEN_RHIEN5_Msk              (0x1ul << GP_INTEN_RHIEN5_Pos)                    /*!< GP_T::INTEN: RHIEN5 Mask                  */

#define GP_INTEN_RHIEN6_Pos              (22)                                              /*!< GP_T::INTEN: RHIEN6 Position              */
#define GP_INTEN_RHIEN6_Msk              (0x1ul << GP_INTEN_RHIEN6_Pos)                    /*!< GP_T::INTEN: RHIEN6 Mask                  */

#define GP_INTEN_RHIEN7_Pos              (23)                                              /*!< GP_T::INTEN: RHIEN7 Position              */
#define GP_INTEN_RHIEN7_Msk              (0x1ul << GP_INTEN_RHIEN7_Pos)                    /*!< GP_T::INTEN: RHIEN7 Mask                  */

#define GP_INTSRC_INTSRC0_Pos            (0)                                               /*!< GP_T::INTSRC: INTSRC0 Position            */
#define GP_INTSRC_INTSRC0_Msk            (0x1ul << GP_INTSRC_INTSRC0_Pos)                  /*!< GP_T::INTSRC: INTSRC0 Mask                */

#define GP_INTSRC_INTSRC1_Pos            (1)                                               /*!< GP_T::INTSRC: INTSRC1 Position            */
#define GP_INTSRC_INTSRC1_Msk            (0x1ul << GP_INTSRC_INTSRC1_Pos)                  /*!< GP_T::INTSRC: INTSRC1 Mask                */

#define GP_INTSRC_INTSRC2_Pos            (2)                                               /*!< GP_T::INTSRC: INTSRC2 Position            */
#define GP_INTSRC_INTSRC2_Msk            (0x1ul << GP_INTSRC_INTSRC2_Pos)                  /*!< GP_T::INTSRC: INTSRC2 Mask                */

#define GP_INTSRC_INTSRC3_Pos            (3)                                               /*!< GP_T::INTSRC: INTSRC3 Position            */
#define GP_INTSRC_INTSRC3_Msk            (0x1ul << GP_INTSRC_INTSRC3_Pos)                  /*!< GP_T::INTSRC: INTSRC3 Mask                */

#define GP_INTSRC_INTSRC4_Pos            (4)                                               /*!< GP_T::INTSRC: INTSRC4 Position            */
#define GP_INTSRC_INTSRC4_Msk            (0x1ul << GP_INTSRC_INTSRC4_Pos)                  /*!< GP_T::INTSRC: INTSRC4 Mask                */

#define GP_INTSRC_INTSRC5_Pos            (5)                                               /*!< GP_T::INTSRC: INTSRC5 Position            */
#define GP_INTSRC_INTSRC5_Msk            (0x1ul << GP_INTSRC_INTSRC5_Pos)                  /*!< GP_T::INTSRC: INTSRC5 Mask                */

#define GP_INTSRC_INTSRC6_Pos            (6)                                               /*!< GP_T::INTSRC: INTSRC6 Position            */
#define GP_INTSRC_INTSRC6_Msk            (0x1ul << GP_INTSRC_INTSRC6_Pos)                  /*!< GP_T::INTSRC: INTSRC6 Mask                */

#define GP_INTSRC_INTSRC7_Pos            (7)                                               /*!< GP_T::INTSRC: INTSRC7 Position            */
#define GP_INTSRC_INTSRC7_Msk            (0x1ul << GP_INTSRC_INTSRC7_Pos)                  /*!< GP_T::INTSRC: INTSRC7 Mask                */

#define GP_INTSRC_INTSRC8_Pos            (8)                                               /*!< GP_T::INTSRC: INTSRC8 Position            */
#define GP_INTSRC_INTSRC8_Msk            (0x1ul << GP_INTSRC_INTSRC8_Pos)                  /*!< GP_T::INTSRC: INTSRC8 Mask                */

#define GP_INTSRC_INTSRC9_Pos            (9)                                               /*!< GP_T::INTSRC: INTSRC9 Position            */
#define GP_INTSRC_INTSRC9_Msk            (0x1ul << GP_INTSRC_INTSRC9_Pos)                  /*!< GP_T::INTSRC: INTSRC9 Mask                */

#define GP_INTSRC_INTSRC10_Pos           (10)                                              /*!< GP_T::INTSRC: INTSRC10 Position           */
#define GP_INTSRC_INTSRC10_Msk           (0x1ul << GP_INTSRC_INTSRC10_Pos)                 /*!< GP_T::INTSRC: INTSRC10 Mask               */

#define GP_INTSRC_INTSRC11_Pos           (11)                                              /*!< GP_T::INTSRC: INTSRC11 Position           */
#define GP_INTSRC_INTSRC11_Msk           (0x1ul << GP_INTSRC_INTSRC11_Pos)                 /*!< GP_T::INTSRC: INTSRC11 Mask               */

#define GP_INTSRC_INTSRC12_Pos           (12)                                              /*!< GP_T::INTSRC: INTSRC12 Position           */
#define GP_INTSRC_INTSRC12_Msk           (0x1ul << GP_INTSRC_INTSRC12_Pos)                 /*!< GP_T::INTSRC: INTSRC12 Mask               */

#define GP_INTSRC_INTSRC13_Pos           (13)                                              /*!< GP_T::INTSRC: INTSRC13 Position           */
#define GP_INTSRC_INTSRC13_Msk           (0x1ul << GP_INTSRC_INTSRC13_Pos)                 /*!< GP_T::INTSRC: INTSRC13 Mask               */

#define GP_INTSRC_INTSRC14_Pos           (14)                                              /*!< GP_T::INTSRC: INTSRC14 Position           */
#define GP_INTSRC_INTSRC14_Msk           (0x1ul << GP_INTSRC_INTSRC14_Pos)                 /*!< GP_T::INTSRC: INTSRC14 Mask               */

#define GP_INTSRC_INTSRC15_Pos           (15)                                              /*!< GP_T::INTSRC: INTSRC15 Position           */
#define GP_INTSRC_INTSRC15_Msk           (0x1ul << GP_INTSRC_INTSRC15_Pos)                 /*!< GP_T::INTSRC: INTSRC15 Mask               */

#define GP_DBCTL_DBCLKSEL_Pos            (0)                                               /*!< GP_T::DBCTL: DBCLKSEL Position            */
#define GP_DBCTL_DBCLKSEL_Msk            (0xful << GP_DBCTL_DBCLKSEL_Pos)                  /*!< GP_T::DBCTL: DBCLKSEL Mask                */

#define GP_DBCTL_DBCLKSRC_Pos            (4)                                               /*!< GP_T::DBCTL: DBCLKSRC Position            */
#define GP_DBCTL_DBCLKSRC_Msk            (0x1ul << GP_DBCTL_DBCLKSRC_Pos)                  /*!< GP_T::DBCTL: DBCLKSRC Mask                */

#define GP_DBCTL_ICLKON_Pos              (5)                                               /*!< GP_T::DBCTL: ICLKON Position              */
#define GP_DBCTL_ICLKON_Msk              (0x1ul << GP_DBCTL_ICLKON_Pos)                    /*!< GP_T::DBCTL: ICLKON Mask                  */

/**@}*/ /* GP_CONST */
/**@}*/ /* end of GP register group */







/*---------------------- Analog to Digital Converter -------------------------*/
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
/**
    @addtogroup ADC Analog to Digital Converter(ADC)
    Memory Mapped Structure for ADC Controller
@{ */

typedef struct {
	__I  uint32_t ADC_DAT;
	__IO uint32_t ADC_CTL;
	__IO uint32_t ADC_STATUS;
	__IO uint32_t ADC_CTL2;
	__IO uint32_t ADC_WDR1;
	__IO uint32_t ADC_WDR2;
	__IO uint32_t ADC_WDR3;
	__IO uint32_t ADC_WDR4;
	__IO uint32_t ADC_WDR5;
	__IO uint32_t ADC_TEMP;
} ADC_T;

// ADC_DAT
#define ADC_DAT_RESULT_Pos                 (0)
#define ADC_DAT_RESULT_Msk                 (0xfffful << ADC_DAT_RESULT_Pos)

// ADC_CTL
#define ADC_CTL_DWIDTH_SEL_Pos             (15)
#define ADC_CTL_DWIDTH_SEL_Msk             (0x1ul << ADC_CTL_DWIDTH_SEL_Pos)

#define ADC_CTL_HW_TMR_SEL_Pos             (13)
#define ADC_CTL_HW_TMR_SEL_Msk             (0x3ul << ADC_CTL_HW_TMR_SEL_Pos)

#define ADC_CTL_HW_TMR_EN_Pos              (12)
#define ADC_CTL_HW_TMR_EN_Msk              (0x3ul << ADC_CTL_HW_TMR_EN_Pos)

#define ADC_CTL_HALF_INT_EN_Pos            (11)
#define ADC_CTL_HALF_INT_EN_Msk            (0x1ul << ADC_CTL_HALF_INT_EN_Pos)

#define ADC_CTL_OV_INT_EN_Pos              (10)
#define ADC_CTL_OV_INT_EN_Msk              (0x1ul << ADC_CTL_OV_INT_EN_Pos)

#define ADC_CTL_EMPTY_INT_EN_Pos           (9)
#define ADC_CTL_EMPTY_INT_EN_Msk           (0x1ul << ADC_CTL_EMPTY_INT_EN_Pos)

#define ADC_CTL_FULL_INT_EN_Pos            (8)
#define ADC_CTL_FULL_INT_EN_Msk            (0x1ul << ADC_CTL_FULL_INT_EN_Pos)

#define ADC_CTL_PWM_CH_SEL_Pos             (7)
#define ADC_CTL_PWM_CH_SEL_Msk             (0x1ul << ADC_CTL_PWM_CH_SEL_Pos)

#define ADC_CTL_SWTRG_Pos                  (6)
#define ADC_CTL_SWTRG_Msk                  (0x1ul << ADC_CTL_SWTRG_Pos)

#define ADC_CTL_FIFO_EN_Pos                (5)
#define ADC_CTL_FIFO_EN_Msk                (0x1ul << ADC_CTL_FIFO_EN_Pos)

#define ADC_CTL_HWTRGEN_Pos                (4)
#define ADC_CTL_HWTRGEN_Msk                (0x1ul << ADC_CTL_HWTRGEN_Pos)

#define ADC_CTL_HWTRGCOND_Pos              (3)
#define ADC_CTL_HWTRGCOND_Msk              (0x1ul << ADC_CTL_HWTRGCOND_Pos)

#define ADC_CTL_HWTRGSEL_Pos               (2)
#define ADC_CTL_HWTRGSEL_Msk               (0x1ul << ADC_CTL_HWTRGSEL_Pos)

#define ADC_CTL_ADCIEN_Pos                 (1)
#define ADC_CTL_ADCIEN_Msk                 (0x1ul << ADC_CTL_ADCIEN_Pos)

#define ADC_CTL_ADCEN_Pos                  (0)
#define ADC_CTL_ADCEN_Msk                  (0x1ul << ADC_CTL_ADCEN_Pos)

// ADC_STATUS
#define ADC_STATUS_ADC_FLAG_Pos            (8)
#define ADC_STATUS_ADC_FLAG_Msk            (0x1ul << ADC_STATUS_ADC_FLAG_Pos)

#define ADC_STATUS_BUSY_Pos                (6)
#define ADC_STATUS_BUSY_Msk                (0x1ul << ADC_STATUS_BUSY_Pos)

#define ADC_STATUS_FIFO_HALF_FALG_Pos      (5)
#define ADC_STATUS_FIFO_HALF_FALG_Msk      (0x1ul << ADC_STATUS_FIFO_HALF_FALG_Pos)

#define ADC_STATUS_FIFO_OV_FALG_Pos        (4)
#define ADC_STATUS_FIFO_OV_FALG_Msk        (0x1ul << ADC_STATUS_FIFO_OV_FALG_Pos)

#define ADC_STATUS_FIFO_EMPTY_FALG_Pos     (3)
#define ADC_STATUS_FIFO_EMPTY_FALG_Msk     (0x1ul << ADC_STATUS_FIFO_EMPTY_FALG_Pos)

#define ADC_STATUS_FIFO_FULL_FALG_Pos      (2)
#define ADC_STATUS_FIFO_FULL_FALG_Msk      (0x1ul << ADC_STATUS_FIFO_FULL_FALG_Pos)

#define ADC_STATUS_OV_Pos                  (1)
#define ADC_STATUS_OV_Msk                  (0x1ul << ADC_STATUS_OV_Pos)

#define ADC_STATUS_VALID_Pos               (0)
#define ADC_STATUS_VALID_Msk               (0x1ul << ADC_STATUS_VALID_Pos)

// ADC_CTL2
#define ADC_CTL2_BIT_TCTRL_Pos             (29)
#define ADC_CTL2_BIT_TCTRL_Msk             (0x7ul << ADC_CTL2_BIT_TCTRL_Pos)

#define ADC_CTL2_BUF_CHOP_Pos              (27)
#define ADC_CTL2_BUF_CHOP_Msk              (0x3ul << ADC_CTL2_BUF_CHOP_Pos)

#define ADC_CTL2_BUFFER_EN_Pos             (26)
#define ADC_CTL2_BUFFER_EN_Msk             (0x1ul << ADC_CTL2_BUFFER_EN_Pos)

#define ADC_CTL2_GPADC_VBG_TRIM_Pos        (18)
#define ADC_CTL2_GPADC_VBG_TRIM_Msk        (0x3ful << ADC_CTL2_GPADC_VBG_TRIM_Pos)

#define ADC_CTL2_GPADC_CHANNEL_SEL_Pos     (16)
#define ADC_CTL2_GPADC_CHANNEL_SEL_Msk     (0x3ul << ADC_CTL2_GPADC_CHANNEL_SEL_Pos)

#define ADC_CTL2_CLK_DIV_Pos               (13)
#define ADC_CTL2_CLK_DIV_Msk               (0x7ul << ADC_CTL2_CLK_DIV_Pos)

#define ADC_CTL2_CLK_SEL_Pos               (11)
#define ADC_CTL2_CLK_SEL_Msk               (0x3ul << ADC_CTL2_CLK_SEL_Pos)

#define ADC_CTL2_DC_TIMES_Pos              (8)
#define ADC_CTL2_DC_TIMES_Msk              (0x7ul << ADC_CTL2_DC_TIMES_Pos)

#define ADC_CTL2_DISC_NUM_Pos              (4)
#define ADC_CTL2_DISC_NUM_Msk              (0xful << ADC_CTL2_DISC_NUM_Pos)

#define ADC_CTL2_ANA_WORK_Pos              (3)
#define ADC_CTL2_ANA_WORK_Msk              (0x1ul << ADC_CTL2_ANA_WORK_Pos)

#define ADC_CTL2_CON_SAMPLE_Pos            (2)
#define ADC_CTL2_CON_SAMPLE_Msk            (0x1ul << ADC_CTL2_CON_SAMPLE_Pos)

#define ADC_CTL2_AVERAGE_Pos               (1)
#define ADC_CTL2_AVERAGE_Msk               (0x1ul << ADC_CTL2_AVERAGE_Pos)

#define ADC_CTL2_TEST_MODE_Pos             (0)
#define ADC_CTL2_TEST_MODE_Msk             (0x1ul << ADC_CTL2_TEST_MODE_Pos)

// ADC_WDR1
#define ADC_WDR1_S1_Pos                    (27)
#define ADC_WDR1_S1_Msk                    (0x1ful << ADC_WDR1_S1_Pos)

#define ADC_WDR1_S11_Pos                   (5)
#define ADC_WDR1_S11_Msk                   (0x7fful << ADC_WDR1_S11_Pos)

#define ADC_WDR1_S0_Pos                    (0)
#define ADC_WDR1_S0_Msk                    (0x1ful << ADC_WDR1_S0_Pos)

// ADC_WDR2
#define ADC_WDR2_S15_Pos                   (18)
#define ADC_WDR2_S15_Msk                   (0x3ffful << ADC_WDR2_S15_Pos)

#define ADC_WDR2_S9_Pos                    (8)
#define ADC_WDR2_S9_Msk                    (0x3fful << ADC_WDR2_S9_Pos)

#define ADC_WDR2_S6_Pos                    (0)
#define ADC_WDR2_S6_Msk                    (0xfful << ADC_WDR2_S6_Pos)

// ADC_WDR3
#define ADC_WDR3_S14_Pos                   (19)
#define ADC_WDR3_S14_Msk                   (0x1ffful << ADC_WDR3_S14_Pos)

#define ADC_WDR3_S8_Pos                    (9)
#define ADC_WDR3_S8_Msk                    (0x3fful << ADC_WDR3_S8_Pos)

#define ADC_WDR3_S7_Pos                    (0)
#define ADC_WDR3_S7_Msk                    (0x1fful << ADC_WDR3_S7_Pos)

// ADC_WDR4
#define ADC_WDR4_S13_Pos                   (19)
#define ADC_WDR4_S13_Msk                   (0x1ffful << ADC_WDR4_S13_Pos)

#define ADC_WDR4_S12_Pos                   (7)
#define ADC_WDR4_S12_Msk                   (0xffful << ADC_WDR4_S12_Pos)

#define ADC_WDR4_S5_Pos                    (0)
#define ADC_WDR4_S5_Msk                    (0x7ful << ADC_WDR4_S5_Pos)

// ADC_WDR5
#define ADC_WDR5_S10_Pos                   (21)
#define ADC_WDR5_S10_Msk                   (0x7fful << ADC_WDR5_S10_Pos)

#define ADC_WDR5_S4_Pos                    (14)
#define ADC_WDR5_S4_Msk                    (0x7ful << ADC_WDR5_S4_Pos)

#define ADC_WDR5_S3_Pos                    (8)
#define ADC_WDR5_S3_Msk                    (0x3ful << ADC_WDR5_S3_Pos)

#define ADC_WDR5_S2_Pos                    (2)
#define ADC_WDR5_S2_Msk                    (0x3ful << ADC_WDR5_S2_Pos)

// ADC_TEMP
#define ADC_TEMP_GPADC_CAL_CAP_EN_Pos      (31)
#define ADC_TEMP_GPADC_CAL_CAP_EN_Msk      (0x1ul << ADC_TEMP_GPADC_CAL_CAP_EN_Pos)

#define ADC_TEMP_GPADC_CAL_CMP_EN_Pos      (30)
#define ADC_TEMP_GPADC_CAL_CMP_EN_Msk      (0x1ul << ADC_TEMP_GPADC_CAL_CMP_EN_Pos)

#define ADC_TEMP_GPADC_CALCAP_SEL_Pos      (27)
#define ADC_TEMP_GPADC_CALCAP_SEL_Msk      (0x7ul << ADC_TEMP_GPADC_CALCAP_SEL_Pos)

#define ADC_TEMP_GPADC_CALCAP_SIDE_SEL_Pos (26)
#define ADC_TEMP_GPADC_CALCAP_SIDE_SEL_Msk (0x1ul << ADC_TEMP_GPADC_CALCAP_SIDE_SEL_Pos)

#define ADC_TEMP_GPADC_CALCMP_VOS_Pos      (20)
#define ADC_TEMP_GPADC_CALCMP_VOS_Msk      (0x3ful << ADC_TEMP_GPADC_CALCMP_VOS_Pos)

#define ADC_TEMP_GPADC_VREFP_BYP_EN_Pos    (18)
#define ADC_TEMP_GPADC_VREFP_BYP_EN_Msk    (0x1ul << ADC_TEMP_GPADC_VREFP_BYP_EN_Pos)

#define ADC_TEMP_GPADC_VREFP_VBG_EN_Pos    (17)
#define ADC_TEMP_GPADC_VREFP_VBG_EN_Msk    (0x1ul << ADC_TEMP_GPADC_VREFP_VBG_EN_Pos)

#define ADC_TEMP_GPADC_DIFFMODE_Pos        (16)
#define ADC_TEMP_GPADC_DIFFMODE_Msk        (0x1ul << ADC_TEMP_GPADC_DIFFMODE_Pos)

#define ADC_TEMP_GPADC_PGA_GAIN_Pos        (13)
#define ADC_TEMP_GPADC_PGA_GAIN_Msk        (0x7ul << ADC_TEMP_GPADC_PGA_GAIN_Pos)

#define ADC_TEMP_GPADC_VBOOST_BYP_EN_Pos   (12)
#define ADC_TEMP_GPADC_VBOOST_BYP_EN_Msk   (0x1ul << ADC_TEMP_GPADC_VBOOST_BYP_EN_Pos)

/**@}*/ /* ADC_CONST */
/**@}*/ /* end of ADC register group */






/*---------------------- System Manger Controller -------------------------*/
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
/**
    @addtogroup SYS System Manger Controller(SYS)
    Memory Mapped Structure for SYS Controller
@{ */

typedef struct {
	__IO uint32_t SYS_P0_MFP;
	__IO uint32_t SYS_P1_MFP;
	__IO uint32_t SYS_P2_MFP;
	__IO uint32_t SYS_REGCTRL;
	__IO uint32_t SYS_CTRL;
	__IO uint32_t ANA_RFLDO;
	__IO uint32_t ANA_DFT;
} SYSTEM_T;


#define SYS_P0_MFP_EXT7_Pos                  (23)
#define SYS_P0_MFP_EXT7_Msk                  (0x1ul << SYS_P0_MFP_EXT7_Pos)

#define SYS_P0_MFP_EXT6_Pos                  (22)
#define SYS_P0_MFP_EXT6_Msk                  (0x1ul << SYS_P0_MFP_EXT6_Pos)

#define SYS_P0_MFP_EXT5_Pos                  (21)
#define SYS_P0_MFP_EXT5_Msk                  (0x1ul << SYS_P0_MFP_EXT5_Pos)

#define SYS_P0_MFP_EXT4_Pos                  (20)
#define SYS_P0_MFP_EXT4_Msk                  (0x1ul << SYS_P0_MFP_EXT4_Pos)

#define SYS_P0_MFP_EXT3_Pos                  (19)
#define SYS_P0_MFP_EXT3_Msk                  (0x1ul << SYS_P0_MFP_EXT3_Pos)

#define SYS_P0_MFP_EXT2_Pos                  (18)
#define SYS_P0_MFP_EXT2_Msk                  (0x1ul << SYS_P0_MFP_EXT2_Pos)

#define SYS_P0_MFP_EXT1_Pos                  (17)
#define SYS_P0_MFP_EXT1_Msk                  (0x1ul << SYS_P0_MFP_EXT1_Pos)

#define SYS_P0_MFP_EXT0_Pos                  (16)
#define SYS_P0_MFP_EXT0_Msk                  (0x1ul << SYS_P0_MFP_EXT0_Pos)

#define SYS_P0_MFP_ALT7_Pos                  (15)
#define SYS_P0_MFP_ALT7_Msk                  (0x1ul << SYS_P0_MFP_ALT7_Pos)

#define SYS_P0_MFP_ALT6_Pos                  (14)
#define SYS_P0_MFP_ALT6_Msk                  (0x1ul << SYS_P0_MFP_ALT6_Pos)

#define SYS_P0_MFP_ALT5_Pos                  (13)
#define SYS_P0_MFP_ALT5_Msk                  (0x1ul << SYS_P0_MFP_ALT5_Pos)

#define SYS_P0_MFP_ALT4_Pos                  (12)
#define SYS_P0_MFP_ALT4_Msk                  (0x1ul << SYS_P0_MFP_ALT4_Pos)

#define SYS_P0_MFP_ALT3_Pos                  (11)
#define SYS_P0_MFP_ALT3_Msk                  (0x1ul << SYS_P0_MFP_ALT3_Pos)

#define SYS_P0_MFP_ALT2_Pos                  (10)
#define SYS_P0_MFP_ALT2_Msk                  (0x1ul << SYS_P0_MFP_ALT2_Pos)

#define SYS_P0_MFP_ALT1_Pos                  (9)
#define SYS_P0_MFP_ALT1_Msk                  (0x1ul << SYS_P0_MFP_ALT1_Pos)

#define SYS_P0_MFP_ALT0_Pos                  (8)
#define SYS_P0_MFP_ALT0_Msk                  (0x1ul << SYS_P0_MFP_ALT0_Pos)

#define SYS_P0_MFP_MFP7_Pos                  (7)
#define SYS_P0_MFP_MFP7_Msk                  (0x1ul << SYS_P0_MFP_MFP7_Pos)

#define SYS_P0_MFP_MFP6_Pos                  (6)
#define SYS_P0_MFP_MFP6_Msk                  (0x1ul << SYS_P0_MFP_MFP6_Pos)

#define SYS_P0_MFP_MFP5_Pos                  (5)
#define SYS_P0_MFP_MFP5_Msk                  (0x1ul << SYS_P0_MFP_MFP5_Pos)

#define SYS_P0_MFP_MFP4_Pos                  (4)
#define SYS_P0_MFP_MFP4_Msk                  (0x1ul << SYS_P0_MFP_MFP4_Pos)

#define SYS_P0_MFP_MFP3_Pos                  (3)
#define SYS_P0_MFP_MFP3_Msk                  (0x1ul << SYS_P0_MFP_MFP3_Pos)

#define SYS_P0_MFP_MFP2_Pos                  (2)
#define SYS_P0_MFP_MFP2_Msk                  (0x1ul << SYS_P0_MFP_MFP2_Pos)

#define SYS_P0_MFP_MFP1_Pos                  (1)
#define SYS_P0_MFP_MFP1_Msk                  (0x1ul << SYS_P0_MFP_MFP1_Pos)

#define SYS_P0_MFP_MFP0_Pos                  (0)
#define SYS_P0_MFP_MFP0_Msk                  (0x1ul << SYS_P0_MFP_MFP0_Pos)

#define SYS_P1_MFP_EXT7_Pos                  (23)
#define SYS_P1_MFP_EXT7_Msk                  (0x1ul << SYS_P1_MFP_EXT7_Pos)

#define SYS_P1_MFP_EXT6_Pos                  (22)
#define SYS_P1_MFP_EXT6_Msk                  (0x1ul << SYS_P1_MFP_EXT6_Pos)

#define SYS_P1_MFP_EXT5_Pos                  (21)
#define SYS_P1_MFP_EXT5_Msk                  (0x1ul << SYS_P1_MFP_EXT5_Pos)

#define SYS_P1_MFP_EXT4_Pos                  (20)
#define SYS_P1_MFP_EXT4_Msk                  (0x1ul << SYS_P1_MFP_EXT4_Pos)

#define SYS_P1_MFP_EXT3_Pos                  (19)
#define SYS_P1_MFP_EXT3_Msk                  (0x1ul << SYS_P1_MFP_EXT3_Pos)

#define SYS_P1_MFP_EXT2_Pos                  (18)
#define SYS_P1_MFP_EXT2_Msk                  (0x1ul << SYS_P1_MFP_EXT2_Pos)

#define SYS_P1_MFP_EXT1_Pos                  (17)
#define SYS_P1_MFP_EXT1_Msk                  (0x1ul << SYS_P1_MFP_EXT1_Pos)

#define SYS_P1_MFP_EXT0_Pos                  (16)
#define SYS_P1_MFP_EXT0_Msk                  (0x1ul << SYS_P1_MFP_EXT0_Pos)

#define SYS_P1_MFP_ALT7_Pos                  (15)
#define SYS_P1_MFP_ALT7_Msk                  (0x1ul << SYS_P1_MFP_ALT7_Pos)

#define SYS_P1_MFP_ALT6_Pos                  (14)
#define SYS_P1_MFP_ALT6_Msk                  (0x1ul << SYS_P1_MFP_ALT6_Pos)

#define SYS_P1_MFP_ALT5_Pos                  (13)
#define SYS_P1_MFP_ALT5_Msk                  (0x1ul << SYS_P1_MFP_ALT5_Pos)

#define SYS_P1_MFP_ALT4_Pos                  (12)
#define SYS_P1_MFP_ALT4_Msk                  (0x1ul << SYS_P1_MFP_ALT4_Pos)

#define SYS_P1_MFP_ALT3_Pos                  (11)
#define SYS_P1_MFP_ALT3_Msk                  (0x1ul << SYS_P1_MFP_ALT3_Pos)

#define SYS_P1_MFP_ALT2_Pos                  (10)
#define SYS_P1_MFP_ALT2_Msk                  (0x1ul << SYS_P1_MFP_ALT2_Pos)

#define SYS_P1_MFP_ALT1_Pos                  (9)
#define SYS_P1_MFP_ALT1_Msk                  (0x1ul << SYS_P1_MFP_ALT1_Pos)

#define SYS_P1_MFP_ALT0_Pos                  (8)
#define SYS_P1_MFP_ALT0_Msk                  (0x1ul << SYS_P1_MFP_ALT0_Pos)

#define SYS_P1_MFP_MFP7_Pos                  (7)
#define SYS_P1_MFP_MFP7_Msk                  (0x1ul << SYS_P1_MFP_MFP7_Pos)

#define SYS_P1_MFP_MFP6_Pos                  (6)
#define SYS_P1_MFP_MFP6_Msk                  (0x1ul << SYS_P1_MFP_MFP6_Pos)

#define SYS_P1_MFP_MFP5_Pos                  (5)
#define SYS_P1_MFP_MFP5_Msk                  (0x1ul << SYS_P1_MFP_MFP5_Pos)

#define SYS_P1_MFP_MFP4_Pos                  (4)
#define SYS_P1_MFP_MFP4_Msk                  (0x1ul << SYS_P1_MFP_MFP4_Pos)

#define SYS_P1_MFP_MFP3_Pos                  (3)
#define SYS_P1_MFP_MFP3_Msk                  (0x1ul << SYS_P1_MFP_MFP3_Pos)

#define SYS_P1_MFP_MFP2_Pos                  (2)
#define SYS_P1_MFP_MFP2_Msk                  (0x1ul << SYS_P1_MFP_MFP2_Pos)

#define SYS_P1_MFP_MFP1_Pos                  (1)
#define SYS_P1_MFP_MFP1_Msk                  (0x1ul << SYS_P1_MFP_MFP1_Pos)

#define SYS_P1_MFP_MFP0_Pos                  (0)
#define SYS_P1_MFP_MFP0_Msk                  (0x1ul << SYS_P1_MFP_MFP0_Pos)

#define SYS_P2_MFP_EXT7_Pos                  (23)
#define SYS_P2_MFP_EXT7_Msk                  (0x1ul << SYS_P2_MFP_EXT7_Pos)

#define SYS_P2_MFP_EXT6_Pos                  (22)
#define SYS_P2_MFP_EXT6_Msk                  (0x1ul << SYS_P2_MFP_EXT6_Pos)

#define SYS_P2_MFP_EXT5_Pos                  (21)
#define SYS_P2_MFP_EXT5_Msk                  (0x1ul << SYS_P2_MFP_EXT5_Pos)

#define SYS_P2_MFP_EXT4_Pos                  (20)
#define SYS_P2_MFP_EXT4_Msk                  (0x1ul << SYS_P2_MFP_EXT4_Pos)

#define SYS_P2_MFP_EXT3_Pos                  (19)
#define SYS_P2_MFP_EXT3_Msk                  (0x1ul << SYS_P2_MFP_EXT3_Pos)

#define SYS_P2_MFP_EXT2_Pos                  (18)
#define SYS_P2_MFP_EXT2_Msk                  (0x1ul << SYS_P2_MFP_EXT2_Pos)

#define SYS_P2_MFP_EXT1_Pos                  (17)
#define SYS_P2_MFP_EXT1_Msk                  (0x1ul << SYS_P2_MFP_EXT1_Pos)

#define SYS_P2_MFP_EXT0_Pos                  (16)
#define SYS_P2_MFP_EXT0_Msk                  (0x1ul << SYS_P2_MFP_EXT0_Pos)

#define SYS_P2_MFP_ALT7_Pos                  (15)
#define SYS_P2_MFP_ALT7_Msk                  (0x1ul << SYS_P2_MFP_ALT7_Pos)

#define SYS_P2_MFP_ALT6_Pos                  (14)
#define SYS_P2_MFP_ALT6_Msk                  (0x1ul << SYS_P2_MFP_ALT6_Pos)

#define SYS_P2_MFP_ALT5_Pos                  (13)
#define SYS_P2_MFP_ALT5_Msk                  (0x1ul << SYS_P2_MFP_ALT5_Pos)

#define SYS_P2_MFP_ALT4_Pos                  (12)
#define SYS_P2_MFP_ALT4_Msk                  (0x1ul << SYS_P2_MFP_ALT4_Pos)

#define SYS_P2_MFP_ALT3_Pos                  (11)
#define SYS_P2_MFP_ALT3_Msk                  (0x1ul << SYS_P2_MFP_ALT3_Pos)

#define SYS_P2_MFP_ALT2_Pos                  (10)
#define SYS_P2_MFP_ALT2_Msk                  (0x1ul << SYS_P2_MFP_ALT2_Pos)

#define SYS_P2_MFP_ALT1_Pos                  (9)
#define SYS_P2_MFP_ALT1_Msk                  (0x1ul << SYS_P2_MFP_ALT1_Pos)

#define SYS_P2_MFP_ALT0_Pos                  (8)
#define SYS_P2_MFP_ALT0_Msk                  (0x1ul << SYS_P2_MFP_ALT0_Pos)

#define SYS_P2_MFP_MFP7_Pos                  (7)
#define SYS_P2_MFP_MFP7_Msk                  (0x1ul << SYS_P2_MFP_MFP7_Pos)

#define SYS_P2_MFP_MFP6_Pos                  (6)
#define SYS_P2_MFP_MFP6_Msk                  (0x1ul << SYS_P2_MFP_MFP6_Pos)

#define SYS_P2_MFP_MFP5_Pos                  (5)
#define SYS_P2_MFP_MFP5_Msk                  (0x1ul << SYS_P2_MFP_MFP5_Pos)

#define SYS_P2_MFP_MFP4_Pos                  (4)
#define SYS_P2_MFP_MFP4_Msk                  (0x1ul << SYS_P2_MFP_MFP4_Pos)

#define SYS_P2_MFP_MFP3_Pos                  (3)
#define SYS_P2_MFP_MFP3_Msk                  (0x1ul << SYS_P2_MFP_MFP3_Pos)

#define SYS_P2_MFP_MFP2_Pos                  (2)
#define SYS_P2_MFP_MFP2_Msk                  (0x1ul << SYS_P2_MFP_MFP2_Pos)

#define SYS_P2_MFP_MFP1_Pos                  (1)
#define SYS_P2_MFP_MFP1_Msk                  (0x1ul << SYS_P2_MFP_MFP1_Pos)

#define SYS_P2_MFP_MFP0_Pos                  (0)
#define SYS_P2_MFP_MFP0_Msk                  (0x1ul << SYS_P2_MFP_MFP0_Pos)

#define SYS_REGCTRL_REGCTRL_Pos              (0)
#define SYS_REGCTRL_REGCTRL_Msk              (0xfful << SYS_REGCTRL_REGCTRL_Pos)

#define SYS_CTRL_TMR_CAPT_SEL_Pos            (26)
#define SYS_CTRL_TMR_CAPT_SEL_Msk            (0x1ul << SYS_CTRL_TMR_CAPT_SEL_Pos)

#define SYS_CTRL_CLK_OUT_SEL_Pos             (25)
#define SYS_CTRL_CLK_OUT_SEL_Msk             (0x1ul << SYS_CTRL_CLK_OUT_SEL_Pos)

#define SYS_CTRL_USB_PU2_Pos                 (18)
#define SYS_CTRL_USB_PU2_Msk                 (0x1ul << SYS_CTRL_USB_PU2_Pos)

#define SYS_CTRL_USB_PU_Pos                  (17)
#define SYS_CTRL_USB_PU_Msk                  (0x1ul << SYS_CTRL_USB_PU_Pos)

#define SYS_CTRL_USB_EN_Pos                  (16)
#define SYS_CTRL_USB_EN_Msk                  (0x1ul << SYS_CTRL_USB_EN_Pos)

#define SYS_CTRL_VALID_REMOVAL_CYCLES_Pos    (0)
#define SYS_CTRL_VALID_REMOVAL_CYCLES_Msk    (0xfffful << SYS_CTRL_VALID_REMOVAL_CYCLES_Pos)

// ANA_RFLDO
#define ANA_RFLDO_LDO_ADC_TRIM_Pos        (10)
#define ANA_RFLDO_LDO_ADC_TRIM_Msk        (0xful << ANA_RFLDO_LDO_ADC_TRIM_Pos)

#define ANA_RFLDO_ADCLDO_LOAD_SEL_Pos     (8)
#define ANA_RFLDO_ADCLDO_LOAD_SEL_Msk     (0x3ul << ANA_RFLDO_ADCLDO_LOAD_SEL_Pos)

#define ANA_RFLDO_LDO_RFFE_TRIM_Pos       (4)
#define ANA_RFLDO_LDO_RFFE_TRIM_Msk       (0xful << ANA_RFLDO_LDO_RFFE_TRIM_Pos)

#define ANA_RFLDO_EN_LDO_ADC_Pos          (0)
#define ANA_RFLDO_EN_LDO_ADC_Msk          (0x1ul << ANA_RFLDO_EN_LDO_ADC_Pos)

// ANA_DFT
#define ANA_DFT_EN_DFT_V_Pos              (14)
#define ANA_DFT_EN_DFT_V_Msk              (0x1ul << ANA_DFT_EN_DFT_V_Pos)

#define ANA_DFT_DFT_V_SEL_Pos             (7)
#define ANA_DFT_DFT_V_SEL_Msk             (0xful << ANA_DFT_DFT_V_SEL_Pos)

#define ANA_DFT_DFT_V_BYPASS_Pos          (0)
#define ANA_DFT_DFT_V_BYPASS_Msk          (0x1ul << ANA_DFT_DFT_V_BYPASS_Pos)

/**@}*/ /* SYS_CONST */
/**@}*/ /* end of SYS register group */








/**
  * @}
  */
/*---------------------- System Clock Controller -------------------------*/
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
/**
    @addtogroup CLK System Clock Controller(CLK)
    Memory Mapped Structure for CLK Controller
@{ */

typedef struct {
	__IO uint32_t RSTSTS;
	__IO uint32_t IPRST0;
	__IO uint32_t IPRST1;
	__IO uint32_t AHB_CLK_CTRL;
	__IO uint32_t APB1_CLK_CTRL0;
} RCC_T;


#define RSTSTS_CHIP1RF_Pos                 (8)
#define RSTSTS_CHIP1RF_Msk                 (0x1ul << RSTSTS_CHIP1RF_Pos)

#define RSTSTS_CPURF_Pos                   (7)
#define RSTSTS_CPURF_Msk                   (0x1ul << RSTSTS_CPURF_Pos)

#define RSTSTS_PORRF_Pos                   (6)
#define RSTSTS_PORRF_Msk                   (0x1ul << RSTSTS_PORRF_Pos)

#define RSTSTS_SYSRF_Pos                   (5)
#define RSTSTS_SYSRF_Msk                   (0x1ul << RSTSTS_SYSRF_Pos)

#define RSTSTS_BODRF_Pos                   (4)
#define RSTSTS_BODRF_Msk                   (0x1ul << RSTSTS_BODRF_Pos)

#define RSTSTS_WDTRF_Pos                   (2)
#define RSTSTS_WDTRF_Msk                   (0x1ul << RSTSTS_WDTRF_Pos)

#define RSTSTS_PINRF_Pos                   (1)
#define RSTSTS_PINRF_Msk                   (0x1ul << RSTSTS_PINRF_Pos)

#define RSTSTS_CHIP0RF_Pos                 (0)
#define RSTSTS_CHIP0RF_Msk                 (0x1ul << RSTSTS_CHIP0RF_Pos)

#define IPRST0_CHIP1RST_Pos                (8)
#define IPRST0_CHIP1RST_Msk                (0x1ul << IPRST0_CHIP1RST_Pos)

#define IPRST0_MDMRST_Pos                  (7)
#define IPRST0_MDMRST_Msk                  (0x1ul << IPRST0_MDMRST_Pos)

#define IPRST0_USBRST_Pos                  (5)
#define IPRST0_USBRST_Msk                  (0x1ul << IPRST0_USBRST_Pos)

#define IPRST0_MACRST_Pos                  (3)
#define IPRST0_MACRST_Msk                  (0x1ul << IPRST0_MACRST_Pos)

#define IPRST0_CPURST_Pos                  (1)
#define IPRST0_CPURST_Msk                  (0x1ul << IPRST0_CPURST_Pos)

#define IPRST0_CHIPRST_Pos                 (0)
#define IPRST0_CHIPRST_Msk                 (0x1ul << IPRST0_CHIPRST_Pos)

#define IPRST1_GPIORST_Pos                 (15)
#define IPRST1_GPIORST_Msk                 (0x1ul << IPRST1_GPIORST_Pos)

#define IPRST1_KEYSCANRST_Pos              (14)
#define IPRST1_KEYSCANRST_Msk              (0x1ul << IPRST1_KEYSCANRST_Pos)

#define IPRST1_CLKMEASRST_Pos              (13)
#define IPRST1_CLKMEASRST_Msk              (0x1ul << IPRST1_CLKMEASRST_Pos)

#define IPRST1_TMR0RST_Pos                 (12)
#define IPRST1_TMR0RST_Msk                 (0x1ul << IPRST1_TMR0RST_Pos)

#define IPRST1_WDTRST_Pos                  (10)
#define IPRST1_WDTRST_Msk                  (0x1ul << IPRST1_WDTRST_Pos)

#define IPRST1_ADCRST_Pos                  (9)
#define IPRST1_ADCRST_Msk                  (0x1ul << IPRST1_ADCRST_Pos)

#define IPRST1_PWM0RST_Pos                 (8)
#define IPRST1_PWM0RST_Msk                 (0x1ul << IPRST1_PWM0RST_Pos)

#define IPRST1_UART0RST_Pos                (6)
#define IPRST1_UART0RST_Msk                (0x1ul << IPRST1_UART0RST_Pos)

#define IPRST1_SPI0RST_Pos                 (2)
#define IPRST1_SPI0RST_Msk                 (0x1ul << IPRST1_SPI0RST_Pos)

#define IPRST1_I2C0RST_Pos                 (0)
#define IPRST1_I2C0RST_Msk                 (0x1ul << IPRST1_I2C0RST_Pos)

#define AHB_CLK_CTRL_XTH_DIV_Pos           (23)
#define AHB_CLK_CTRL_XTH_DIV_Msk           (0x1fful << AHB_CLK_CTRL_XTH_DIV_Pos)

#define AHB_CLK_CTRL_XTL_QUICK_EN_Pos      (22)
#define AHB_CLK_CTRL_XTL_QUICK_EN_Msk      (0x1ul << AHB_CLK_CTRL_XTL_QUICK_EN_Pos)

#define AHB_CLK_CTRL_USB_48M_CLK_EN_Pos    (20)
#define AHB_CLK_CTRL_USB_48M_CLK_EN_Msk    (0x1ul << AHB_CLK_CTRL_USB_48M_CLK_EN_Pos)

#define AHB_CLK_CTRL_USB_AHB_CLK_EN_Pos    (19)
#define AHB_CLK_CTRL_USB_AHB_CLK_EN_Msk    (0x1ul << AHB_CLK_CTRL_USB_AHB_CLK_EN_Pos)

#define AHB_CLK_CTRL_PHY_CLK_EN_Pos        (18)
#define AHB_CLK_CTRL_PHY_CLK_EN_Msk        (0x1ul << AHB_CLK_CTRL_PHY_CLK_EN_Pos)

#define AHB_CLK_CTRL_MAC_CLK_EN_Pos        (17)
#define AHB_CLK_CTRL_MAC_CLK_EN_Msk        (0x1ul << AHB_CLK_CTRL_MAC_CLK_EN_Pos)

#define AHB_CLK_CTRL_APB1_CLK_EN_Pos       (15)
#define AHB_CLK_CTRL_APB1_CLK_EN_Msk       (0x1ul << AHB_CLK_CTRL_APB1_CLK_EN_Pos)

#define AHB_CLK_CTRL_ST_CLK_EN_Pos         (14)
#define AHB_CLK_CTRL_ST_CLK_EN_Msk         (0x1ul << AHB_CLK_CTRL_ST_CLK_EN_Pos)

#define AHB_CLK_CTRL_GPIO_CLK_EN_Pos       (13)
#define AHB_CLK_CTRL_GPIO_CLK_EN_Msk       (0x1ul << AHB_CLK_CTRL_GPIO_CLK_EN_Pos)

#define AHB_CLK_CTRL_SLP_AHB_CLK_DIV_Pos   (8)
#define AHB_CLK_CTRL_SLP_AHB_CLK_DIV_Msk   (0xful << AHB_CLK_CTRL_SLP_AHB_CLK_DIV_Pos)

#define AHB_CLK_CTRL_APB1_DIV_Pos          (4)
#define AHB_CLK_CTRL_APB1_DIV_Msk          (0xful << AHB_CLK_CTRL_APB1_DIV_Pos)

#define AHB_CLK_CTRL_AHB_DIV_Pos           (0)
#define AHB_CLK_CTRL_AHB_DIV_Msk           (0xful << AHB_CLK_CTRL_AHB_DIV_Pos)

#define APB1_CLK_CTRL0_KEYSCAN_DIV_Pos     (23)
#define APB1_CLK_CTRL0_KEYSCAN_DIV_Msk     (0x1fful << APB1_CLK_CTRL0_KEYSCAN_DIV_Pos)

#define APB1_CLK_CTRL0_ADC_CLK_SEL_Pos     (22)
#define APB1_CLK_CTRL0_ADC_CLK_SEL_Msk     (0x1ul << APB1_CLK_CTRL0_ADC_CLK_SEL_Pos)

#define APB1_CLK_CTRL0_KEYSCAN_CLK_SEL_Pos (21)
#define APB1_CLK_CTRL0_KEYSCAN_CLK_SEL_Msk (0x1ul << APB1_CLK_CTRL0_KEYSCAN_CLK_SEL_Pos)

#define APB1_CLK_CTRL0_PWM0_45CKSEL_Pos    (20)
#define APB1_CLK_CTRL0_PWM0_45CKSEL_Msk    (0x1ul << APB1_CLK_CTRL0_PWM0_45CKSEL_Pos)

#define APB1_CLK_CTRL0_PWM0_23CKSEL_Pos    (19)
#define APB1_CLK_CTRL0_PWM0_23CKSEL_Msk    (0x1ul << APB1_CLK_CTRL0_PWM0_23CKSEL_Pos)

#define APB1_CLK_CTRL0_PWM0_01CKSEL_Pos    (18)
#define APB1_CLK_CTRL0_PWM0_01CKSEL_Msk    (0x1ul << APB1_CLK_CTRL0_PWM0_01CKSEL_Pos)

#define APB1_CLK_CTRL0_TMR0SEL_Pos         (16)
#define APB1_CLK_CTRL0_TMR0SEL_Msk         (0x3ul << APB1_CLK_CTRL0_TMR0SEL_Pos)

#define APB1_CLK_CTRL0_KEYSCANCKEN_Pos     (14)
#define APB1_CLK_CTRL0_KEYSCANCKEN_Msk     (0x1ul << APB1_CLK_CTRL0_KEYSCANCKEN_Pos)

#define APB1_CLK_CTRL0_CLKMEASCKEN_Pos     (13)
#define APB1_CLK_CTRL0_CLKMEASCKEN_Msk     (0x1ul << APB1_CLK_CTRL0_CLKMEASCKEN_Pos)

#define APB1_CLK_CTRL0_TMR0CKEN_Pos        (12)
#define APB1_CLK_CTRL0_TMR0CKEN_Msk        (0x1ul << APB1_CLK_CTRL0_TMR0CKEN_Pos)

#define APB1_CLK_CTRL0_WDTCKEN_Pos         (10)
#define APB1_CLK_CTRL0_WDTCKEN_Msk         (0x1ul << APB1_CLK_CTRL0_WDTCKEN_Pos)

#define APB1_CLK_CTRL0_ADCCKEN_Pos         (9)
#define APB1_CLK_CTRL0_ADCCKEN_Msk         (0x1ul << APB1_CLK_CTRL0_ADCCKEN_Pos)

#define APB1_CLK_CTRL0_PWM0CKEN_Pos        (8)
#define APB1_CLK_CTRL0_PWM0CKEN_Msk        (0x1ul << APB1_CLK_CTRL0_PWM0CKEN_Pos)

#define APB1_CLK_CTRL0_PWM0_45CKEN_Pos     (6)
#define APB1_CLK_CTRL0_PWM0_45CKEN_Msk     (0x1ul << APB1_CLK_CTRL0_PWM0_45CKEN_Pos)

#define APB1_CLK_CTRL0_PWM0_23CKEN_Pos     (5)
#define APB1_CLK_CTRL0_PWM0_23CKEN_Msk     (0x1ul << APB1_CLK_CTRL0_PWM0_23CKEN_Pos)

#define APB1_CLK_CTRL0_PWM0_01CKEN_Pos     (4)
#define APB1_CLK_CTRL0_PWM0_01CKEN_Msk     (0x1ul << APB1_CLK_CTRL0_PWM0_01CKEN_Pos)

#define APB1_CLK_CTRL0_UART0CKEN_Pos       (3)
#define APB1_CLK_CTRL0_UART0CKEN_Msk       (0x1ul << APB1_CLK_CTRL0_UART0CKEN_Pos)

#define APB1_CLK_CTRL0_SPI0CKEN_Pos        (1)
#define APB1_CLK_CTRL0_SPI0CKEN_Msk        (0x1ul << APB1_CLK_CTRL0_SPI0CKEN_Pos)

#define APB1_CLK_CTRL0_I2C0CKEN_Pos        (0)
#define APB1_CLK_CTRL0_I2C0CKEN_Msk        (0x1ul << APB1_CLK_CTRL0_I2C0CKEN_Pos)


// /**@}*/ /* CLK_CONST */
// /**@}*/ /* end of CLK register group */


/*---------------------- Watch Dog Timer Controller -------------------------*/
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
/**
    @addtogroup WDT Watch Dog Timer Controller(WDT)
    Memory Mapped Structure for WDT Controller
@{ */

typedef struct {


    /**
     * CTL
     * ===================================================================================================
     * Offset: 0x00  WDT Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[0]     |RSTCNT    |Reset WDT Up Counter (Write Protect)
     * |        |          |0 = No effect.
     * |        |          |1 = Reset the internal 18-bit WDT up counter value.
     * |        |          |Note1: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |        |          |Note2: This bit will be automatically cleared by hardware.
     * |[1]     |RSTEN     |WDT Time-out Reset Enable Bit (Write Protect)
     * |        |          |Setting this bit will enable the WDT time-out reset function If the WDT up counter value has not been cleared after the specific WDT reset delay period expires.
     * |        |          |0 = WDT time-out reset function Disabled.
     * |        |          |1 = WDT time-out reset function Enabled.
     * |        |          |Note: This bit is write-protected. Refer to the SYS_REGLCTL register.
     * |[2]     |RSTF      |WDT Time-out Reset Flag
     * |        |          |This bit indicates the system has been reset by WDT time-out reset or not.
     * |        |          |0 = WDT time-out reset did not occur.
     * |        |          |1 = WDT time-out reset occurred.
     * |        |          |Note: This bit is cleared by writing 1 to it.
     * |[3]     |IF        |WDT Time-out Interrupt Flag
     * |        |          |This bit will be set to 1 while WDT up counter value reaches the selected WDT time-out interval
     * |        |          |0 = WDT time-out interrupt did not occur.
     * |        |          |1 = WDT time-out interrupt occurred.
     * |        |          |Note: This bit is cleared by writing 1 to it.
     * |[4]     |WKEN      |WDT Time-out Wake-up Function Control (Write Protect)
     * |        |          |If this bit is set to 1, while WDT time-out interrupt flag IF (WDT_CTL[3]) is generated to 1 and interrupt enable bit INTEN (WDT_CTL[6]) is enabled, the WDT time-out interrupt signal will generate a wake-up trigger event to chip.
     * |        |          |0 = Wake-up trigger event Disabled if WDT time-out interrupt signal generated.
     * |        |          |1 = Wake-up trigger event Enabled if WDT time-out interrupt signal generated.
     * |        |          |Note1: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |        |          |Note2: Chip can be woken-up by WDT time-out interrupt signal generated only if WDT clock source is selected to LIRC or LXT.
     * |[5]     |WKF       |WDT Time-out Wake-up Flag (Write Protect)
     * |        |          |This bit indicates the interrupt wake-up flag status of WDT
     * |        |          |0 = WDT does not cause chip wake-up.
     * |        |          |1 = Chip wake-up from Idle or Power-down mode if WDT time-out interrupt signal generated.
     * |        |          |Note1: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |        |          |Note2: This bit is cleared by writing 1 to it.
     * |[6]     |INTEN     |WDT Time-out Interrupt Enable Bit (Write Protect)
     * |        |          |If this bit is enabled, the WDT time-out interrupt signal is generated and inform to CPU.
     * |        |          |0 = WDT time-out interrupt Disabled.
     * |        |          |1 = WDT time-out interrupt Enabled.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[7]     |WDTEN     |WDT Enable Bit (Write Protect)
     * |        |          |0 = WDT Disabled (This action will reset the internal up counter value).
     * |        |          |1 = WDT Enabled.
     * |        |          |Note1: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |        |          |Note2: If CWDTEN[2:0] (combined by Config0[31] and Config0[4:3]) bits is not configure to 111, this bit is forced as 1 and user cannot change this bit to 0.
     * |[10:8]  |TOUTSEL   |WDT Time-out Interval Selection (Write Protect)
     * |        |          |These three bits select the time-out interval period for the WDT.
     * |        |          |000 = 2^4 * WDT_CLK.
     * |        |          |001 = 2^6 * WDT_CLK.
     * |        |          |010 = 2^8 * WDT_CLK.
     * |        |          |011 = 2^10 * WDT_CLK.
     * |        |          |100 = 2^12 * WDT_CLK.
     * |        |          |101 = 2^14 * WDT_CLK.
     * |        |          |110 = 2^16 * WDT_CLK.
     * |        |          |111 = 2^18 * WDT_CLK.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |[16]     |TOF      |WDT Time-out Flag
     * |        |          |This bit will be set to 1 while WDT up counter value reaches the selected WDT time-out interval
     * |        |          |0 = WDT time-out did not occur.
     * |        |          |1 = WDT time-out occurred.
     * |        |          |Note: This bit is cleared by writing 1 to it.
     * |[31]    |ICEDEBUG  |ICE Debug Mode Acknowledge Disable Bit (Write Protect)
     * |        |          |0 = ICE debug mode acknowledgement affects WDT counting.
     * |        |          |WDT up counter will be held while CPU is held by ICE.
     * |        |          |1 = ICE debug mode acknowledgement Disabled.
     * |        |          |WDT up counter will keep going no matter CPU is held by ICE or not.
     * |        |          |Note: This bit is write protected. Refer to the SYS_REGLCTL register.
    */
    __IO uint32_t CTL;

    /**
     * ALTCTL
     * ===================================================================================================
     * Offset: 0x04  WDT Alternative Control Register
     * ---------------------------------------------------------------------------------------------------
     * |Bits    |Field     |Descriptions
     * | :----: | :----:   | :---- |
     * |[1:0]   |RSTDSEL   |WDT Reset Delay Selection (Write Protect)
     * |        |          |When WDT time-out happened, user has a time named WDT Reset Delay Period to clear WDT counter by setting RSTCNT (WDT_CTL[0]) to prevent WDT time-out reset happened.
     * |        |          |User can select a suitable setting of RSTDSEL for different WDT Reset Delay Period.
     * |        |          |00 = WDT Reset Delay Period is 1025 * WDT_CLK.
     * |        |          |01 = WDT Reset Delay Period is 129 * WDT_CLK.
     * |        |          |10 = WDT Reset Delay Period is 17 * WDT_CLK.
     * |        |          |11 = WDT Reset Delay Period is 2 * WDT_CLK.
     * |        |          |Note1: This bit is write protected. Refer to the SYS_REGLCTL register.
     * |        |          |Note2: This register will be reset to 0 if WDT time-out reset happened.
    */
    __IO uint32_t ALTCTL;

} WDT_T;

/**
    @addtogroup WDT_CONST WDT Bit Field Definition
    Constant Definitions for WDT Controller
@{ */

#define WDT_CTL_RSTCNT_Pos               (0)                                               /*!< WDT_T::CTL: RSTCNT Position               */
#define WDT_CTL_RSTCNT_Msk               (0x1ul << WDT_CTL_RSTCNT_Pos)                     /*!< WDT_T::CTL: RSTCNT Mask                   */

#define WDT_CTL_RSTEN_Pos                (1)                                               /*!< WDT_T::CTL: RSTEN Position                */
#define WDT_CTL_RSTEN_Msk                (0x1ul << WDT_CTL_RSTEN_Pos)                      /*!< WDT_T::CTL: RSTEN Mask                    */

#define WDT_CTL_RSTF_Pos                 (2)                                               /*!< WDT_T::CTL: RSTF Position                 */
#define WDT_CTL_RSTF_Msk                 (0x1ul << WDT_CTL_RSTF_Pos)                       /*!< WDT_T::CTL: RSTF Mask                     */

#define WDT_CTL_IF_Pos                   (3)                                               /*!< WDT_T::CTL: IF Position                   */
#define WDT_CTL_IF_Msk                   (0x1ul << WDT_CTL_IF_Pos)                         /*!< WDT_T::CTL: IF Mask                       */

#define WDT_CTL_WKEN_Pos                 (4)                                               /*!< WDT_T::CTL: WKEN Position                 */
#define WDT_CTL_WKEN_Msk                 (0x1ul << WDT_CTL_WKEN_Pos)                       /*!< WDT_T::CTL: WKEN Mask                     */

#define WDT_CTL_WKF_Pos                  (5)                                               /*!< WDT_T::CTL: WKF Position                  */
#define WDT_CTL_WKF_Msk                  (0x1ul << WDT_CTL_WKF_Pos)                        /*!< WDT_T::CTL: WKF Mask                      */

#define WDT_CTL_INTEN_Pos                (6)                                               /*!< WDT_T::CTL: INTEN Position                */
#define WDT_CTL_INTEN_Msk                (0x1ul << WDT_CTL_INTEN_Pos)                      /*!< WDT_T::CTL: INTEN Mask                    */

#define WDT_CTL_WDTEN_Pos                (7)                                               /*!< WDT_T::CTL: WDTEN Position                */
#define WDT_CTL_WDTEN_Msk                (0x1ul << WDT_CTL_WDTEN_Pos)                      /*!< WDT_T::CTL: WDTEN Mask                    */

#define WDT_CTL_TOUTSEL_Pos              (8)                                               /*!< WDT_T::CTL: TOUTSEL Position              */
#define WDT_CTL_TOUTSEL_Msk              (0xful << WDT_CTL_TOUTSEL_Pos)                    /*!< WDT_T::CTL: TOUTSEL Mask                  */

#define WDT_CTL_RST_REGION_SEL_Pos       (12)                                               /*!< WDT_T::CTL: TOUTSEL Position              */
#define WDT_CTL_RST_REGION_SEL_Msk       (0x1ul << WDT_CTL_RST_REGION_SEL_Pos)                    /*!< WDT_T::CTL: TOUTSEL Mask                  */

#define WDT_CTL_TOF_Pos                  (16)                                               /*!< WDT_T::CTL: TOF Position                   */
#define WDT_CTL_TOF_Msk                  (0x1ul << WDT_CTL_TOF_Pos)                         /*!< WDT_T::CTL: TOF Mask                       */

#define WDT_CTL_ICEDEBUG_Pos             (31)                                              /*!< WDT_T::CTL: ICEDEBUG Position             */
#define WDT_CTL_ICEDEBUG_Msk             (0x1ul << WDT_CTL_ICEDEBUG_Pos)                   /*!< WDT_T::CTL: ICEDEBUG Mask                 */

#define WDT_ALTCTL_RSTDSEL_Pos           (0)                                               /*!< WDT_T::ALTCTL: RSTDSEL Position           */
#define WDT_ALTCTL_RSTDSEL_Msk           (0x3ul << WDT_ALTCTL_RSTDSEL_Pos)                 /*!< WDT_T::ALTCTL: RSTDSEL Mask               */

/**@}*/ /* WDT_CONST */
/**@}*/ /* end of WDT register group */




/*---------------------- Pulse Width Modulation Controller -------------------------*/
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
/**
    @addtogroup PWM Pulse Width Modulation Controller(PWM)
    Memory Mapped Structure for PWM Controller
@{ */

typedef struct {
	__IO uint32_t PWM_CLKPSC_L;
	__IO uint32_t PWM_CLKDIV;
	__IO uint32_t PWM_CTL;
	__IO uint32_t PWM_PERIOD0;
	__IO uint32_t PWM_PERIOD1;
	__IO uint32_t PWM_PERIOD2;
	__IO uint32_t PWM_PERIOD3;
	__IO uint32_t PWM_PERIOD4;
	__IO uint32_t PWM_PERIOD5;
	__IO uint32_t RESERVED0;
	__IO uint32_t RESERVED1;
	__IO uint32_t PWM_CMPDAT0;
	__IO uint32_t PWM_CMPDAT1;
	__IO uint32_t PWM_CMPDAT2;
	__IO uint32_t PWM_CMPDAT3;
	__IO uint32_t PWM_CMPDAT4;
	__IO uint32_t PWM_CMPDAT5;
	__IO uint32_t RESERVED2;
	__IO uint32_t RESERVED3;
	__IO uint32_t PWM_CTL2;
	__IO uint32_t PWM_FLAG;
	__IO uint32_t PWM_INTEN;
	__IO uint32_t PWM_INTSTS;
	__IO uint32_t PWM_POEN;
	__IO uint32_t RESERVED4;
	__IO uint32_t PWM_DTCTL;
	__IO uint32_t PWM_CLKPSC_H;
} PWM_T;

/**
    @addtogroup PWM_CONST PWM Bit Field Definition
    Constant Definitions for PWM Controller
@{ */

#define PWM_CLKPSC_L_CLKPSC2_Pos (16)
#define PWM_CLKPSC_L_CLKPSC2_Msk (0xfful << PWM_CLKPSC_L_CLKPSC2_Pos)

#define PWM_CLKPSC_L_CLKPSC1_Pos (8)
#define PWM_CLKPSC_L_CLKPSC1_Msk (0xfful << PWM_CLKPSC_L_CLKPSC1_Pos)

#define PWM_CLKPSC_L_CLKPSC0_Pos (0)
#define PWM_CLKPSC_L_CLKPSC0_Msk (0xfful << PWM_CLKPSC_L_CLKPSC0_Pos)

#define PWM_CLKDIV_CLKDIV5_Pos   (20)
#define PWM_CLKDIV_CLKDIV5_Msk   (0x7ul << PWM_CLKDIV_CLKDIV5_Pos)

#define PWM_CLKDIV_CLKDIV4_Pos   (16)
#define PWM_CLKDIV_CLKDIV4_Msk   (0x7ul << PWM_CLKDIV_CLKDIV4_Pos)

#define PWM_CLKDIV_CLKDIV3_Pos   (12)
#define PWM_CLKDIV_CLKDIV3_Msk   (0x7ul << PWM_CLKDIV_CLKDIV3_Pos)

#define PWM_CLKDIV_CLKDIV2_Pos   (8)
#define PWM_CLKDIV_CLKDIV2_Msk   (0x7ul << PWM_CLKDIV_CLKDIV2_Pos)

#define PWM_CLKDIV_CLKDIV1_Pos   (4)
#define PWM_CLKDIV_CLKDIV1_Msk   (0x7ul << PWM_CLKDIV_CLKDIV1_Pos)

#define PWM_CLKDIV_CLKDIV0_Pos   (0)
#define PWM_CLKDIV_CLKDIV0_Msk   (0x7ul << PWM_CLKDIV_CLKDIV0_Pos)

#define PWM_CTL_PINV5_Pos        (22)
#define PWM_CTL_PINV5_Msk        (0x1ul << PWM_CTL_PINV5_Pos)

#define PWM_CTL_CNTEN5_Pos       (20)
#define PWM_CTL_CNTEN5_Msk       (0x1ul << PWM_CTL_CNTEN5_Pos)

#define PWM_CTL_PINV4_Pos        (18)
#define PWM_CTL_PINV4_Msk        (0x1ul << PWM_CTL_PINV4_Pos)

#define PWM_CTL_CNTEN4_Pos       (16)
#define PWM_CTL_CNTEN4_Msk       (0x1ul << PWM_CTL_CNTEN4_Pos)

#define PWM_CTL_PINV3_Pos        (14)
#define PWM_CTL_PINV3_Msk        (0x1ul << PWM_CTL_PINV3_Pos)

#define PWM_CTL_CNTEN3_Pos       (12)
#define PWM_CTL_CNTEN3_Msk       (0x1ul << PWM_CTL_CNTEN3_Pos)

#define PWM_CTL_PINV2_Pos        (10)
#define PWM_CTL_PINV2_Msk        (0x1ul << PWM_CTL_PINV2_Pos)

#define PWM_CTL_CNTEN2_Pos       (8)
#define PWM_CTL_CNTEN2_Msk       (0x1ul << PWM_CTL_CNTEN2_Pos)

#define PWM_CTL_PINV1_Pos        (6)
#define PWM_CTL_PINV1_Msk        (0x1ul << PWM_CTL_PINV1_Pos)

#define PWM_CTL_CNTEN1_Pos       (4)
#define PWM_CTL_CNTEN1_Msk       (0x1ul << PWM_CTL_CNTEN1_Pos)

#define PWM_CTL_PINV0_Pos        (2)
#define PWM_CTL_PINV0_Msk        (0x1ul << PWM_CTL_PINV0_Pos)

#define PWM_CTL_CNTEN0_Pos       (0)
#define PWM_CTL_CNTEN0_Msk       (0x1ul << PWM_CTL_CNTEN0_Pos)

#define PWM_PERIOD0_PERIOD0_Pos  (0)
#define PWM_PERIOD0_PERIOD0_Msk  (0xfffful << PWM_PERIOD0_PERIOD0_Pos)

#define PWM_PERIOD1_PERIOD1_Pos  (0)
#define PWM_PERIOD1_PERIOD1_Msk  (0xfffful << PWM_PERIOD1_PERIOD1_Pos)

#define PWM_PERIOD2_PERIOD2_Pos  (0)
#define PWM_PERIOD2_PERIOD2_Msk  (0xfffful << PWM_PERIOD2_PERIOD2_Pos)

#define PWM_PERIOD3_PERIOD3_Pos  (0)
#define PWM_PERIOD3_PERIOD3_Msk  (0xfffful << PWM_PERIOD3_PERIOD3_Pos)

#define PWM_PERIOD4_PERIOD4_Pos  (0)
#define PWM_PERIOD4_PERIOD4_Msk  (0xfffful << PWM_PERIOD4_PERIOD4_Pos)

#define PWM_PERIOD5_PERIOD5_Pos  (0)
#define PWM_PERIOD5_PERIOD5_Msk  (0xfffful << PWM_PERIOD5_PERIOD5_Pos)

#define PWM_CMPDAT0_CMP0_Pos     (0)
#define PWM_CMPDAT0_CMP0_Msk     (0xfffful << PWM_CMPDAT0_CMP0_Pos)

#define PWM_CMPDAT1_CMP1_Pos     (0)
#define PWM_CMPDAT1_CMP1_Msk     (0xfffful << PWM_CMPDAT1_CMP1_Pos)

#define PWM_CMPDAT2_CMP2_Pos     (0)
#define PWM_CMPDAT2_CMP2_Msk     (0xfffful << PWM_CMPDAT2_CMP2_Pos)

#define PWM_CMPDAT3_CMP3_Pos     (0)
#define PWM_CMPDAT3_CMP3_Msk     (0xfffful << PWM_CMPDAT3_CMP3_Pos)

#define PWM_CMPDAT4_CMP4_Pos     (0)
#define PWM_CMPDAT4_CMP4_Msk     (0xfffful << PWM_CMPDAT4_CMP4_Pos)

#define PWM_CMPDAT5_CMP5_Pos     (0)
#define PWM_CMPDAT5_CMP5_Msk     (0xfffful << PWM_CMPDAT5_CMP5_Pos)

#define PWM_CTL2_GROUPEN_Pos     (30)
#define PWM_CTL2_GROUPEN_Msk     (0x1ul << PWM_CTL2_GROUPEN_Pos)

#define PWM_CTL2_MODE_Pos        (28)
#define PWM_CTL2_MODE_Msk        (0x3ul << PWM_CTL2_MODE_Pos)

#define PWM_CTL2_DTCNT45_Pos     (26)
#define PWM_CTL2_DTCNT45_Msk     (0x1ul << PWM_CTL2_DTCNT45_Pos)

#define PWM_CTL2_DTCNT23_Pos     (25)
#define PWM_CTL2_DTCNT23_Msk     (0x1ul << PWM_CTL2_DTCNT23_Pos)

#define PWM_CTL2_DTCNT01_Pos     (24)
#define PWM_CTL2_DTCNT01_Msk     (0x1ul << PWM_CTL2_DTCNT01_Pos)

#define PWM_FLAG_PFn_Pos         (16)
#define PWM_FLAG_PFn_Msk         (0x3ful << PWM_FLAG_PFn_Pos)

#define PWM_FLAG_CMPDFn_Pos      (8)
#define PWM_FLAG_CMPDFn_Msk      (0x3ful << PWM_FLAG_CMPDFn_Pos)

#define PWM_INTEN_PIENn_Pos      (16)
#define PWM_INTEN_PIENn_Msk      (0x3ful << PWM_INTEN_PIENn_Pos)

#define PWM_INTEN_CMPDIENn_Pos   (8)
#define PWM_INTEN_CMPDIENn_Msk   (0x3ful << PWM_INTEN_CMPDIENn_Pos)

#define PWM_INTSTS_PIFn_Pos      (16)
#define PWM_INTSTS_PIFn_Msk      (0xfful << PWM_INTSTS_PIFn_Pos)

#define PWM_INTSTS_CMPDIFn_Pos   (8)
#define PWM_INTSTS_CMPDIFn_Msk   (0x3ful << PWM_INTSTS_CMPDIFn_Pos)

#define PWM_POEN_POENn_Pos       (0)
#define PWM_POEN_POENn_Msk       (0x3ful << PWM_POEN_POENn_Pos)

#define PWM_DTCTL_DTI45_Pos      (16)
#define PWM_DTCTL_DTI45_Msk      (0xfful << PWM_DTCTL_DTI45_Pos)

#define PWM_DTCTL_DTI23_Pos      (8)
#define PWM_DTCTL_DTI23_Msk      (0xfful << PWM_DTCTL_DTI23_Pos)

#define PWM_DTCTL_DTI01_Pos      (0)
#define PWM_DTCTL_DTI01_Msk      (0xfful << PWM_DTCTL_DTI01_Pos)

#define PWM_CLKPSC_H_CLKPSC5_Pos (16)
#define PWM_CLKPSC_H_CLKPSC5_Msk (0xfful << PWM_CLKPSC_H_CLKPSC5_Pos)

#define PWM_CLKPSC_H_CLKPSC4_Pos (8)
#define PWM_CLKPSC_H_CLKPSC4_Msk (0xfful << PWM_CLKPSC_H_CLKPSC4_Pos)

#define PWM_CLKPSC_H_CLKPSC3_Pos (0)
#define PWM_CLKPSC_H_CLKPSC3_Msk (0xfful << PWM_CLKPSC_H_CLKPSC3_Pos)

/**@}*/ /* PWM_CONST */
/**@}*/ /* end of PWM register group */




/*---------------------- Flash Memory Controller -------------------------*/
/**
    @addtogroup OTP Memory Controller (OTP)
    Memory Mapped Structure for OTP Controller
@{ */

typedef struct {
	__IO uint32_t MODE_CTL;
	__IO uint32_t READ_PROG_CTL;
	__IO uint32_t BYTE_ADDR;
	__IO uint32_t ADDR_BIT;
	__IO uint32_t TIMING_CTL;
	__IO uint32_t OPERATE_DATA_0;
	__IO uint32_t RESERVED0;
	__IO uint32_t RESERVED1;
	__IO uint32_t RESERVED2;
	__IO uint32_t OPERATE_TRG;
} OTP_T;

// MODE_CTL
#define MODE_CTL_WORK_MODE_Pos              (0)
#define MODE_CTL_WORK_MODE_Msk              (0x3ul << MODE_CTL_WORK_MODE_Pos)

// READ_PROG_CTL
#define READ_PROG_CTL_DMA_VPP_SAMP_FLAG_Pos (16)
#define READ_PROG_CTL_DMA_VPP_SAMP_FLAG_Msk (0x1ul << READ_PROG_CTL_DMA_VPP_SAMP_FLAG_Pos)

#define READ_PROG_CTL_PROG_REDUNDANCY_Pos   (12)
#define READ_PROG_CTL_PROG_REDUNDANCY_Msk   (0x1ul << READ_PROG_CTL_PROG_REDUNDANCY_Pos)

#define READ_PROG_CTL_PTM_Pos               (4)
#define READ_PROG_CTL_PTM_Msk               (0x1ful << READ_PROG_CTL_PTM_Pos)

#define READ_PROG_CTL_ECC_DISABLE_Pos       (0)
#define READ_PROG_CTL_ECC_DISABLE_Msk       (0x1ul << READ_PROG_CTL_ECC_DISABLE_Pos)

// BYTE_ADDR
#define BYTE_ADDR_OPERATE_ADDR_Pos          (0)
#define BYTE_ADDR_OPERATE_ADDR_Msk          (0x3ffful << BYTE_ADDR_OPERATE_ADDR_Pos)

// ADDR_BIT
#define ADDR_BIT_ADDR_BIT_Pos               (0)
#define ADDR_BIT_ADDR_BIT_Msk               (0x1ful << ADDR_BIT_ADDR_BIT_Pos)

// TIMING_CTL
#define TIMING_CTL_PROG_TIMING_STEP_Pos     (0)
#define TIMING_CTL_PROG_TIMING_STEP_Msk     (0xful << TIMING_CTL_PROG_TIMING_STEP_Pos)

// OPERATE_DATA_0
#define OPERATE_DATA_0_DATA_0_Pos           (0)
#define OPERATE_DATA_0_DATA_0_Msk           (0xfffffffful << OPERATE_DATA_0_DATA_0_Pos)

// OPERATE_TRG
#define OPERATE_TRG_TRG_Pos                 (0)
#define OPERATE_TRG_TRG_Msk                 (0x1ul << OPERATE_TRG_TRG_Pos)

/**@}*/ /* OTP_CONST */
/**@}*/ /* end of OTP register group */


typedef struct 
{
    __IO uint32_t X_CACHE_EN;
	__IO uint32_t X_RESVR;
	__IO uint32_t X_CACHE_INI;
}CR_T;


/*---------------------- Serial Peripheral Interface Controller -------------------------*/
/**
    @addtogroup SPI Serial Peripheral Interface Controller(SPI)
    Memory Mapped Structure for SPI Controller
@{ */

typedef struct {
	__IO uint32_t SSPCR0;
	__IO uint32_t SSPCR1;
	__IO uint32_t SSPDR;
	__I uint32_t SSPSR;
	__IO uint32_t SSPCPSR;
	__IO uint32_t SSPIMSC;
	__IO uint32_t SSPRIS;
	__I uint32_t SSPMIS;
	__IO uint32_t SSPICR;
} SPI_T;


#define SSPCR0_SCR_Pos      (8)
#define SSPCR0_SCR_Msk      (0xfful << SSPCR0_SCR_Pos)

#define SSPCR0_SPH_Pos      (7)
#define SSPCR0_SPH_Msk      (0x1ul << SSPCR0_SPH_Pos)

#define SSPCR0_SPO_Pos      (6)
#define SSPCR0_SPO_Msk      (0x1ul << SSPCR0_SPO_Pos)

#define SSPCR0_DSS_Pos      (0)
#define SSPCR0_DSS_Msk      (0xful << SSPCR0_DSS_Pos)

#define SSPCR1_SOD_Pos      (3)
#define SSPCR1_SOD_Msk      (0x1ul << SSPCR1_SOD_Pos)

#define SSPCR1_MS_Pos       (2)
#define SSPCR1_MS_Msk       (0x1ul << SSPCR1_MS_Pos)

#define SSPCR1_SSE_Pos      (1)
#define SSPCR1_SSE_Msk      (0x1ul << SSPCR1_SSE_Pos)

#define SSPCR1_LBM_Pos      (0)
#define SSPCR1_LBM_Msk      (0x1ul << SSPCR1_LBM_Pos)

#define SSPDR_DATA_Pos      (0)
#define SSPDR_DATA_Msk      (0xfful << SSPDR_DATA_Pos)

#define SSPSR_BSY_Pos       (4)
#define SSPSR_BSY_Msk       (0x1ul << SSPSR_BSY_Pos)

#define SSPSR_RFF_Pos       (3)
#define SSPSR_RFF_Msk       (0x1ul << SSPSR_RFF_Pos)

#define SSPSR_RNE_Pos       (2)
#define SSPSR_RNE_Msk       (0x1ul << SSPSR_RNE_Pos)

#define SSPSR_TNF_Pos       (1)
#define SSPSR_TNF_Msk       (0x1ul << SSPSR_TNF_Pos)

#define SSPSR_TFE_Pos       (0)
#define SSPSR_TFE_Msk       (0x1ul << SSPSR_TFE_Pos)

#define SSPCPSR_CPSDVSR_Pos (0)
#define SSPCPSR_CPSDVSR_Msk (0xfful << SSPCPSR_CPSDVSR_Pos)

#define SSPIMSC_TXIM_Pos    (3)
#define SSPIMSC_TXIM_Msk    (0x1ul << SSPIMSC_TXIM_Pos)

#define SSPIMSC_RXIM_Pos    (2)
#define SSPIMSC_RXIM_Msk    (0x1ul << SSPIMSC_RXIM_Pos)

#define SSPIMSC_RTIM_Pos    (1)
#define SSPIMSC_RTIM_Msk    (0x1ul << SSPIMSC_RTIM_Pos)

#define SSPIMSC_RORIM_Pos   (0)
#define SSPIMSC_RORIM_Msk   (0x1ul << SSPIMSC_RORIM_Pos)

#define SSPRIS_TXRIS_Pos    (3)
#define SSPRIS_TXRIS_Msk    (0x1ul << SSPRIS_TXRIS_Pos)

#define SSPRIS_RXRIS_Pos    (2)
#define SSPRIS_RXRIS_Msk    (0x1ul << SSPRIS_RXRIS_Pos)

#define SSPRIS_RTRIS_Pos    (1)
#define SSPRIS_RTRIS_Msk    (0x1ul << SSPRIS_RTRIS_Pos)

#define SSPRIS_RORRIS_Pos   (0)
#define SSPRIS_RORRIS_Msk   (0x1ul << SSPRIS_RORRIS_Pos)

#define SSPMIS_TXMIS_Pos    (3)
#define SSPMIS_TXMIS_Msk    (0x1ul << SSPMIS_TXMIS_Pos)

#define SSPMIS_RXMIS_Pos    (2)
#define SSPMIS_RXMIS_Msk    (0x1ul << SSPMIS_RXMIS_Pos)

#define SSPMIS_RTMIS_Pos    (1)
#define SSPMIS_RTMIS_Msk    (0x1ul << SSPMIS_RTMIS_Pos)

#define SSPMIS_RORMIS_Pos   (0)
#define SSPMIS_RORMIS_Msk   (0x1ul << SSPMIS_RORMIS_Pos)

#define SSPICR_RTIC_Pos     (1)
#define SSPICR_RTIC_Msk     (0x1ul << SSPICR_RTIC_Pos)

#define SSPICR_RORIC_Pos    (0)
#define SSPICR_RORIC_Msk    (0x1ul << SSPICR_RORIC_Pos)


/**@}*/ /* SPI_CONST */
/**@}*/ /* end of SPI register group */







/*---------------------- Universal Asynchronous Receiver/Transmitter Controller -------------------------*/
/**
    @addtogroup UART Universal Asynchronous Receiver/Transmitter Controller(UART)
    Memory Mapped Structure for UART Controller
@{ */

typedef struct {
	__I uint32_t UART_RXDATA;
	__IO uint32_t UART_TXDATA;
	__IO uint32_t UART_CTL;
	__I uint32_t UART_STATUS;
	__IO uint32_t UART_BCNT;
	__IO uint32_t TIMEOUT_CTL;
} UART_T;


#define UART_RXDATA_RX_DATA_Pos       (0)
#define UART_RXDATA_RX_DATA_Msk       (0xfful << UART_RXDATA_RX_DATA_Pos)

#define UART_TXDATA_TX_DATA_Pos       (0)
#define UART_TXDATA_TX_DATA_Msk       (0xfful << UART_TXDATA_TX_DATA_Pos)

#define UART_CTL_RX_TIMEOUT_INT_EN_Pos (16)
#define UART_CTL_RX_TIMEOUT_INT_EN_Msk (0x1ul << UART_CTL_RX_TIMEOUT_INT_EN_Pos)

#define UART_CTL_RX_NE_MASK_Pos       (12)
#define UART_CTL_RX_NE_MASK_Msk       (0x3ul << UART_CTL_RX_NE_MASK_Pos)

#define UART_CTL_RX_NE_INT_EN_Pos     (11)
#define UART_CTL_RX_NE_INT_EN_Msk     (0x1ul << UART_CTL_RX_NE_INT_EN_Pos)

#define UART_CTL_RX_OV_INT_EN_Pos     (10)
#define UART_CTL_RX_OV_INT_EN_Msk     (0x1ul << UART_CTL_RX_OV_INT_EN_Pos)

#define UART_CTL_RX_ERR_INT_EN_Pos    (9)
#define UART_CTL_RX_ERR_INT_EN_Msk    (0x1ul << UART_CTL_RX_ERR_INT_EN_Pos)

#define UART_CTL_RX_EN_Pos            (8)
#define UART_CTL_RX_EN_Msk            (0x1ul << UART_CTL_RX_EN_Pos)

#define UART_CTL_TX_NF_MASK_Pos       (2)
#define UART_CTL_TX_NF_MASK_Msk       (0x3ul << UART_CTL_TX_NF_MASK_Pos)

#define UART_CTL_TX_NF_INT_EN_Pos     (1)
#define UART_CTL_TX_NF_INT_EN_Msk     (0x1ul << UART_CTL_TX_NF_INT_EN_Pos)

#define UART_CTL_TX_EN_Pos            (0)
#define UART_CTL_TX_EN_Msk            (0x1ul << UART_CTL_TX_EN_Pos)

#define UART_STATUS_TX_NF_FLAG_Pos    (22)
#define UART_STATUS_TX_NF_FLAG_Msk    (0x1ul << UART_STATUS_TX_NF_FLAG_Pos)

#define UART_STATUS_TX_FIFO_NUM_Pos   (19)
#define UART_STATUS_TX_FIFO_NUM_Msk   (0x7ul << UART_STATUS_TX_FIFO_NUM_Pos)

#define UART_STATUS_TX_FULL_FLAG_Pos  (18)
#define UART_STATUS_TX_FULL_FLAG_Msk  (0x1ul << UART_STATUS_TX_FULL_FLAG_Pos)

#define UART_STATUS_TX_EMPTY_FLAG_Pos (17)
#define UART_STATUS_TX_EMPTY_FLAG_Msk (0x1ul << UART_STATUS_TX_EMPTY_FLAG_Pos)

#define UART_STATUS_TX_BUSY_Pos       (16)
#define UART_STATUS_TX_BUSY_Msk       (0x1ul << UART_STATUS_TX_BUSY_Pos)

#define UART_STATUS_RX_TIMEOUT_FLAG_Pos (9)
#define UART_STATUS_RX_TIMEOUT_FLAG_Msk (0x1ul << UART_STATUS_RX_TIMEOUT_FLAG_Pos)

#define UART_STATUS_RX_NE_FLAG_Pos    (8)
#define UART_STATUS_RX_NE_FLAG_Msk    (0x1ul << UART_STATUS_RX_NE_FLAG_Pos)

#define UART_STATUS_RX_FIFO_NUM_Pos   (5)
#define UART_STATUS_RX_FIFO_NUM_Msk   (0x7ul << UART_STATUS_RX_FIFO_NUM_Pos)

#define UART_STATUS_RX_OV_FLAG_Pos    (4)
#define UART_STATUS_RX_OV_FLAG_Msk    (0x1ul << UART_STATUS_RX_OV_FLAG_Pos)

#define UART_STATUS_RX_ERR_FLAG_Pos   (3)
#define UART_STATUS_RX_ERR_FLAG_Msk   (0x1ul << UART_STATUS_RX_ERR_FLAG_Pos)

#define UART_STATUS_RX_FULL_FLAG_Pos  (2)
#define UART_STATUS_RX_FULL_FLAG_Msk  (0x1ul << UART_STATUS_RX_FULL_FLAG_Pos)

#define UART_STATUS_RX_EMPTY_FLAG_Pos (1)
#define UART_STATUS_RX_EMPTY_FLAG_Msk (0x1ul << UART_STATUS_RX_EMPTY_FLAG_Pos)

#define UART_STATUS_RX_BUSY_Pos       (0)
#define UART_STATUS_RX_BUSY_Msk       (0x1ul << UART_STATUS_RX_BUSY_Pos)

#define UART_BCNT_BCNT_Pos            (0)
#define UART_BCNT_BCNT_Msk            (0xfffful << UART_BCNT_BCNT_Pos)

#define TIMEOUT_CTL_TIMEOUT_EN_Pos    (20)
#define TIMEOUT_CTL_TIMEOUT_EN_Msk    (0x1ul << TIMEOUT_CTL_TIMEOUT_EN_Pos)

#define TIMEOUT_CTL_TIMEOUT_CNT_TOTAL_Pos (0)
#define TIMEOUT_CTL_TIMEOUT_CNT_TOTAL_Msk (0xffffful << TIMEOUT_CTL_TIMEOUT_CNT_TOTAL_Pos)

/**@}*/ /* UART_CONST */
/**@}*/ /* end of UART register group */


/*---------------------- Inter-IC Bus Controller -------------------------*/
/**
    @addtogroup I2C Inter-IC Bus Controller(I2C)
    Memory Mapped Structure for I2C Controller
@{ */

/**
  * @brief Inter-integrated Circuit Interface
  */
typedef struct
{
    __IO uint32_t CON               ;//(I2C_BASE + 0x000)
    __IO uint32_t DATACMD           ;//(I2C_BASE + 0x004)
    __IO uint32_t STATUS            ;//(I2C_BASE + 0x008)
}I2C_T;

//CON
#define IIC_CON_CR2_Pos                          (7)
#define IIC_CON_CR2_Msk                          (0x1ul << IIC_CON_CR2_Pos)
#define IIC_CON_IIC_EN_Pos                       (6)
#define IIC_CON_IIC_EN_Msk                       (0x1ul << IIC_CON_IIC_EN_Pos)
#define IIC_CON_START_FLAG_Pos                   (5)
#define IIC_CON_START_FLAG_Msk                   (0x1ul << IIC_CON_START_FLAG_Pos)
#define IIC_CON_STOP_FLAG_Pos                    (4)
#define IIC_CON_STOP_FLAG_Msk                    (0x1ul << IIC_CON_STOP_FLAG_Pos)
#define IIC_CON_INT_FLAG_Pos                     (3)
#define IIC_CON_INT_FLAG_Msk                     (0x1ul << IIC_CON_INT_FLAG_Pos)
#define IIC_CON_ACK_EN_Pos                       (2)
#define IIC_CON_ACK_EN_Msk                       (0x1ul << IIC_CON_ACK_EN_Pos)
#define IIC_CON_CR1_CR0_Pos                      (0)
#define IIC_CON_CR1_CR0_Msk                      (0x3ul << IIC_CON_CR1_CR0_Pos)

//DATACMD
#define IIC_DATACMD_Pos                          (0)
#define IIC_DATACMD_Msk                          (0xFFul << IIC_DATACMD_Pos)

//STATUS
#define IIC_STATUS_IIC_FLAG_Pos                  (0)
#define IIC_STATUS_IIC_FLAG_Msk                  (0xFFul << KS_DLY_DISABLE_Pos)


typedef struct
{
    __IO uint32_t KS_EN                     ; //0x00
    __IO uint32_t KS_IO_CFG                 ; //0x04
    __IO uint32_t KS_INT_CFG                ; //0X08
    __IO uint32_t KS_CFG                    ; //0X0C
    __I  uint32_t KS_INFO0                  ; //0X10
    __I  uint32_t KS_INFO1                  ; //0X14
}KSCAN_T;

//KS_EN	
#define KS_DLY_DISABLE_Pos                  (2)
#define KS_DLY_DISABLE_Msk                  (0x1ul << KS_DLY_DISABLE_Pos)
#define KS_WK_ENABLE_Pos                    (1)
#define KS_WK_ENABLE_Msk                    (0x1ul << KS_WK_ENABLE_Pos)
#define KS_ENABLE_Pos                       (0)
#define KS_ENABLE_Msk                       (0x1ul << KS_ENABLE_Pos)
//KS_IO_CFG	
#define KS_OUTPUT_ENABLE_Pos                (6)
#define KS_OUTPUT_ENABLE_Msk                (0x3Ful << KS_OUTPUT_ENABLE_Pos)
#define KS_INPUT_ENABLE_Pos                 (0)
#define KS_INPUT_ENABLE_Msk                 (0x3ful << KS_INPUT_ENABLE_Pos)
//KS_INT_CFG  
#define KS_WK_FLAG_Pos                      (3)
#define KS_WK_FLAG_Msk                      (0x1ul << KS_WK_FLAG_Pos)
#define KS_INT_FLAG_Pos                     (2)
#define KS_INT_FLAG_Msk                     (0x1ul << KS_INT_FLAG_Pos)
#define KS_FLAG_Pos                         (1)
#define KS_FLAG_Msk                         (0x1ul << KS_FLAG_Pos)
#define KS_INT_ENABLE_Pos                   (0)
#define KS_INT_ENABLE_Msk                   (0x1ul << KS_INT_ENABLE_Pos)
//KS_CFG        
#define KS_POLARITY_VALID_HIGH_Pos          (13)
#define KS_POLARITY_VALID_HIGH_Msk          (0x1ul << KS_POLARITY_VALID_HIGH_Pos)
#define KS_STATUS_CLEAR_Pos                 (12)
#define KS_STATUS_CLEAR_Msk                 (0x1ul << KS_STATUS_CLEAR_Pos)
#define KS_SCAN_INTERVAL_Pos                (8)
#define KS_SCAN_INTERVAL_Msk                (0xFul << KS_SCAN_INTERVAL_Pos)
#define KS_ROW_INTERVAL_Pos                 (4)
#define KS_ROW_INTERVAL_Msk                 (0xFul << KS_ROW_INTERVAL_Pos)
#define KS_DEBOUNCE_TIME_Pos                (0)
#define KS_DEBOUNCE_TIME_Msk                (0xFul << KS_DEBOUNCE_TIME_Pos)
//KS_INFO0


/*ANACTL*/
typedef struct {
	__IO uint32_t LP_CTRL0;
	__IO uint32_t LP_FLAG_CTRL;
	__IO uint32_t SLPTMR_CTRL;
	__I  uint32_t SLPTMR_CURR;
	__IO uint32_t SLPTMR_CMP0;
	__IO uint32_t SLPTMR_CMP1;
	__IO uint32_t SLPTMR_LOAD;
	__IO uint32_t CPU_ADDR_REMAP_CTRL;
	__IO uint32_t BLD_CTRL;
	__IO uint32_t SYS_CLK_CTRL;
	__IO uint32_t RCL_CTRL;
	__IO uint32_t RCH_CTRL;
	__IO uint32_t XTL_CTRL;
	__IO uint32_t XTH_CTRL;
	__IO uint32_t DPLL_CTRL;
	__IO uint32_t ANA_MISC;
	__IO uint32_t ANA_HPLDO;
	__IO uint32_t ANA_LPLDO;
	__IO uint32_t ANA_ANALDO;
	__IO uint32_t ANA_RESERVED;
} PMU_T;

// LP_CTRL0
#define LP_CTRL0_DLY3_TIME_Pos                    (20)
#define LP_CTRL0_DLY3_TIME_Msk                    (0x1ful << LP_CTRL0_DLY3_TIME_Pos)

#define LP_CTRL0_DLY2_TIME_Pos                    (16)
#define LP_CTRL0_DLY2_TIME_Msk                    (0xful << LP_CTRL0_DLY2_TIME_Pos)

#define LP_CTRL0_SW1_LP_CTRL_Pos                  (15)
#define LP_CTRL0_SW1_LP_CTRL_Msk                  (0x1ul << LP_CTRL0_SW1_LP_CTRL_Pos)

#define LP_CTRL0_SW0_LP_CTRL_Pos                  (14)
#define LP_CTRL0_SW0_LP_CTRL_Msk                  (0x1ul << LP_CTRL0_SW0_LP_CTRL_Pos)

#define LP_CTRL0_SLOW_CLK_LP_CTRL_Pos             (13)
#define LP_CTRL0_SLOW_CLK_LP_CTRL_Msk             (0x1ul << LP_CTRL0_SLOW_CLK_LP_CTRL_Pos)

#define LP_CTRL0_HPLDO_LP_CTRL_Pos                (12)
#define LP_CTRL0_HPLDO_LP_CTRL_Msk                (0x1ul << LP_CTRL0_HPLDO_LP_CTRL_Pos)

#define LP_CTRL0_OTP_LP_CTRL_Pos                  (10)
#define LP_CTRL0_OTP_LP_CTRL_Msk                  (0x3ul << LP_CTRL0_OTP_LP_CTRL_Pos)

#define LP_CTRL0_DEBUGREQ_LP_EN_Pos               (5)
#define LP_CTRL0_DEBUGREQ_LP_EN_Msk               (0x1ul << LP_CTRL0_DEBUGREQ_LP_EN_Pos)

#define LP_CTRL0_PMU_ISOLATE_EN_Pos               (4)
#define LP_CTRL0_PMU_ISOLATE_EN_Msk               (0x1ul << LP_CTRL0_PMU_ISOLATE_EN_Pos)

#define LP_CTRL0_CPU_LP_RET_EN_Pos                (3)
#define LP_CTRL0_CPU_LP_RET_EN_Msk                (0x1ul << LP_CTRL0_CPU_LP_RET_EN_Pos)

#define LP_CTRL0_LP_INT_EN_Pos                    (2)
#define LP_CTRL0_LP_INT_EN_Msk                    (0x1ul << LP_CTRL0_LP_INT_EN_Pos)

#define LP_CTRL0_SLEEP_MODE_Pos                   (0)
#define LP_CTRL0_SLEEP_MODE_Msk                   (0x3ul << LP_CTRL0_SLEEP_MODE_Pos)

// LP_FLAG_CTRL
#define LP_FLAG_CTRL_INT_FLAG_MASK                (0x430007ul)

#define LP_FLAG_CTRL_BODOUT_Pos                   (23)
#define LP_FLAG_CTRL_BODOUT_Msk                   (0x1ul << LP_FLAG_CTRL_BODOUT_Pos)

#define LP_FLAG_CTRL_BODIF_Pos                    (22)
#define LP_FLAG_CTRL_BODIF_Msk                    (0x1ul << LP_FLAG_CTRL_BODIF_Pos)

#define LP_FLAG_CTRL_ACT32K_UP_DONE_Pos           (20)
#define LP_FLAG_CTRL_ACT32K_UP_DONE_Msk           (0x1ul << LP_FLAG_CTRL_ACT32K_UP_DONE_Pos)

#define LP_FLAG_CTRL_SLPTMR_FLG1_Pos              (17)
#define LP_FLAG_CTRL_SLPTMR_FLG1_Msk              (0x1ul << LP_FLAG_CTRL_SLPTMR_FLG1_Pos)

#define LP_FLAG_CTRL_SLPTMR_FLG0_Pos              (16)
#define LP_FLAG_CTRL_SLPTMR_FLG0_Msk              (0x1ul << LP_FLAG_CTRL_SLPTMR_FLG0_Pos)

#define LP_FLAG_CTRL_STANDBY_M0_FLG_Pos           (2)
#define LP_FLAG_CTRL_STANDBY_M0_FLG_Msk           (0x1ul << LP_FLAG_CTRL_STANDBY_M0_FLG_Pos)

#define LP_FLAG_CTRL_STANDBY_M1_FLG_Pos           (1)
#define LP_FLAG_CTRL_STANDBY_M1_FLG_Msk           (0x1ul << LP_FLAG_CTRL_STANDBY_M1_FLG_Pos)

#define LP_FLAG_CTRL_DEEPSLEEP_FLG_Pos            (0)
#define LP_FLAG_CTRL_DEEPSLEEP_FLG_Msk            (0x1ul << LP_FLAG_CTRL_DEEPSLEEP_FLG_Pos)

// SLPTMR_CTRL
#define SLPTMR_CTRL_SLPTMR_LOAD_TRG_Pos           (7)
#define SLPTMR_CTRL_SLPTMR_LOAD_TRG_Msk           (0x1ul << SLPTMR_CTRL_SLPTMR_LOAD_TRG_Pos)

#define SLPTMR_CTRL_SLPTMR_CMP1_EN_Pos            (4)
#define SLPTMR_CTRL_SLPTMR_CMP1_EN_Msk            (0x1ul << SLPTMR_CTRL_SLPTMR_CMP1_EN_Pos)

#define SLPTMR_CTRL_SLPTMR_CMP0_EN_Pos            (3)
#define SLPTMR_CTRL_SLPTMR_CMP0_EN_Msk            (0x1ul << SLPTMR_CTRL_SLPTMR_CMP0_EN_Pos)

#define SLPTMR_CTRL_SLPTMR_WK_EN_Pos              (2)
#define SLPTMR_CTRL_SLPTMR_WK_EN_Msk              (0x1ul << SLPTMR_CTRL_SLPTMR_WK_EN_Pos)

#define SLPTMR_CTRL_SLPTMR_INT_EN_Pos             (1)
#define SLPTMR_CTRL_SLPTMR_INT_EN_Msk             (0x1ul << SLPTMR_CTRL_SLPTMR_INT_EN_Pos)

#define SLPTMR_CTRL_SLPTMR_EN_Pos                 (0)
#define SLPTMR_CTRL_SLPTMR_EN_Msk                 (0x1ul << SLPTMR_CTRL_SLPTMR_EN_Pos)

// SLPTMR_CURR
#define SLPTMR_CURR_CURR_VAL_Pos                  (0)
#define SLPTMR_CURR_CURR_VAL_Msk                  (0xfffffffful << SLPTMR_CURR_CURR_VAL_Pos)

// SLPTMR_CMP0
#define SLPTMR_CMP0_CMP_VAL_Pos                   (0)
#define SLPTMR_CMP0_CMP_VAL_Msk                   (0xfffffffful << SLPTMR_CMP0_CMP_VAL_Pos)

// SLPTMR_CMP1
#define SLPTMR_CMP1_CMP_VAL_Pos                   (0)
#define SLPTMR_CMP1_CMP_VAL_Msk                   (0xfffffffful << SLPTMR_CMP1_CMP_VAL_Pos)

// SLPTMR_LOAD
#define SLPTMR_LOAD_LOAD_VAL_Pos                  (0)
#define SLPTMR_LOAD_LOAD_VAL_Msk                  (0xfffffffful << SLPTMR_LOAD_LOAD_VAL_Pos)

// CPU_ADDR_REMAP_CTRL
#define CPU_ADDR_REMAP_CTRL_CPU_ADDR_REMAP_EN_Pos (31)
#define CPU_ADDR_REMAP_CTRL_CPU_ADDR_REMAP_EN_Msk (0x1ul << CPU_ADDR_REMAP_CTRL_CPU_ADDR_REMAP_EN_Pos)

#define CPU_ADDR_REMAP_CTRL_CPU_REMAP_ADDR_Pos    (0)
#define CPU_ADDR_REMAP_CTRL_CPU_REMAP_ADDR_Msk    (0xfffffful << CPU_ADDR_REMAP_CTRL_CPU_REMAP_ADDR_Pos)

// BLD_CTRL
#define BLD_CTRL_LVR_DB_SEL_Pos                   (23)
#define BLD_CTRL_LVR_DB_SEL_Msk                   (0x3ful << BLD_CTRL_LVR_DB_SEL_Pos)

#define BLD_CTRL_PMU_LVR_TEST_EN_AON_Pos          (22)
#define BLD_CTRL_PMU_LVR_TEST_EN_AON_Msk          (0x1ul << BLD_CTRL_PMU_LVR_TEST_EN_AON_Pos)

#define BLD_CTRL_PMU_LVR_EN_AON_Pos               (21)
#define BLD_CTRL_PMU_LVR_EN_AON_Msk               (0x1ul << BLD_CTRL_PMU_LVR_EN_AON_Pos)

#define BLD_CTRL_PMU_BOD_VSEL_V2_AON_Pos          (16)
#define BLD_CTRL_PMU_BOD_VSEL_V2_AON_Msk          (0x1ful << BLD_CTRL_PMU_BOD_VSEL_V2_AON_Pos)

#define BLD_CTRL_BOD_DB_SEL_Pos                   (10)
#define BLD_CTRL_BOD_DB_SEL_Msk                   (0x3ful << BLD_CTRL_BOD_DB_SEL_Pos)

#define BLD_CTRL_BOD_RST_EN_Pos                   (9)
#define BLD_CTRL_BOD_RST_EN_Msk                   (0x1ul << BLD_CTRL_BOD_RST_EN_Pos)

#define BLD_CTRL_PMU_BOD_VREF_TRIM_AON_Pos        (6)
#define BLD_CTRL_PMU_BOD_VREF_TRIM_AON_Msk        (0x7ul << BLD_CTRL_PMU_BOD_VREF_TRIM_AON_Pos)

#define BLD_CTRL_PMU_BOD_VSEL_AON_Pos             (2)
#define BLD_CTRL_PMU_BOD_VSEL_AON_Msk             (0x7ul << BLD_CTRL_PMU_BOD_VSEL_AON_Pos)

#define BLD_CTRL_PMU_BOD_TEST_EN_AON_Pos          (1)
#define BLD_CTRL_PMU_BOD_TEST_EN_AON_Msk          (0x1ul << BLD_CTRL_PMU_BOD_TEST_EN_AON_Pos)

#define BLD_CTRL_PMU_BOD_EN_AON_Pos               (0)
#define BLD_CTRL_PMU_BOD_EN_AON_Msk               (0x1ul << BLD_CTRL_PMU_BOD_EN_AON_Pos)

// SYS_CLK_CTRL
#define SYS_CLK_CTRL_OTP_TR_TRIM_Pos              (16)
#define SYS_CLK_CTRL_OTP_TR_TRIM_Msk              (0xfffful << SYS_CLK_CTRL_OTP_TR_TRIM_Pos)

#define SYS_CLK_CTRL_OTP_CLK_HIGH_CYCLE_Pos       (13)
#define SYS_CLK_CTRL_OTP_CLK_HIGH_CYCLE_Msk       (0x7ul << SYS_CLK_CTRL_OTP_CLK_HIGH_CYCLE_Pos)

#define SYS_CLK_CTRL_OTP_CLK_LOW_CYCLE_Pos        (10)
#define SYS_CLK_CTRL_OTP_CLK_LOW_CYCLE_Msk        (0x7ul << SYS_CLK_CTRL_OTP_CLK_LOW_CYCLE_Pos)

#define SYS_CLK_CTRL_OTP_SW_UP_TR_EN_Pos          (9)
#define SYS_CLK_CTRL_OTP_SW_UP_TR_EN_Msk          (0x1ul << SYS_CLK_CTRL_OTP_SW_UP_TR_EN_Pos)

#define SYS_CLK_CTRL_OTP_WR_EN_Pos                (8)
#define SYS_CLK_CTRL_OTP_WR_EN_Msk                (0x1ul << SYS_CLK_CTRL_OTP_WR_EN_Pos)

#define SYS_CLK_CTRL_OTP_SW_UP_CYCLE_EN_Pos       (7)
#define SYS_CLK_CTRL_OTP_SW_UP_CYCLE_EN_Msk       (0x1ul << SYS_CLK_CTRL_OTP_SW_UP_CYCLE_EN_Pos)

#define SYS_CLK_CTRL_OTP_READ_MODE_SEL_Pos        (6)
#define SYS_CLK_CTRL_OTP_READ_MODE_SEL_Msk        (0x1ul << SYS_CLK_CTRL_OTP_READ_MODE_SEL_Pos)

#define SYS_CLK_CTRL_PAD_RST_MASK_Pos             (5)
#define SYS_CLK_CTRL_PAD_RST_MASK_Msk             (0x1ul << SYS_CLK_CTRL_PAD_RST_MASK_Pos)

#define SYS_CLK_CTRL_ACK32K_CLK_EN_Pos            (4)
#define SYS_CLK_CTRL_ACK32K_CLK_EN_Msk            (0x1ul << SYS_CLK_CTRL_ACK32K_CLK_EN_Pos)

#define SYS_CLK_CTRL_SLOW_CLK_SEL_Pos             (2)
#define SYS_CLK_CTRL_SLOW_CLK_SEL_Msk             (0x3ul << SYS_CLK_CTRL_SLOW_CLK_SEL_Pos)

#define SYS_CLK_CTRL_SYS_CLK_SEL_Pos              (0)
#define SYS_CLK_CTRL_SYS_CLK_SEL_Msk              (0x3ul << SYS_CLK_CTRL_SYS_CLK_SEL_Pos)

// RCL_CTRL
#define RCL_CTRL_CLK_RCL32K_RDY_OUT_AON_Pos       (24)
#define RCL_CTRL_CLK_RCL32K_RDY_OUT_AON_Msk       (0x1ul << RCL_CTRL_CLK_RCL32K_RDY_OUT_AON_Pos)

#define RCL_CTRL_CLK_RCL_TEMPTRIM_AON_Pos         (19)
#define RCL_CTRL_CLK_RCL_TEMPTRIM_AON_Msk         (0x3ul << RCL_CTRL_CLK_RCL_TEMPTRIM_AON_Pos)

#define RCL_CTRL_CLK_RCL_DELAY_AON_Pos            (17)
#define RCL_CTRL_CLK_RCL_DELAY_AON_Msk            (0x3ul << RCL_CTRL_CLK_RCL_DELAY_AON_Pos)

#define RCL_CTRL_CLK_RCL_FREQ_FINE_AON_Pos        (8)
#define RCL_CTRL_CLK_RCL_FREQ_FINE_AON_Msk        (0x1fful << RCL_CTRL_CLK_RCL_FREQ_FINE_AON_Pos)

#define RCL_CTRL_CLK_RCL_FREQ_COARSE_AON_Pos      (4)
#define RCL_CTRL_CLK_RCL_FREQ_COARSE_AON_Msk      (0xful << RCL_CTRL_CLK_RCL_FREQ_COARSE_AON_Pos)

#define RCL_CTRL_CLK_RCL_TEST_EN_AON_Pos          (1)
#define RCL_CTRL_CLK_RCL_TEST_EN_AON_Msk          (0x1ul << RCL_CTRL_CLK_RCL_TEST_EN_AON_Pos)

#define RCL_CTRL_CLK_RCL_EN_AON_Pos               (0)
#define RCL_CTRL_CLK_RCL_EN_AON_Msk               (0x1ul << RCL_CTRL_CLK_RCL_EN_AON_Pos)

// RCH_CTRL
#define RCH_CTRL_CLK_RCH32M_RDY_OUT_AON_Pos       (24)
#define RCH_CTRL_CLK_RCH32M_RDY_OUT_AON_Msk       (0x1ul << RCH_CTRL_CLK_RCH32M_RDY_OUT_AON_Pos)

#define RCH_CTRL_CLK_RCH_RDY_TRIM_AON_Pos         (16)
#define RCH_CTRL_CLK_RCH_RDY_TRIM_AON_Msk         (0x3ul << RCH_CTRL_CLK_RCH_RDY_TRIM_AON_Pos)

#define RCH_CTRL_CLK_RCH_FREQ_TRIM_AON_Pos        (8)
#define RCH_CTRL_CLK_RCH_FREQ_TRIM_AON_Msk        (0xfful << RCH_CTRL_CLK_RCH_FREQ_TRIM_AON_Pos)

#define RCH_CTRL_CLK_RCH_FREQ_COARSE_AON_Pos      (4)
#define RCH_CTRL_CLK_RCH_FREQ_COARSE_AON_Msk      (0x3ul << RCH_CTRL_CLK_RCH_FREQ_COARSE_AON_Pos)

#define RCH_CTRL_CLK_RCH_TEST_EN_AON_Pos          (1)
#define RCH_CTRL_CLK_RCH_TEST_EN_AON_Msk          (0x1ul << RCH_CTRL_CLK_RCH_TEST_EN_AON_Pos)

#define RCH_CTRL_CLK_RCH_EN_AON_Pos               (0)
#define RCH_CTRL_CLK_RCH_EN_AON_Msk               (0x1ul << RCH_CTRL_CLK_RCH_EN_AON_Pos)

// XTL_CTRL
#define XTL_CTRL_CLK_XTL_RDY_OUT_AON_Pos          (24)
#define XTL_CTRL_CLK_XTL_RDY_OUT_AON_Msk          (0x1ul << XTL_CTRL_CLK_XTL_RDY_OUT_AON_Pos)

#define XTL_CTRL_CLK_XTL_DELAY_AON_Pos            (16)
#define XTL_CTRL_CLK_XTL_DELAY_AON_Msk            (0x3ul << XTL_CTRL_CLK_XTL_DELAY_AON_Pos)

#define XTL_CTRL_CLK_XTL_ICORE_AON_Pos            (4)
#define XTL_CTRL_CLK_XTL_ICORE_AON_Msk            (0x7ul << XTL_CTRL_CLK_XTL_ICORE_AON_Pos)

#define XTL_CTRL_CLK_XTL_TEST_EN_AON_Pos          (1)
#define XTL_CTRL_CLK_XTL_TEST_EN_AON_Msk          (0x1ul << XTL_CTRL_CLK_XTL_TEST_EN_AON_Pos)

#define XTL_CTRL_CLK_XTL_EN_AON_Pos               (0)
#define XTL_CTRL_CLK_XTL_EN_AON_Msk               (0x1ul << XTL_CTRL_CLK_XTL_EN_AON_Pos)

// XTH_CTRL
#define XTH_CTRL_FSYNXO_CLKRDY_AON_Pos            (24)
#define XTH_CTRL_FSYNXO_CLKRDY_AON_Msk            (0x1ul << XTH_CTRL_FSYNXO_CLKRDY_AON_Pos)

#define XTH_CTRL_FSYNXO_AMPSEL_AON_Pos            (20)
#define XTH_CTRL_FSYNXO_AMPSEL_AON_Msk            (0x1ul << XTH_CTRL_FSYNXO_AMPSEL_AON_Pos)

#define XTH_CTRL_FSYNXO_CAP2_AON_Pos              (19)
#define XTH_CTRL_FSYNXO_CAP2_AON_Msk              (0x1ul << XTH_CTRL_FSYNXO_CAP2_AON_Pos)

#define XTH_CTRL_FSYNXO_HYS_AON_Pos               (17)
#define XTH_CTRL_FSYNXO_HYS_AON_Msk               (0x3ul << XTH_CTRL_FSYNXO_HYS_AON_Pos)

#define XTH_CTRL_FSYNXO_STARTUP_COUNTER_AON_Pos   (16)
#define XTH_CTRL_FSYNXO_STARTUP_COUNTER_AON_Msk   (0x1ul << XTH_CTRL_FSYNXO_STARTUP_COUNTER_AON_Pos)

#define XTH_CTRL_FSYNXO_RES_AON_Pos               (15)
#define XTH_CTRL_FSYNXO_RES_AON_Msk               (0x1ul << XTH_CTRL_FSYNXO_RES_AON_Pos)

#define XTH_CTRL_FSYNXO_BUFEN_AON_Pos             (14)
#define XTH_CTRL_FSYNXO_BUFEN_AON_Msk             (0x1ul << XTH_CTRL_FSYNXO_BUFEN_AON_Pos)

#define XTH_CTRL_FSYNXO_ICORE_AON_Pos             (12)
#define XTH_CTRL_FSYNXO_ICORE_AON_Msk             (0x3ul << XTH_CTRL_FSYNXO_ICORE_AON_Pos)

#define XTH_CTRL_FSYNXO_FB_AON_Pos                (11)
#define XTH_CTRL_FSYNXO_FB_AON_Msk                (0x1ul << XTH_CTRL_FSYNXO_FB_AON_Pos)

#define XTH_CTRL_FSYNXO_FAST_DLY_AON_Pos          (10)
#define XTH_CTRL_FSYNXO_FAST_DLY_AON_Msk          (0x1ul << XTH_CTRL_FSYNXO_FAST_DLY_AON_Pos)

#define XTH_CTRL_FSYNXO_CAPSEL_AON_Pos            (4)
#define XTH_CTRL_FSYNXO_CAPSEL_AON_Msk            (0x3ful << XTH_CTRL_FSYNXO_CAPSEL_AON_Pos)

#define XTH_CTRL_EN_FSYNXO_DEGLITCH_AON_Pos       (3)
#define XTH_CTRL_EN_FSYNXO_DEGLITCH_AON_Msk       (0x1ul << XTH_CTRL_EN_FSYNXO_DEGLITCH_AON_Pos)

#define XTH_CTRL_FSYNXO_STARTUP_FAST_AON_Pos      (2)
#define XTH_CTRL_FSYNXO_STARTUP_FAST_AON_Msk      (0x1ul << XTH_CTRL_FSYNXO_STARTUP_FAST_AON_Pos)

#define XTH_CTRL_FSYNXO_EN_AON_Pos                (0)
#define XTH_CTRL_FSYNXO_EN_AON_Msk                (0x1ul << XTH_CTRL_FSYNXO_EN_AON_Pos)

// DPLL_CTRL
#define DPLL_CTRL_CLK_DPLL_RDY_OUT_AON_Pos        (24)
#define DPLL_CTRL_CLK_DPLL_RDY_OUT_AON_Msk        (0x1ul << DPLL_CTRL_CLK_DPLL_RDY_OUT_AON_Pos)

#define DPLL_CTRL_CLK_DPLL_DELAY_AON_Pos          (16)
#define DPLL_CTRL_CLK_DPLL_DELAY_AON_Msk          (0x3ul << DPLL_CTRL_CLK_DPLL_DELAY_AON_Pos)

#define DPLL_CTRL_CLK_DPLL_16M_MODE_AON_Pos       (13)
#define DPLL_CTRL_CLK_DPLL_16M_MODE_AON_Msk       (0x1ul << DPLL_CTRL_CLK_DPLL_16M_MODE_AON_Pos)

#define DPLL_CTRL_CLK_DPLL_VCO_ITRIM_AON_Pos      (11)
#define DPLL_CTRL_CLK_DPLL_VCO_ITRIM_AON_Msk      (0x3ul << DPLL_CTRL_CLK_DPLL_VCO_ITRIM_AON_Pos)

#define DPLL_CTRL_CLK_DPLL_KVCO_CTRL_AON_Pos      (8)
#define DPLL_CTRL_CLK_DPLL_KVCO_CTRL_AON_Msk      (0x7ul << DPLL_CTRL_CLK_DPLL_KVCO_CTRL_AON_Pos)

#define DPLL_CTRL_CLK_DPLL_ICP_CTRL_AON_Pos       (6)
#define DPLL_CTRL_CLK_DPLL_ICP_CTRL_AON_Msk       (0x3ul << DPLL_CTRL_CLK_DPLL_ICP_CTRL_AON_Pos)

#define DPLL_CTRL_CLK_DPLL_ICP_BIAS_AON_Pos       (4)
#define DPLL_CTRL_CLK_DPLL_ICP_BIAS_AON_Msk       (0x3ul << DPLL_CTRL_CLK_DPLL_ICP_BIAS_AON_Pos)

#define DPLL_CTRL_CLK_DPLL_REFCLK_SEL_AON_Pos     (3)
#define DPLL_CTRL_CLK_DPLL_REFCLK_SEL_AON_Msk     (0x1ul << DPLL_CTRL_CLK_DPLL_REFCLK_SEL_AON_Pos)

#define DPLL_CTRL_CLK_DPLL_FREQ_SEL_AON_Pos       (2)
#define DPLL_CTRL_CLK_DPLL_FREQ_SEL_AON_Msk       (0x1ul << DPLL_CTRL_CLK_DPLL_FREQ_SEL_AON_Pos)

#define DPLL_CTRL_CLK_DPLL_TEST_EN_AON_Pos        (1)
#define DPLL_CTRL_CLK_DPLL_TEST_EN_AON_Msk        (0x1ul << DPLL_CTRL_CLK_DPLL_TEST_EN_AON_Pos)

#define DPLL_CTRL_CLK_DPLL_EN_AON_Pos             (0)
#define DPLL_CTRL_CLK_DPLL_EN_AON_Msk             (0x1ul << DPLL_CTRL_CLK_DPLL_EN_AON_Pos)

// ANA_MISC
#define ANA_MISC_OTP_REF_EN_AON_Pos               (31)
#define ANA_MISC_OTP_REF_EN_AON_Msk               (0x1ul << ANA_MISC_OTP_REF_EN_AON_Pos)

#define ANA_MISC_OTP_CE_AON_Pos                   (30)
#define ANA_MISC_OTP_CE_AON_Msk                   (0x1ul << ANA_MISC_OTP_CE_AON_Pos)

#define ANA_MISC_OTP_PSD_AON_Pos                  (29)
#define ANA_MISC_OTP_PSD_AON_Msk                  (0x1ul << ANA_MISC_OTP_PSD_AON_Pos)

#define ANA_MISC_TST_LDO_OUT_AON_Pos              (28)
#define ANA_MISC_TST_LDO_OUT_AON_Msk              (0x1ul << ANA_MISC_TST_LDO_OUT_AON_Pos)

#define ANA_MISC_PMU_SW1_LPLDOH_LPLDOL_EN_AON_Pos (23)
#define ANA_MISC_PMU_SW1_LPLDOH_LPLDOL_EN_AON_Msk (0x1ul << ANA_MISC_PMU_SW1_LPLDOH_LPLDOL_EN_AON_Pos)

#define ANA_MISC_PMU_SW0_DVDD_LPLDOL_EN_AON_Pos   (22)
#define ANA_MISC_PMU_SW0_DVDD_LPLDOL_EN_AON_Msk   (0x1ul << ANA_MISC_PMU_SW0_DVDD_LPLDOL_EN_AON_Pos)

#define ANA_MISC_LEVELSHIFT_EN_HPLDO_AON_Pos      (21)
#define ANA_MISC_LEVELSHIFT_EN_HPLDO_AON_Msk      (0x1ul << ANA_MISC_LEVELSHIFT_EN_HPLDO_AON_Pos)

#define ANA_MISC_LEVELSHIFT_EN_GLOBAL_AON_Pos     (20)
#define ANA_MISC_LEVELSHIFT_EN_GLOBAL_AON_Msk     (0x1ul << ANA_MISC_LEVELSHIFT_EN_GLOBAL_AON_Pos)

#define ANA_MISC_PMU_IPOLY_TRIM_AON_Pos           (17)
#define ANA_MISC_PMU_IPOLY_TRIM_AON_Msk           (0x7ul << ANA_MISC_PMU_IPOLY_TRIM_AON_Pos)

#define ANA_MISC_PMU_IPOLY_EN_AON_Pos             (16)
#define ANA_MISC_PMU_IPOLY_EN_AON_Msk             (0x1ul << ANA_MISC_PMU_IPOLY_EN_AON_Pos)

#define ANA_MISC_PMU_PTAT_VTRIM_AON_Pos           (4)
#define ANA_MISC_PMU_PTAT_VTRIM_AON_Msk           (0x7ul << ANA_MISC_PMU_PTAT_VTRIM_AON_Pos)

#define ANA_MISC_PMU_PTAT_BUF_EN_AON_Pos          (3)
#define ANA_MISC_PMU_PTAT_BUF_EN_AON_Msk          (0x1ul << ANA_MISC_PMU_PTAT_BUF_EN_AON_Pos)

#define ANA_MISC_PMU_PTAT_TEMPTRIM_AON_Pos        (1)
#define ANA_MISC_PMU_PTAT_TEMPTRIM_AON_Msk        (0x3ul << ANA_MISC_PMU_PTAT_TEMPTRIM_AON_Pos)

#define ANA_MISC_PMU_PTAT_EN_AON_Pos              (0)
#define ANA_MISC_PMU_PTAT_EN_AON_Msk              (0x1ul << ANA_MISC_PMU_PTAT_EN_AON_Pos)

// ANA_HPLDO
#define ANA_HPLDO_PMU_LDO_RDY_VREF_TRIM_AON_Pos   (8)
#define ANA_HPLDO_PMU_LDO_RDY_VREF_TRIM_AON_Msk   (0x7ul << ANA_HPLDO_PMU_LDO_RDY_VREF_TRIM_AON_Pos)

#define ANA_HPLDO_PMU_HPLDO_VTRIM_AON_Pos         (4)
#define ANA_HPLDO_PMU_HPLDO_VTRIM_AON_Msk         (0xful << ANA_HPLDO_PMU_HPLDO_VTRIM_AON_Pos)

#define ANA_HPLDO_PMU_HPLDO_CAP_SEL_AON_Pos       (2)
#define ANA_HPLDO_PMU_HPLDO_CAP_SEL_AON_Msk       (0x1ul << ANA_HPLDO_PMU_HPLDO_CAP_SEL_AON_Pos)

#define ANA_HPLDO_PMU_HPLDO_EN_AON_Pos            (0)
#define ANA_HPLDO_PMU_HPLDO_EN_AON_Msk            (0x1ul << ANA_HPLDO_PMU_HPLDO_EN_AON_Pos)

// ANA_LPLDO
#define ANA_LPLDO_PMU_LPLDOH_VSEL_AON_Pos         (20)
#define ANA_LPLDO_PMU_LPLDOH_VSEL_AON_Msk         (0xful << ANA_LPLDO_PMU_LPLDOH_VSEL_AON_Pos)

#define ANA_LPLDO_PMU_LPLDOH_MODE2_EN_AON_Pos     (17)
#define ANA_LPLDO_PMU_LPLDOH_MODE2_EN_AON_Msk     (0x1ul << ANA_LPLDO_PMU_LPLDOH_MODE2_EN_AON_Pos)

#define ANA_LPLDO_PMU_LPLDOL_EN_AON_Pos           (0)
#define ANA_LPLDO_PMU_LPLDOL_EN_AON_Msk           (0x1ul << ANA_LPLDO_PMU_LPLDOL_EN_AON_Pos)

// ANA_ANALDO
#define ANA_ANALDO_EN_LDO_FSYN_AON_Pos            (16)
#define ANA_ANALDO_EN_LDO_FSYN_AON_Msk            (0x1ul << ANA_ANALDO_EN_LDO_FSYN_AON_Pos)

#define ANA_ANALDO_LDO_ANA_TRIM_AON_Pos           (4)
#define ANA_ANALDO_LDO_ANA_TRIM_AON_Msk           (0xful << ANA_ANALDO_LDO_ANA_TRIM_AON_Pos)

#define ANA_ANALDO_EN_LDO_ANA_AON_Pos             (0)
#define ANA_ANALDO_EN_LDO_ANA_AON_Msk             (0x1ul << ANA_ANALDO_EN_LDO_ANA_AON_Pos)

// ANA_RESERVED
#define ANA_RESERVED_RESERVED1_H_AON_Pos           (21)
#define ANA_RESERVED_RESERVED1_H_AON_Msk           (0x7fful << ANA_RESERVED_RESERVED1_H_AON_Pos)

#define ANA_RESERVED_PMU_LPLDOL_VSEL_AON_Pos       (17)
#define ANA_RESERVED_PMU_LPLDOL_VSEL_AON_Msk       (0xful << ANA_RESERVED_PMU_LPLDOL_VSEL_AON_Pos)

#define ANA_RESERVED_RESERVED_H_AON_Pos            (16)
#define ANA_RESERVED_RESERVED_H_AON_Msk            (0x1ul << ANA_RESERVED_RESERVED_H_AON_Pos)

#define ANA_RESERVED_RESERVED_L_AON_Pos            (0)
#define ANA_RESERVED_RESERVED_L_AON_Msk            (0xfffful << ANA_RESERVED_RESERVED_L_AON_Pos)


/*secure*/
typedef struct {
	__IO uint32_t VERIFI_DEBUG_KEY0;
	__IO uint32_t VERIFI_DEBUG_KEY1;
	__IO uint32_t VERIFI_DEBUG_KEY2;
	__IO uint32_t VERIFI_DEBUG_KEY3;
} CHIP_SECURITY_T;


#define VERIFI_DEBUG_KEY0_verify_debug_key0_Pos (0)
#define VERIFI_DEBUG_KEY0_verify_debug_key0_Msk (0xfffffffful << VERIFI_DEBUG_KEY0_verify_debug_key0_Pos)

#define VERIFI_DEBUG_KEY1_verify_debug_key1_Pos (0)
#define VERIFI_DEBUG_KEY1_verify_debug_key1_Msk (0xfffffffful << VERIFI_DEBUG_KEY1_verify_debug_key1_Pos)

#define VERIFI_DEBUG_KEY2_verify_debug_key2_Pos (0)
#define VERIFI_DEBUG_KEY2_verify_debug_key2_Msk (0xfffffffful << VERIFI_DEBUG_KEY2_verify_debug_key2_Pos)

#define VERIFI_DEBUG_KEY3_verify_debug_key3_Pos (0)
#define VERIFI_DEBUG_KEY3_verify_debug_key3_Msk (0xfffffffful << VERIFI_DEBUG_KEY3_verify_debug_key3_Pos)



// USB regs
typedef struct {
	__IO uint8_t FADDR    			;//0X00;
	__IO uint8_t POWER    			;//0X01;
	__IO uint8_t INT_IN1  			;//0X02;
	__IO uint8_t INT_IN2  			;//0X03;
	__IO uint8_t INT_OUT1 			;//0X04;
	__IO uint8_t INT_OUT2 			;//0X05;
	__IO uint8_t INT_USB  			;//0X06;
	__IO uint8_t INT_IN1E 			;//0X07;
	__IO uint8_t INT_IN2E 			;//0X08;
	__IO uint8_t INT_OUT1E			;//0X09;
	__IO uint8_t INT_OUT2E			;//0X0A;
	__IO uint8_t INT_USBE 			;//0X0B;
	__IO uint8_t FRAME1   			;//0X0C;
	__IO uint8_t FRAME2   			;//0X0D;
	__IO uint8_t INDEX    			;//0X0E;
	__IO uint8_t REV     			;//0X0F;
	__IO uint8_t MAX_PKT_IN   		;//0X10;
	__IO uint8_t CSR0_INCSR1   		;//0X11;
	__IO uint8_t IN_CSR2   			;//0X12;
	__IO uint8_t MAX_PKT_OUT  		;//0X13;
	__IO uint8_t OUT_CSR1  			;//0X14;
	__IO uint8_t OUT_CSR2  			;//0X15;
	__IO uint8_t OUT_COUNT1			;//0X16;
	__IO uint8_t OUT_COUNT2			;//0X17;
	__IO uint8_t REV0[7]			;//0X18~1E;
	__IO uint8_t USB_DEBUG			;//0X1F;
	__IO uint8_t FIFO_EP0 			;//0X20;
	__IO uint8_t REV1[3]			;//0X21~23;
	__IO uint8_t FIFO_EP1 			;//0X24; 
	__IO uint8_t REV2[3]			;//0X25~27;
	__IO uint8_t FIFO_EP2 			;//0X28;
	__IO uint8_t REV3[3]			;//0X29~2B;
	__IO uint8_t FIFO_EP3 			;//0X2C;
	__IO uint8_t REV4[3]			;//0X2D~2F;
	__IO uint8_t FIFO_EP4 			;//0X30;
}USB_T;

/*private rf*/
typedef struct {
	__IO uint32_t PHY1;
	__IO uint32_t RESERVED0;
	__IO uint32_t PHY3;
	__I  uint32_t PHY4;
	__IO uint32_t INTR1;
	__IO uint32_t INTCLR;
	__IO uint32_t INTMSK;
	__IO uint32_t INTR4;
	__IO uint32_t R00_CTL;
	__IO uint32_t R01_INT;
	__IO uint32_t R02_TMR_CTL;
	__IO uint32_t R03_RX_ADDR_L;
	__IO uint32_t R04_RX_CTL;
	__IO uint32_t R05_TX_ADDR_L;
	__IO uint32_t R06_TX_CTL;
	__IO uint32_t R07_SRAM_CTL;
	__IO uint32_t R08_RX_ADDR1_H;
	__IO uint32_t R09_RX_ADDR1_L;
	__IO uint32_t R10_CFG;
	__IO uint32_t R11_CFG;
	__IO uint32_t R12_WL;
	__IO uint32_t R13_WL_CFG;
} PRI_RF_T;

// PHY1
#define PHY1_PHY_DRV_NRST_Pos                 (24)
#define PHY1_PHY_DRV_NRST_Msk                 (0x1ul << PHY1_PHY_DRV_NRST_Pos)

#define PHY1_PHY_DRV_RSP_BUF_ERROR_Pos        (23)
#define PHY1_PHY_DRV_RSP_BUF_ERROR_Msk        (0x1ul << PHY1_PHY_DRV_RSP_BUF_ERROR_Pos)

#define PHY1_PHY_DRV_RSP_BUF_FULL_Pos         (22)
#define PHY1_PHY_DRV_RSP_BUF_FULL_Msk         (0x1ul << PHY1_PHY_DRV_RSP_BUF_FULL_Pos)

#define PHY1_PHY_DRV_RSP_BUF_EMPTY_Pos        (21)
#define PHY1_PHY_DRV_RSP_BUF_EMPTY_Msk        (0x1ul << PHY1_PHY_DRV_RSP_BUF_EMPTY_Pos)

#define PHY1_PHY_DRV_CFG_BUF_ERROR_Pos        (20)
#define PHY1_PHY_DRV_CFG_BUF_ERROR_Msk        (0x1ul << PHY1_PHY_DRV_CFG_BUF_ERROR_Pos)

#define PHY1_PHY_DRV_CFG_BUF_FULL_Pos         (19)
#define PHY1_PHY_DRV_CFG_BUF_FULL_Msk         (0x1ul << PHY1_PHY_DRV_CFG_BUF_FULL_Pos)

#define PHY1_PHY_DRV_CFG_BUF_EMPTY_Pos        (18)
#define PHY1_PHY_DRV_CFG_BUF_EMPTY_Msk        (0x1ul << PHY1_PHY_DRV_CFG_BUF_EMPTY_Pos)

#define PHY1_PHY1_IF_CLK_PRESCALE_Pos              (13)
#define PHY1_PHY1_IF_CLK_PRESCALE_Msk              (0x1ful << PHY1_PHY1_IF_CLK_PRESCALE_Pos)

#define PHY1_PHY_DRV_RSSI_REG_ADDR_Pos        (5)
#define PHY1_PHY_DRV_RSSI_REG_ADDR_Msk        (0xfful << PHY1_PHY_DRV_RSSI_REG_ADDR_Pos)

#define PHY1_PHY_DRV_CFG_TRG_Pos              (4)
#define PHY1_PHY_DRV_CFG_TRG_Msk              (0x1ul << PHY1_PHY_DRV_CFG_TRG_Pos)

#define PHY1_PHY_DRV_CFG_REG_Pos              (2)
#define PHY1_PHY_DRV_CFG_REG_Msk              (0x3ul << PHY1_PHY_DRV_CFG_REG_Pos)

#define PHY1_PHY_DRV_RSP_BUF_FLUSH_Pos        (1)
#define PHY1_PHY_DRV_RSP_BUF_FLUSH_Msk        (0x1ul << PHY1_PHY_DRV_RSP_BUF_FLUSH_Pos)

#define PHY1_PHY_DRV_CFG_BUF_FLUSH_Pos        (0)
#define PHY1_PHY_DRV_CFG_BUF_FLUSH_Msk        (0x1ul << PHY1_PHY_DRV_CFG_BUF_FLUSH_Pos)

// PHY3
#define PHY3_PHY_DRV_CFG_BUF_DIN_Pos          (0)
#define PHY3_PHY_DRV_CFG_BUF_DIN_Msk          (0xfffffffful << PHY3_PHY_DRV_CFG_BUF_DIN_Pos)

// PHY4
#define PHY4_PHY_DRV_RSP_BUF_DOUT_Pos         (0)
#define PHY4_PHY_DRV_RSP_BUF_DOUT_Msk         (0xfffffffful << PHY4_PHY_DRV_RSP_BUF_DOUT_Pos)

// INTR1
#define INTR1_IC_PHY_DRV_CFG_DONE_Pos        (0)
#define INTR1_IC_PHY_DRV_CFG_DONE_Msk        (0x1ul << INTR1_IC_PHY_DRV_CFG_DONE_Pos)

// INTCLR
#define INTCLR_IC_PHY_DRV_CFG_DONE_CLR_Pos  (0)
#define INTCLR_IC_PHY_DRV_CFG_DONE_CLR_Msk  (0x1ul << INTCLR_IC_PHY_DRV_CFG_DONE_CLR_Pos)

// INTMSK
#define INTMSK_IC_PHY_DRV_CFG_DONE_MASK_Pos (0)
#define INTMSK_IC_PHY_DRV_CFG_DONE_MASK_Msk (0x1ul << INTMSK_IC_PHY_DRV_CFG_DONE_MASK_Pos)

// INTR4
#define INTR4_16M_OSC_Pos                    (19)
#define INTR4_16M_OSC_Msk                    (0x1ul << INTR4_16M_OSC_Pos)

#define INTR4_FEC_DEC_THR_Pos                (9)
#define INTR4_FEC_DEC_THR_Msk                (0x3fful << INTR4_FEC_DEC_THR_Pos)

#define INTR4_IC_SLP_NRST_Pos                (8)
#define INTR4_IC_SLP_NRST_Msk                (0x1ul << INTR4_IC_SLP_NRST_Pos)

#define INTR4_IC_POLARITY_IRQ_Pos            (7)
#define INTR4_IC_POLARITY_IRQ_Msk            (0x1ul << INTR4_IC_POLARITY_IRQ_Pos)

#define INTR4_IC_CLR_IRQ_Pos                 (6)
#define INTR4_IC_CLR_IRQ_Msk                 (0x1ul << INTR4_IC_CLR_IRQ_Pos)

#define INTR4_IC_ONE_PULSE_FLAG_IRQ_Pos      (5)
#define INTR4_IC_ONE_PULSE_FLAG_IRQ_Msk      (0x1ul << INTR4_IC_ONE_PULSE_FLAG_IRQ_Pos)

#define INTR4_IC_MODE_IRQ_Pos                (4)
#define INTR4_IC_MODE_IRQ_Msk                (0x1ul << INTR4_IC_MODE_IRQ_Pos)

#define INTR4_IC_PULSE_CNT_Pos               (0)
#define INTR4_IC_PULSE_CNT_Msk               (0xful << INTR4_IC_PULSE_CNT_Pos)

// R00_CTL
#define R00_CTL_TX_PAYLOAD_LENGTH_Pos              (24)
#define R00_CTL_TX_PAYLOAD_LENGTH_Msk              (0xfful << R00_CTL_TX_PAYLOAD_LENGTH_Pos)

#define R00_CTL_RX_PAYLOAD_LENGTH_Pos              (16)
#define R00_CTL_RX_PAYLOAD_LENGTH_Msk              (0xfful << R00_CTL_RX_PAYLOAD_LENGTH_Pos)

#define R00_CTL_CRC_ACCESS_ADR_Pos             		(15)
#define R00_CTL_CRC_ACCESS_ADR_Msk             		(0x1ul << R00_CTL_CRC_ACCESS_ADR_Pos)

#define R00_CTL_CRC_SEL24_Pos                      (14)
#define R00_CTL_CRC_SEL24_Msk                      (0x1ul << R00_CTL_CRC_SEL24_Pos)

#define R00_CTL_ADDR_BYTE_LENGTH_Pos               (12)
#define R00_CTL_ADDR_BYTE_LENGTH_Msk               (0x3ul << R00_CTL_ADDR_BYTE_LENGTH_Pos)

#define R00_CTL_DPY_EN_Pos                         (11)
#define R00_CTL_DPY_EN_Msk                         (0x1ul << R00_CTL_DPY_EN_Pos)

#define R00_CTL_CRC_EN_Pos                     	   (10)
#define R00_CTL_CRC_EN_Msk                     		(0x1ul << R00_CTL_CRC_EN_Pos)

#define R00_CTL_CRC_SEL16_Pos                      (9)
#define R00_CTL_CRC_SEL16_Msk                      (0x1ul << R00_CTL_CRC_SEL16_Pos)

#define R00_CTL_SCR_EN_Pos                     		(8)
#define R00_CTL_SCR_EN_Msk                     		(0x1ul << R00_CTL_SCR_EN_Pos)

#define R00_CTL_NRF_ENHANCE_Pos                 	(7)
#define R00_CTL_NRF_ENHANCE_Msk                 	(0x1ul << R00_CTL_NRF_ENHANCE_Pos)

#define R00_CTL_ENHANCE_Pos                        (6)
#define R00_CTL_ENHANCE_Msk                        (0x1ul << R00_CTL_ENHANCE_Pos)

#define R00_CTL_BW_MODE_Pos                        (5)
#define R00_CTL_BW_MODE_Msk                        (0x1ul << R00_CTL_BW_MODE_Pos)

#define R00_CTL_CHIP_MODE_Pos                      (3)
#define R00_CTL_CHIP_MODE_Msk                      (0x3ul << R00_CTL_CHIP_MODE_Pos)

#define R00_CTL_RX_ACK_PAYLOAD_EN_Pos              (2)
#define R00_CTL_RX_ACK_PAYLOAD_EN_Msk              (0x1ul << R00_CTL_RX_ACK_PAYLOAD_EN_Pos)

#define R00_CTL_TX_NOACK_EN_Pos                    (1)
#define R00_CTL_TX_NOACK_EN_Msk                    (0x1ul << R00_CTL_TX_NOACK_EN_Pos)

#define R00_CTL_PRI_RX_Pos                         (0)
#define R00_CTL_PRI_RX_Msk                         (0x1ul << R00_CTL_PRI_RX_Pos)

// R01_INT
#define R01_INT_OM_FRAME1_EN_Pos                   (31)
#define R01_INT_OM_FRAME1_EN_Msk                   (0x1ul << R01_INT_OM_FRAME1_EN_Pos)

#define R01_INT_RX_PID_Pos                         (29)
#define R01_INT_RX_PID_Msk                         (0x3ul << R01_INT_RX_PID_Pos)

#define R01_INT_TX_PID_Pos                         (27)
#define R01_INT_TX_PID_Msk                         (0x3ul << R01_INT_TX_PID_Pos)

#define R01_INT_PRI_RX_LNGTH_ERR_Pos               (26)
#define R01_INT_PRI_RX_LNGTH_ERR_Msk               (0x1ul << R01_INT_PRI_RX_LNGTH_ERR_Pos)

#define R01_INT_PRI_ACC_ADDR_ERR_Pos               (25)
#define R01_INT_PRI_ACC_ADDR_ERR_Msk               (0x1ul << R01_INT_PRI_ACC_ADDR_ERR_Pos)

#define R01_INT_OM_SCR_MODE_EN_Pos                 (24)
#define R01_INT_OM_SCR_MODE_EN_Msk                 (0x1ul << R01_INT_OM_SCR_MODE_EN_Pos)

#define R01_INT_ACC_ADDR_ERR_MASK_Pos              (23)
#define R01_INT_ACC_ADDR_ERR_MASK_Msk              (0x1ul << R01_INT_ACC_ADDR_ERR_MASK_Pos)

#define R01_INT_PRI_RX_LNGTH_ERR_IRQ_MASK_Pos      (22)
#define R01_INT_PRI_RX_LNGTH_ERR_IRQ_MASK_Msk      (0x1ul << R01_INT_PRI_RX_LNGTH_ERR_IRQ_MASK_Pos)

#define R01_INT_PRI_ENDIAN_Pos                     (21)
#define R01_INT_PRI_ENDIAN_Msk                     (0x1ul << R01_INT_PRI_ENDIAN_Pos)

#define R01_INT_TX_DONE_IRQ_EN_Pos                 (20)
#define R01_INT_TX_DONE_IRQ_EN_Msk                 (0x1ul << R01_INT_TX_DONE_IRQ_EN_Pos)

#define R01_INT_RX_DONE_IRQ_EN_Pos                 (19)
#define R01_INT_RX_DONE_IRQ_EN_Msk                 (0x1ul << R01_INT_RX_DONE_IRQ_EN_Pos)

#define R01_INT_PRI_RX_GOON_Pos                        (18)
#define R01_INT_PRI_RX_GOON_Msk                        (0x1ul << R01_INT_PRI_RX_GOON_Pos)

#define R01_INT_PRI_RST_Pos                        (17)
#define R01_INT_PRI_RST_Msk                        (0x1ul << R01_INT_PRI_RST_Pos)

#define R01_INT_PRI_EXIT_RX_Pos                    (16)
#define R01_INT_PRI_EXIT_RX_Msk                    (0x1ul << R01_INT_PRI_EXIT_RX_Pos)

#define R01_INT_TX_DONE_IRQ_FLAG_Pos               (15)
#define R01_INT_TX_DONE_IRQ_FLAG_Msk               (0x1ul << R01_INT_TX_DONE_IRQ_FLAG_Pos)

#define R01_INT_RX_DONE_IRQ_FLAG_Pos               (14)
#define R01_INT_RX_DONE_IRQ_FLAG_Msk               (0x1ul << R01_INT_RX_DONE_IRQ_FLAG_Pos)

#define R01_INT_RX_PID_ERR_IRQ_FLAG_Pos            (13)
#define R01_INT_RX_PID_ERR_IRQ_FLAG_Msk            (0x1ul << R01_INT_RX_PID_ERR_IRQ_FLAG_Pos)

#define R01_INT_RX_CRC_ERR_IRQ_FLAG_Pos            (12)
#define R01_INT_RX_CRC_ERR_IRQ_FLAG_Msk            (0x1ul << R01_INT_RX_CRC_ERR_IRQ_FLAG_Pos)

#define R01_INT_RX_TIMEOUT_IRQ_FLAG_Pos            (11)
#define R01_INT_RX_TIMEOUT_IRQ_FLAG_Msk            (0x1ul << R01_INT_RX_TIMEOUT_IRQ_FLAG_Pos)

#define R01_INT_PRI_RETRANS_FLAG_Pos               (10)
#define R01_INT_PRI_RETRANS_FLAG_Msk               (0x1ul << R01_INT_PRI_RETRANS_FLAG_Pos)

#define R01_INT_RX_IRQ_FLAG_Pos                    (9)
#define R01_INT_RX_IRQ_FLAG_Msk                    (0x1ul << R01_INT_RX_IRQ_FLAG_Pos)

#define R01_INT_IRQ_CLR_EN_Pos                   (8)
#define R01_INT_IRQ_CLR_EN_Msk                   (0x1ul << R01_INT_IRQ_CLR_EN_Pos)

#define R01_INT_RX_CRC_ERR_IRQ_MASK_Pos            (7)
#define R01_INT_RX_CRC_ERR_IRQ_MASK_Msk            (0x1ul << R01_INT_RX_CRC_ERR_IRQ_MASK_Pos)

#define R01_INT_RX_PID_ERR_IRQ_MASK_Pos            (6)
#define R01_INT_RX_PID_ERR_IRQ_MASK_Msk            (0x1ul << R01_INT_RX_PID_ERR_IRQ_MASK_Pos)

#define R01_INT_TX_RXACK_OUTTIME_IRQ_MASK_Pos      (5)
#define R01_INT_TX_RXACK_OUTTIME_IRQ_MASK_Msk      (0x1ul << R01_INT_TX_RXACK_OUTTIME_IRQ_MASK_Pos)

#define R01_INT_ACC_MATCH_IRQ_MASK_Pos             (4)
#define R01_INT_ACC_MATCH_IRQ_MASK_Msk             (0x1ul << R01_INT_ACC_MATCH_IRQ_MASK_Pos)

#define R01_INT_RX_IRQ_MASK_Pos                    (3)
#define R01_INT_RX_IRQ_MASK_Msk                    (0x1ul << R01_INT_RX_IRQ_MASK_Pos)

#define R01_INT_OM_FRAME_EN_Pos                    (2)
#define R01_INT_OM_FRAME_EN_Msk                    (0x1ul << R01_INT_OM_FRAME_EN_Pos)

#define R01_INT_MULTI_RX_ACC_ADDR_Pos              (0)
#define R01_INT_MULTI_RX_ACC_ADDR_Msk              (0x3ul << R01_INT_MULTI_RX_ACC_ADDR_Pos)

// R02_TMR_CTL
#define R02_TMR_CTL_RX_WAIT_TIME_Pos                   (16)
#define R02_TMR_CTL_RX_WAIT_TIME_Msk                   (0xfffful << R02_TMR_CTL_RX_WAIT_TIME_Pos)

#define R02_TMR_CTL_RX_WAIT_TIME_OUT_EN_Pos            (15)
#define R02_TMR_CTL_RX_WAIT_TIME_OUT_EN_Msk            (0x1ul << R02_TMR_CTL_RX_WAIT_TIME_OUT_EN_Pos)

#define R02_TMR_CTL_TRX_TRANS_WAIT_TIME_Pos            (0)
#define R02_TMR_CTL_TRX_TRANS_WAIT_TIME_Msk            (0x7ffful << R02_TMR_CTL_TRX_TRANS_WAIT_TIME_Pos)

// R03_RX_ADDR_L
#define R03_RX_ADDR_L_RX_ADDR_Pos                        (0)
#define R03_RX_ADDR_L_RX_ADDR_Msk                        (0xfffffffful << R03_RX_ADDR_L_RX_ADDR_Pos)

// R04_RX_CTL
#define R04_RX_CTL_RX_HEADER1_Pos                     (24)
#define R04_RX_CTL_RX_HEADER1_Msk                     (0xfful << R04_RX_CTL_RX_HEADER1_Pos)

#define R04_RX_CTL_RX_HEADER0_Pos                     (16)
#define R04_RX_CTL_RX_HEADER0_Msk                     (0xfful << R04_RX_CTL_RX_HEADER0_Pos)

#define R04_RX_CTL_PRI_RX_FEC_Pos                     (15)
#define R04_RX_CTL_PRI_RX_FEC_Msk                     (0x1ul << R04_RX_CTL_PRI_RX_FEC_Pos)

#define R04_RX_CTL_PRI_TX_FEC_Pos                     (14)
#define R04_RX_CTL_PRI_TX_FEC_Msk                     (0x1ul << R04_RX_CTL_PRI_TX_FEC_Pos)

#define R04_RX_CTL_PRI_CI_MODE_Pos                    (12)
#define R04_RX_CTL_PRI_CI_MODE_Msk                    (0x3ul << R04_RX_CTL_PRI_CI_MODE_Pos)

#define R04_RX_CTL_NORMAL_M1_Pos                      (11)
#define R04_RX_CTL_NORMAL_M1_Msk                      (0x1ul << R04_RX_CTL_NORMAL_M1_Pos)

#define R04_RX_CTL_HDR_LEN_NUMB_Pos                   (9)
#define R04_RX_CTL_HDR_LEN_NUMB_Msk                   (0x3ul << R04_RX_CTL_HDR_LEN_NUMB_Pos)

#define R04_RX_CTL_HDR_LEN_EXIST_Pos                  (8)
#define R04_RX_CTL_HDR_LEN_EXIST_Msk                  (0x1ul << R04_RX_CTL_HDR_LEN_EXIST_Pos)

#define R04_RX_CTL_RX_ADDR_Pos                        (0)
#define R04_RX_CTL_RX_ADDR_Msk                        (0xfful << R04_RX_CTL_RX_ADDR_Pos)

// R05_TX_ADDR_L
#define R05_TX_ADDR_L_TX_ADDR_Pos                        (0)
#define R05_TX_ADDR_L_TX_ADDR_Msk                        (0xfffffffful << R05_TX_ADDR_L_TX_ADDR_Pos)

// R06_TX_CTL
#define R06_TX_CTL_TX_HEADER1_Pos                     (24)
#define R06_TX_CTL_TX_HEADER1_Msk                     (0xfful << R06_TX_CTL_TX_HEADER1_Pos)

#define R06_TX_CTL_TX_HEADER0_Pos                     (16)
#define R06_TX_CTL_TX_HEADER0_Msk                     (0xfful << R06_TX_CTL_TX_HEADER0_Pos)

#define R06_TX_CTL_PRI_RX_MAX_CTRL_EN_Pos             (15)
#define R06_TX_CTL_PRI_RX_MAX_CTRL_EN_Msk             (0x1ul << R06_TX_CTL_PRI_RX_MAX_CTRL_EN_Pos)

#define R06_TX_CTL_WHITENING_INIT_Pos                 (8)
#define R06_TX_CTL_WHITENING_INIT_Msk                 (0x7ful << R06_TX_CTL_WHITENING_INIT_Pos)

#define R06_TX_CTL_TX_ADDR_Pos                        (0)
#define R06_TX_CTL_TX_ADDR_Msk                        (0xfful << R06_TX_CTL_TX_ADDR_Pos)

// R07_SRAM_CTL
#define R07_SRAM_CTL_RCVD_PAYLOAD_LENGTH_Pos            (23)
#define R07_SRAM_CTL_RCVD_PAYLOAD_LENGTH_Msk            (0xfful << R07_SRAM_CTL_RCVD_PAYLOAD_LENGTH_Pos)

#define R07_SRAM_CTL_RX_RAM_START_ADDR_Pos              (17)
#define R07_SRAM_CTL_RX_RAM_START_ADDR_Msk              (0x3ful << R07_SRAM_CTL_RX_RAM_START_ADDR_Pos)

#define R07_SRAM_CTL_RX_RAM_READY_Pos                   (16)
#define R07_SRAM_CTL_RX_RAM_READY_Msk                   (0x1ul << R07_SRAM_CTL_RX_RAM_READY_Pos)

#define R07_SRAM_CTL_TX_RAM_START_ADDR_Pos              (1)
#define R07_SRAM_CTL_TX_RAM_START_ADDR_Msk              (0x3ful << R07_SRAM_CTL_TX_RAM_START_ADDR_Pos)

#define R07_SRAM_CTL_TX_RAM_READY_Pos                   (0)
#define R07_SRAM_CTL_TX_RAM_READY_Msk                   (0x1ul << R07_SRAM_CTL_TX_RAM_READY_Pos)

// R08_RX_ADDR1_H
#define R08_RX_ADDR1_H_ADDR1_Pos                          (0)
#define R08_RX_ADDR1_H_ADDR1_Msk                          (0xfffffffful << R08_RX_ADDR1_H_ADDR1_Pos)

// R09_RX_ADDR1_L
#define R09_RX_ADDR1_L_PRI_TST_SELECT_1_Pos               (23)
#define R09_RX_ADDR1_L_PRI_TST_SELECT_1_Msk               (0xful << R09_RX_ADDR1_L_PRI_TST_SELECT_1_Pos)

#define R09_RX_ADDR1_L_PRI_TST_SELECT_0_Pos               (19)
#define R09_RX_ADDR1_L_PRI_TST_SELECT_0_Msk               (0xful << R09_RX_ADDR1_L_PRI_TST_SELECT_0_Pos)

#define R09_RX_ADDR2_EN_Pos                        			(18)
#define R09_RX_ADDR2_EN_Msk                        			(0x1ul << R09_RX_ADDR2_EN_Pos)

#define R09_RX_ADDR1_EN_Pos                        			(17)
#define R09_RX_ADDR1_EN_Msk                        			(0x1ul << R09_RX_ADDR1_EN_Pos)

#define R09_RX_ADDR0_EN_Pos                        			(16)
#define R09_RX_ADDR0_EN_Msk                        			(0x1ul << R09_RX_ADDR0_EN_Pos)

#define R09_RX_ADDR1_L_P2_ADDR_LSB_Pos                    (8)
#define R09_RX_ADDR1_L_P2_ADDR_LSB_Msk                    (0xfful << R09_RX_ADDR1_L_P2_ADDR_LSB_Pos)

#define R09_RX_ADDR1_L_P1_ADDR_LSB_Pos                    (0)
#define R09_RX_ADDR1_L_P1_ADDR_LSB_Msk                    (0xfful << R09_RX_ADDR1_L_P1_ADDR_LSB_Pos)

// R10_CFG
#define R10_CFG_NDC_PID_ORDER_Pos                  (31)
#define R10_CFG_NDC_PID_ORDER_Msk                  (0x1ul << R10_CFG_NDC_PID_ORDER_Pos)

#define R10_CFG_NDC_ACK_SEL_Pos                    (30)
#define R10_CFG_NDC_ACK_SEL_Msk                    (0x1ul << R10_CFG_NDC_ACK_SEL_Pos)

#define R10_CFG_TX_ON_DLY_SYMB_Pos                 (26)
#define R10_CFG_TX_ON_DLY_SYMB_Msk                 (0xful << R10_CFG_TX_ON_DLY_SYMB_Pos)

#define R10_CFG_RX_DATA_INVERT_EN_Pos              (24)
#define R10_CFG_RX_DATA_INVERT_EN_Msk              (0x1ul << R10_CFG_RX_DATA_INVERT_EN_Pos)

#define R10_CFG_PRE_SYNC_12B_EN_Pos                (23)
#define R10_CFG_PRE_SYNC_12B_EN_Msk                (0x1ul << R10_CFG_PRE_SYNC_12B_EN_Pos)

#define R10_CFG_PRE_SYNC_8B_EN_Pos                 (22)
#define R10_CFG_PRE_SYNC_8B_EN_Msk                 (0x1ul << R10_CFG_PRE_SYNC_8B_EN_Pos)

#define R10_CFG_PRE_SYNC_4B_EN_Pos                 (21)
#define R10_CFG_PRE_SYNC_4B_EN_Msk                 (0x1ul << R10_CFG_PRE_SYNC_4B_EN_Pos)

#define R10_CFG_PRE_SYNC_ENABLE_Pos                (20)
#define R10_CFG_PRE_SYNC_ENABLE_Msk                (0x1ul << R10_CFG_PRE_SYNC_ENABLE_Pos)

#define R10_CFG_250K_MODE_EN_Pos                   (19)
#define R10_CFG_250K_MODE_EN_Msk                   (0x1ul << R10_CFG_250K_MODE_EN_Pos)

#define R10_CFG_BOE_PRE_SEL_Pos                    (18)
#define R10_CFG_BOE_PRE_SEL_Msk                    (0x1ul << R10_CFG_BOE_PRE_SEL_Pos)

#define R10_CFG_LQI_EN_FOR_SYNC_Pos                (17)
#define R10_CFG_LQI_EN_FOR_SYNC_Msk                (0x1ul << R10_CFG_LQI_EN_FOR_SYNC_Pos)

#define R10_CFG_RX_PID_MANUAL_Pos                  (15)
#define R10_CFG_RX_PID_MANUAL_Msk                  (0x3ul << R10_CFG_RX_PID_MANUAL_Pos)

#define R10_CFG_TX_PID_MANUAL_Pos                  (13)
#define R10_CFG_TX_PID_MANUAL_Msk                  (0x3ul << R10_CFG_TX_PID_MANUAL_Pos)

#define R10_CFG_PRE_2BYTE_MODE_Pos                 (12)
#define R10_CFG_PRE_2BYTE_MODE_Msk                 (0x1ul << R10_CFG_PRE_2BYTE_MODE_Pos)

#define R10_CFG_TX_DATA_INVERT_EN_Pos              (11)
#define R10_CFG_TX_DATA_INVERT_EN_Msk              (0x1ul << R10_CFG_TX_DATA_INVERT_EN_Pos)

#define R10_CFG_SCR_INI_Pos                        (9)
#define R10_CFG_SCR_INI_Msk                        (0x3ul << R10_CFG_SCR_INI_Pos)

#define R10_CFG_250K_PRE_LEN_Pos                   (6)
#define R10_CFG_250K_PRE_LEN_Msk                   (0x7ul << R10_CFG_250K_PRE_LEN_Pos)

#define R10_CFG_BOE_MODE_EN_Pos                    (5)
#define R10_CFG_BOE_MODE_EN_Msk                    (0x1ul << R10_CFG_BOE_MODE_EN_Pos)

#define R10_CFG_ADDR_ERR_THR_Pos                   (2)
#define R10_CFG_ADDR_ERR_THR_Msk                   (0x7ul << R10_CFG_ADDR_ERR_THR_Pos)

#define R10_CFG_PID_MANUAL_EN_Pos                  (1)
#define R10_CFG_PID_MANUAL_EN_Msk                  (0x1ul << R10_CFG_PID_MANUAL_EN_Pos)

#define R10_CFG_ADDR_SCR_DIS_Pos                   (0)
#define R10_CFG_ADDR_SCR_DIS_Msk                   (0x1ul << R10_CFG_ADDR_SCR_DIS_Pos)

// R11_CFG
#define R11_CFG_ACK_ADDR_AUTO_SET_EN_Pos           (31)
#define R11_CFG_ACK_ADDR_AUTO_SET_EN_Msk           (0x1ul << R11_CFG_ACK_ADDR_AUTO_SET_EN_Pos)

#define R11_CFG_ACC_ADDR_ERR_RETRANS_EN_Pos        (30)
#define R11_CFG_ACC_ADDR_ERR_RETRANS_EN_Msk        (0x1ul << R11_CFG_ACC_ADDR_ERR_RETRANS_EN_Pos)

#define R11_CFG_LENGTH_ERR_RETRANS_EN_Pos          (29)
#define R11_CFG_LENGTH_ERR_RETRANS_EN_Msk          (0x1ul << R11_CFG_LENGTH_ERR_RETRANS_EN_Pos)

#define R11_CFG_CRC_ERR_RETRANS_EN_Pos             (28)
#define R11_CFG_CRC_ERR_RETRANS_EN_Msk             (0x1ul << R11_CFG_CRC_ERR_RETRANS_EN_Pos)

#define R11_CFG_TIMEOUT_RETRANS_EN_Pos             (27)
#define R11_CFG_TIMEOUT_RETRANS_EN_Msk             (0x1ul << R11_CFG_TIMEOUT_RETRANS_EN_Pos)

#define R11_CFG_RETRANSMIT_MAX_CNT_Pos             (24)
#define R11_CFG_RETRANSMIT_MAX_CNT_Msk             (0x7ul << R11_CFG_RETRANSMIT_MAX_CNT_Pos)

#define R11_CFG_RETANS_WAIT_TIME_Pos               (9)
#define R11_CFG_RETANS_WAIT_TIME_Msk               (0x7ffful << R11_CFG_RETANS_WAIT_TIME_Pos)

#define R11_CFG_ACC_MATCH_IRQ_FLAG_Pos             (8)
#define R11_CFG_ACC_MATCH_IRQ_FLAG_Msk             (0x1ul << R11_CFG_ACC_MATCH_IRQ_FLAG_Pos)

#define R11_CFG_PRI_RX_IRQ_FLAG_Pos                (7)
#define R11_CFG_PRI_RX_IRQ_FLAG_Msk                (0x1ul << R11_CFG_PRI_RX_IRQ_FLAG_Pos)

#define R11_CFG_PRI_PID_ERR_IRQ_FLAG_Pos           (6)
#define R11_CFG_PRI_PID_ERR_IRQ_FLAG_Msk           (0x1ul << R11_CFG_PRI_PID_ERR_IRQ_FLAG_Pos)

#define R11_CFG_PRI_TIMEOUT_IRQ_FLAG_Pos           (5)
#define R11_CFG_PRI_TIMEOUT_IRQ_FLAG_Msk           (0x1ul << R11_CFG_PRI_TIMEOUT_IRQ_FLAG_Pos)

#define R11_CFG_PRI_CRC_ERR_IRQ_FLAG_Pos           (4)
#define R11_CFG_PRI_CRC_ERR_IRQ_FLAG_Msk           (0x1ul << R11_CFG_PRI_CRC_ERR_IRQ_FLAG_Pos)

#define R11_CFG_PRI_RX_DONE_IRQ_FLAG_Pos           (3)
#define R11_CFG_PRI_RX_DONE_IRQ_FLAG_Msk           (0x1ul << R11_CFG_PRI_RX_DONE_IRQ_FLAG_Pos)

#define R11_CFG_PRI_TX_DONE_IRQ_FLAG_Pos           (2)
#define R11_CFG_PRI_TX_DONE_IRQ_FLAG_Msk           (0x1ul << R11_CFG_PRI_TX_DONE_IRQ_FLAG_Pos)

#define R11_CFG_PRI_LENGTH_ERR_IRQ_FLAG_Pos        (1)
#define R11_CFG_PRI_LENGTH_ERR_IRQ_FLAG_Msk        (0x1ul << R11_CFG_PRI_LENGTH_ERR_IRQ_FLAG_Pos)

#define R11_CFG_PRI_ACC_ADDR_ERR_IRQ_FLAG_Pos      (0)
#define R11_CFG_PRI_ACC_ADDR_ERR_IRQ_FLAG_Msk      (0x1ul << R11_CFG_PRI_ACC_ADDR_ERR_IRQ_FLAG_Pos)

// R12_WL
#define R12_WL_PRI_RX_WL_ADVA_LSB_Pos             (0)
#define R12_WL_PRI_RX_WL_ADVA_LSB_Msk             (0xfffffffful << R12_WL_PRI_RX_WL_ADVA_LSB_Pos)

// R13_WL_CFG
#define R13_WL_CFG_PRI_RX_PLD_START_BYTE_Pos          (21)
#define R13_WL_CFG_PRI_RX_PLD_START_BYTE_Msk          (0x3ful << R13_WL_CFG_PRI_RX_PLD_START_BYTE_Pos)

#define R13_WL_CFG_PRI_RX_WL_MATCH_MODE_Pos           (18)
#define R13_WL_CFG_PRI_RX_WL_MATCH_MODE_Msk           (0x7ul << R13_WL_CFG_PRI_RX_WL_MATCH_MODE_Pos)

#define R13_WL_CFG_PRI_RX_BLELEN_MATCH_MODE_Pos       (16)
#define R13_WL_CFG_PRI_RX_BLELEN_MATCH_MODE_Msk       (0x3ul << R13_WL_CFG_PRI_RX_BLELEN_MATCH_MODE_Pos)

#define R13_WL_CFG_PRI_RX_WL_ADVA_MSB_Pos             (0)
#define R13_WL_CFG_PRI_RX_WL_ADVA_MSB_Msk             (0xfffful << R13_WL_CFG_PRI_RX_WL_ADVA_MSB_Pos)

/** @addtogroup Peripheral_memory_map
  * @{
  */

/******************************************************************************/
/*                            Peripheral memory map                           */
/******************************************************************************/

#define CR_BASE 		  	   (0x00400000Ul)
#define SRAM_BASE              (0x20000000UL)

#define AHB1_PERIPH_BASE       (0x40000000UL)
//#define AHB2_PERIPH_BASE       (0x50000000UL)

#define APB1_PERIPH_BASE       (0x40000000UL)
#define APB2_PERIPH_BASE       (0x40010000UL)

#define BLE_PERIPH_BASE        (0x50020000UL)

/******************************************************************************/
/*                             AHB1--APB1                                     */
/******************************************************************************/
#define I2C0_BASE              (APB1_PERIPH_BASE + 0x00000000UL) //0x40000000-0x40000FFF
#define SPI0_BASE              (APB1_PERIPH_BASE + 0x00001000UL) //0x40001000-0x40001FFF
#define UART0_BASE             (APB1_PERIPH_BASE + 0x00003000UL) //0x40003000-0x40003FFF
#define PWM_BASE               (APB1_PERIPH_BASE + 0x00004000UL) //0x40004000-0x40004FFF
#define ADC_BASE               (APB1_PERIPH_BASE + 0x00005000UL) //0x40005000-0x40005FFF
#define WDT_BASE               (APB1_PERIPH_BASE + 0x00006000UL) //0x40006000-0x40006FFF
#define CLKMEAS_BASE           (APB1_PERIPH_BASE + 0x00007000UL) //0x40007000-0x40007FFF
#define TIMER0_BASE            (APB1_PERIPH_BASE + 0x00008000UL) //0x40008000-0x40008FFF
#define KSCAN_BASE             (APB1_PERIPH_BASE + 0x00009000UL) //0x40009000-0x40009FFF
#define UART1_BASE             (APB1_PERIPH_BASE + 0x0000a000UL) //0x4000a000-0x4000aFFF

/******************************************************************************/
/*                             AHB2                                           */
/******************************************************************************/
#define GPIO_BASE              (AHB1_PERIPH_BASE + 0x00020000UL) //0x40020000-0x4002FFFF
#define P0_BASE                (GPIO_BASE        + 0x00000000UL) //0x40020030-0x4002003F
#define P1_BASE                (GPIO_BASE        + 0x00000040UL) //0x40020040-0x4002004F
#define P2_BASE                (GPIO_BASE        + 0x00000080UL) //0x40020050-0x4002005F
#define GPIO_DBNCECON_BASE     (GPIO_BASE        + 0x00000180UL) //0x40020060-0x4002006F

#define SYS_BASE               (AHB1_PERIPH_BASE + 0x00030000UL) //0x40030000-0x4003FFFF
#define CLK_BASE               (AHB1_PERIPH_BASE + 0x00040000UL) //0x40040000-0x4004FFFF
#define OTP_BASE               (AHB1_PERIPH_BASE + 0x00050000UL) //0x40050000-0x4005FFFF
#define PMU_BASE               (AHB1_PERIPH_BASE + 0x00070000UL) //0x40070000-0x4007FFFF
#define SECURE_BASE            (AHB1_PERIPH_BASE + 0x00080000UL) //0x40080000-0x4008FFFF
#define USB_BASE               (AHB1_PERIPH_BASE + 0x000A0000UL) //0x400a0000-0x400a01FF


#define PRI_RF_BASE            (BLE_PERIPH_BASE  + 0X00000428UL) //0x50020400-0x500204dc
/******************************************************************************/
/*                         Peripheral declaration                             */
/******************************************************************************/
#define CR                	   ((CR_T *) CR_BASE)

/*APB1*/
#define I2C                    ((I2C_T *)         I2C0_BASE     )
#define SPI0                   ((SPI_T *)         SPI0_BASE     )
#define UART0                  ((UART_T *)        UART0_BASE    )
#define PWM                    ((PWM_T *)         PWM_BASE      )
#define ADC                    ((ADC_T *)         ADC_BASE      )
#define WDT                    ((WDT_T *)         WDT_BASE      )
#define CLKMEAS                ((CLKMEAS_T *)     CLKMEAS_BASE  )
#define TIMER0                 ((TIMER_T *)       TIMER0_BASE   )
#define KSCAN                  ((KSCAN_T *)       KSCAN_BASE    )
#define UART1                  ((UART_T *)        UART1_BASE    )

/*AHB*/
#define GPIO                   ((GPIO_T *)        GPIO_BASE     )
#define P0                     ((GPIO_T *)        P0_BASE       )
#define P1                     ((GPIO_T *)        P1_BASE       )
#define P2                     ((GPIO_T *)        P2_BASE       )
#define P3                     ((GPIO_T *)        P3_BASE       )
#define GPIO_DB                ((GPIO_DB_T *) GPIO_DBNCECON_BASE)  ///< Pointer to GPIO de-bounce register structure

#define SYS                    ((SYSTEM_T  *)     SYS_BASE      )
#define CLK                    ((RCC_T  *)        CLK_BASE      )
#define OTP              	   ((OTP_T *)         OTP_BASE      )
#define PMU                    ((PMU_T *)         PMU_BASE      )
#define SECURE                 ((CHIP_SECURITY_T *) SECURE_BASE   )
#define USB                    ((USB_T *)         USB_BASE      )


#define PRI_RF                 ((PRI_RF_T *)      PRI_RF_BASE   )

/**
  * @}
  */



/** @addtogroup Peripheral_declaration
  * @{
  */

/**
  * @}
  */


/** @addtogroup Exported_macro
  * @{
  */
#ifndef NULL
#define NULL           (0)      ///< NULL pointer
#endif

#ifndef __ZEPHYR__
#define TRUE           (1)      ///< Boolean true, define to use in API parameters or return value
#define FALSE          (0)      ///< Boolean false, define to use in API parameters or return value
#endif

/*
#define ENABLE         (1)      ///< Enable, define to use in API parameters
#define DISABLE        (0)      ///< Disable, define to use in API parameters
*/
/* Define one bit mask */
#define BIT0     (0x00000001UL)       ///< Bit 0 mask of an 32 bit integer
#define BIT1     (0x00000002UL)       ///< Bit 1 mask of an 32 bit integer
#define BIT2     (0x00000004UL)       ///< Bit 2 mask of an 32 bit integer
#define BIT3     (0x00000008UL)       ///< Bit 3 mask of an 32 bit integer
#define BIT4     (0x00000010UL)       ///< Bit 4 mask of an 32 bit integer
#define BIT5     (0x00000020UL)       ///< Bit 5 mask of an 32 bit integer
#define BIT6     (0x00000040UL)       ///< Bit 6 mask of an 32 bit integer
#define BIT7     (0x00000080UL)       ///< Bit 7 mask of an 32 bit integer
#define BIT8     (0x00000100UL)       ///< Bit 8 mask of an 32 bit integer
#define BIT9     (0x00000200UL)       ///< Bit 9 mask of an 32 bit integer
#define BIT10    (0x00000400UL)       ///< Bit 10 mask of an 32 bit integer
#define BIT11    (0x00000800UL)       ///< Bit 11 mask of an 32 bit integer
#define BIT12    (0x00001000UL)       ///< Bit 12 mask of an 32 bit integer
#define BIT13    (0x00002000UL)       ///< Bit 13 mask of an 32 bit integer
#define BIT14    (0x00004000UL)       ///< Bit 14 mask of an 32 bit integer
#define BIT15    (0x00008000UL)       ///< Bit 15 mask of an 32 bit integer
#define BIT16    (0x00010000UL)       ///< Bit 16 mask of an 32 bit integer
#define BIT17    (0x00020000UL)       ///< Bit 17 mask of an 32 bit integer
#define BIT18    (0x00040000UL)       ///< Bit 18 mask of an 32 bit integer
#define BIT19    (0x00080000UL)       ///< Bit 19 mask of an 32 bit integer
#define BIT20    (0x00100000UL)       ///< Bit 20 mask of an 32 bit integer
#define BIT21    (0x00200000UL)       ///< Bit 21 mask of an 32 bit integer
#define BIT22    (0x00400000UL)       ///< Bit 22 mask of an 32 bit integer
#define BIT23    (0x00800000UL)       ///< Bit 23 mask of an 32 bit integer
#define BIT24    (0x01000000UL)       ///< Bit 24 mask of an 32 bit integer
#define BIT25    (0x02000000UL)       ///< Bit 25 mask of an 32 bit integer
#define BIT26    (0x04000000UL)       ///< Bit 26 mask of an 32 bit integer
#define BIT27    (0x08000000UL)       ///< Bit 27 mask of an 32 bit integer
#define BIT28    (0x10000000UL)       ///< Bit 28 mask of an 32 bit integer
#define BIT29    (0x20000000UL)       ///< Bit 29 mask of an 32 bit integer
#define BIT30    (0x40000000UL)       ///< Bit 30 mask of an 32 bit integer
#define BIT31    (0x80000000UL)       ///< Bit 31 mask of an 32 bit integer

#ifndef __ZEPHYR__
#define BIT(x)   (0x1UL << (x))
#endif

/* Byte Mask Definitions */
#define BYTE0_Msk              (0x000000FF)         ///< Mask to get bit0~bit7 from a 32 bit integer
#define BYTE1_Msk              (0x0000FF00)         ///< Mask to get bit8~bit15 from a 32 bit integer
#define BYTE2_Msk              (0x00FF0000)         ///< Mask to get bit16~bit23 from a 32 bit integer
#define BYTE3_Msk              (0xFF000000)         ///< Mask to get bit24~bit31 from a 32 bit integer

#define GET_BYTE0(u32Param)    ((u32Param & BYTE0_Msk)      )  /*!< Extract Byte 0 (Bit  0~ 7) from parameter u32Param */
#define GET_BYTE1(u32Param)    ((u32Param & BYTE1_Msk) >>  8)  /*!< Extract Byte 1 (Bit  8~15) from parameter u32Param */
#define GET_BYTE2(u32Param)    ((u32Param & BYTE2_Msk) >> 16)  /*!< Extract Byte 2 (Bit 16~23) from parameter u32Param */
#define GET_BYTE3(u32Param)    ((u32Param & BYTE3_Msk) >> 24)  /*!< Extract Byte 3 (Bit 24~31) from parameter u32Param */


#define SET_BIT(REG, BIT)     ((REG) |= (BIT))

#define CLEAR_BIT(REG, BIT)   ((REG) &= ~(BIT))

#define READ_BIT(REG, BIT)    ((REG) & (BIT))

#define CLEAR_REG(REG)        ((REG) = (0x0))

#define WRITE_REG(REG, VAL)   ((REG) = (VAL))

#define READ_REG(REG)         (REG)

#define MODIFY_REG(REG, CLEARMASK, SETMASK)  WRITE_REG((REG), (((READ_REG(REG)) & (~(CLEARMASK))) | (SETMASK)))

/**
  * @}
  */


#ifdef  USE_FULL_ASSERT

  #define assert_param(expr) ((expr) ? (void)0 : assert_failed((uint8_t *)__FILE__, __LINE__))
  void assert_failed(uint8_t* file, uint32_t line);

#else

  #define assert_param(expr) ((void)0)

#endif

#define FREQ_RCL 32000
#define FREQ_XTL 32768

#include "pan_log.h"
#include "pan_sys.h"
#include "pan_clk.h"
#include "pan_adc.h"
#include "pan_otp.h"
#include "pan_gpio.h"
#include "pan_spi.h"
#include "pan_uart.h"
#include "pan_i2c.h"
#include "pan_timer.h"
#include "pan_pwm.h"
#include "pan_wdt.h"
#include "pan_lp.h"
#include "pan_kscan.h"
#include "pan_clktrim.h"

#endif //__PAN271X_H__
