/**************************************************************************
 * @file     pan_clktrim.h
 * @version  V1.00
 * $Revision: 3 $
 * $Date:    2023/11/08 $  
 * @brief    Panchip series CLKTRIM driver header file
 *
 * @note
 * Copyright (C) 2023 Panchip Technology Corp. All rights reserved.
 *****************************************************************************/

#ifndef		_PAN_CLKTRIM_H_
#define		_PAN_CLKTRIM_H_

/**
 * @brief Clktrim Interface
 * @defgroup clktrim_interface Clktrim Interface
 * @{
 */
#ifdef __cplusplus
extern "C"
{
#endif

#include "pan271x.h"


/**@defgroup CLKMEAS_SAMPLE_NUMBER clock measure sample number 
 * @brief       clock measure sample number constants definitions
 * @{ */ 
typedef enum _CLKMEAS_SampleNumDef
{
    CLKMEAS_SAMPLE_NUM_1   = 0, /*!< Sample number 1 */
    CLKMEAS_SAMPLE_NUM_2   = 1, /*!< Sample number 2 */
    CLKMEAS_SAMPLE_NUM_4   = 2, /*!< Sample number 4 */
    CLKMEAS_SAMPLE_NUM_8   = 3, /*!< Sample number 8 */
    CLKMEAS_SAMPLE_NUM_16  = 4, /*!< Sample number 16 */
    CLKMEAS_SAMPLE_NUM_32  = 5, /*!< Sample number 32 */
    CLKMEAS_SAMPLE_NUM_64  = 6, /*!< Sample number 64 */
    CLKMEAS_SAMPLE_NUM_128 = 7, /*!< Sample number 128 */
} CLKMEAS_SampleNumDef;
/**@} */


/**@defgroup CLKMEAS_MODE_SEL clock measure mode select 
 * @brief       clock measure mode select constants definitions
 * @{ */ 
typedef enum _CLKMEAS_ModeSelDef
{
    CLKMEAS_MODE_SEL_RISING          = 0, /*!< Rising edge valid */
    CLKMEAS_MODE_SEL_FAILING         = 1, /*!< Failing edge valid */
    CLKMEAS_MODE_SEL_HIGH_VOLTAGE    = 2, /*!< High voltage valid */
    CLKMEAS_MODE_SEL_LOW_VOLTAGE     = 3, /*!< Low voltage valid */
} CLKMEAS_ModeSelDef;
/**@} */

/**@defgroup CLKMEAS_CLK_SRC_SEL clock source select 
 * @brief       clock source select constants definitions
 * @{ */ 
typedef enum _CLKMEAS_ClkSrcSelDef
{
    CLKMEAS_MEASURE_CLK_SEL_INTERNAL    = 0, /*!< Internal clock to be tested */
    CLKMEAS_MEASURE_CLK_SEL_EXTERNAL    = 1, /*!< External clock to be tested */
} CLKMEAS_ClkSrcSelDef;
/**@} */


/**
* @brief  Get the specified flag 
* @param  clkmeas: The base address of clkmeas module
* @return none
*/
__STATIC_INLINE uint32_t CLKMEAS_GetFlag(CLKMEAS_T *clkmeas)
{
	 return (clkmeas->CLKMEAS_CTL & CLKMEAS_MEASURE_FLAG_Msk);
}

/**
* @brief  Clear the specified flag 
* @param  clkmeas: The base address of clkmeas module
* @return none
*/
__STATIC_INLINE void CLKMEAS_ClearFlag(CLKMEAS_T *clkmeas)
{
	 clkmeas->CLKMEAS_CTL |= CLKMEAS_MEASURE_FLAG_Msk;
}

/**
* @brief  Set sampling times
* @param  clkmeas: The base address of clkmeas module
* @param  u8SampleNum: sample number, Valid values are
*                              \ref CLKMEAS_SAMPLE_NUMBER
* @return none
*/
__STATIC_INLINE void CLKMEAS_SetSampleNum(CLKMEAS_T *clkmeas, CLKMEAS_SampleNumDef u8SampleNum)
{
	clkmeas->CLKMEAS_CTL = (clkmeas->CLKMEAS_CTL & ~CLKMEAS_SAMP_NUM_Msk) | (u8SampleNum << CLKMEAS_SAMP_NUM_Pos);
}

/**
* @brief  Get sampling times
* @param  clkmeas: The base address of clkmeas module
* @return sample number
*/
__STATIC_INLINE uint8_t CLKMEAS_GetSampleNum(CLKMEAS_T *clkmeas)
{
	 return (0x1 << ((clkmeas->CLKMEAS_CTL & CLKMEAS_SAMP_NUM_Msk) >> CLKMEAS_SAMP_NUM_Pos));
}

/**
* @brief  Set the edge of the clock to be measured
* @param  clkmeas: The base address of clkmeas module
* @param  u8ModeSel: clk measure mode selection, Valid values are
*                              \ref CLKMEAS_MODE_SEL
* @return none
*/
__STATIC_INLINE void CLKMEAS_SetMeasureMode(CLKMEAS_T *clkmeas, CLKMEAS_ModeSelDef u8ModeSel)
{
	clkmeas->CLKMEAS_CTL = (clkmeas->CLKMEAS_CTL & ~CLKMEAS_MODE_SEL_Msk) | (u8ModeSel << CLKMEAS_MODE_SEL_Pos);
}

/**
* @brief  Set the clock source to be tested
* @param  clkmeas: The base address of clkmeas module
* @param  u8ClkSrcSel: clk source selection, Valid values are
*                              \ref CLKMEAS_CLK_SRC_SEL
* @return none
*/
__STATIC_INLINE void CLKMEAS_SetMeasuredClkSrc(CLKMEAS_T *clkmeas, CLKMEAS_ClkSrcSelDef u8ClkSrcSel)
{
	clkmeas->CLKMEAS_CTL = (clkmeas->CLKMEAS_CTL & ~CLKMEAS_CLK_SEL_Msk) | (u8ClkSrcSel << CLKMEAS_CLK_SEL_Pos);
}

/**
* @brief  This function used to enable interrupt
* @param[in]  clkmeas: where clkmeas is clkmeas module base address
* @param[in]  NewState: new state of the clk clkmeas.This parameter can be: ENABLE or DISABLE.
* @return   none
*/
__STATIC_INLINE void CLKMEAS_EnableInt(CLKMEAS_T *clkmeas, FunctionalState NewState)
{
	(NewState != DISABLE)?(clkmeas->CLKMEAS_CTL |= CLKMEAS_INT_EN_Msk):(clkmeas->CLKMEAS_CTL &= ~CLKMEAS_INT_EN_Msk);
}


/**
* @brief  This function used to enable measure function
* @param[in]  clkmeas: where clkmeas is clkmeas module base address
* @param[in]  NewState: new state of the clk clkmeas.This parameter can be: ENABLE or DISABLE.
* @return   none
*/
__STATIC_INLINE void CLKMEAS_EnableMeasure(CLKMEAS_T *clkmeas, FunctionalState NewState)
{
	(NewState != DISABLE)?(clkmeas->CLKMEAS_CTL |= CLKMEAS_MEASURE_EN_Msk):(clkmeas->CLKMEAS_CTL &= ~CLKMEAS_MEASURE_EN_Msk);
}

/**
* @brief  This function used to get measure count
* @param[in]  clkmeas: where clkmeas is clkmeas module base address
* @param[in]  NewState: new state of the clk clkmeas.This parameter can be: ENABLE or DISABLE.
* @return   none
*/
__STATIC_INLINE uint32_t CLKMEAS_GetMeasureCnt(CLKMEAS_T *clkmeas)
{
	return (clkmeas->CLKMEAS_CNT & CLKMEAS_MEASURE_CNT_Msk);
}


/**@} */


#ifdef __cplusplus
}
#endif

#endif /* _PAN_CLKTRIM_H_ */

